import textwrap
import json


_YT_PROXY = "hahn"
_REQUEST_DATA_LOGS_ROOT = "logs/nanpu-request-data-log/"
_EVENT_LOGS_ROOT = "logs/nanpu-event-log/"


def _make_limits_str(limits):
    return ','.join('("{}", {})'.format(key, int(int(limit) * 1.05)) for key, limit in limits.iteritems())


def make_aggregation_query(work_prefix, user_subquery, pt_list, logs_interval_type="1d", interval_size=21):
    """
    Query that prepares aggregated data from logs for following selections of ammo according to limits.
    """

    pt_list_str = json.dumps(pt_list)

    user_filtration = ""
    if user_subquery:
        user_filtration = textwrap.dedent(
            """
            DEFINE SUBQUERY $select_user_request_ids($begin, $end) AS
                {user_subquery}
            END DEFINE;

            $user_graph_ids = select * from $select_user_request_ids($begin, $end);

            $user_graph_ids_path = $work_prefix || "/user_graph_ids";
            INSERT INTO $user_graph_ids_path WITH TRUNCATE
            SELECT * FROM $user_graph_ids;

            $select_request_filtered =
            SELECT
                *
            FROM
                $select_request_filtered
            WHERE
                GraphID in $user_graph_ids;

            $select_event_filtered =
            SELECT
                *
            FROM
                $select_event_filtered
            WHERE
                RequestID in $user_graph_ids;
        """
        ).format(
            user_subquery=user_subquery,
        )

    return textwrap.dedent(
        """
        USE {yt_proxy};
        PRAGMA yt.ExpirationInterval = "14d";

        $log_interval = "{logs_interval_type}";

        $work_prefix = "{work_prefix}";
        $request_data_logs_root = "{request_data_logs_root}" || $log_interval;
        $event_logs_root = "{event_logs_root}" || if($log_interval == "1h", "30min", "1d");

        $processing_type_list = {pt_list_str};

        -- Select raw logs for requested period
        $end_datetime = DateTime::StartOf(CurrentTzDatetime("Europe/Moscow"), if($log_interval == "1h", DateTime::IntervalFromMinutes(30), DateTime::IntervalFromDays(1)));
        $interval = if($log_interval == "1h", DateTime::IntervalFromHours({interval_size}), DateTime::IntervalFromDays({interval_size}));
        $begin_datetime = DateTime::MakeTzTimestamp($end_datetime) - $interval;

        $format = DateTime::Format("%Y-%m-%dT%H:%M:%S");
        $end = $format($end_datetime);
        $begin = $format($begin_datetime);

        $select_request_filtered = (SELECT * FROM RANGE($request_data_logs_root, $begin, $end));
        $select_event_filtered = (SELECT * FROM RANGE($event_logs_root, $begin, $end));

        -- User filtration
        {user_filtration}

        -- Join of ProcessingType
        $filtered_request_ids =
        SELECT
            GraphID
        FROM
            $select_request_filtered
        WHERE
            Handler IS NOT NULL;

        $filtered_processing_types =
        SELECT
            RequestID as GraphID,
            ProcessingType
        FROM
            $select_event_filtered
        WHERE
            ProcessingType in $processing_type_list AND
            HandlerName == "ad" AND
            RequestID IN $filtered_request_ids;

        -- Write to cluster
        $request_data_range_path = $work_prefix || "/request_data_range";
        INSERT INTO $request_data_range_path WITH TRUNCATE
        SELECT $begin as begin, $end as end;

        $filtered_processing_types_path = $work_prefix || "/filtered_processing_types_graph_ids";
        INSERT INTO $filtered_processing_types_path WITH TRUNCATE
        SELECT * FROM $filtered_processing_types ORDER BY GraphID;

        $unique_processing_types_path = $work_prefix || "/unique_processing_types";
        INSERT INTO $unique_processing_types_path WITH TRUNCATE
        SELECT DISTINCT ProcessingType FROM $filtered_processing_types;

        $unique_handlers_path = $work_prefix || "/unique_handlers";
        INSERT INTO $unique_handlers_path WITH TRUNCATE
        SELECT DISTINCT Handler FROM $select_request_filtered;
    """
    ).format(
        yt_proxy=_YT_PROXY,
        work_prefix=work_prefix,
        request_data_logs_root=_REQUEST_DATA_LOGS_ROOT,
        event_logs_root=_EVENT_LOGS_ROOT,
        interval_size=interval_size,
        logs_interval_type=logs_interval_type,
        pt_list_str=pt_list_str,
        user_filtration=user_filtration,
    )


def make_select_query(
    work_prefix, handler_limits, pt_limits, logs_interval_type="1d", had_user_subquery=False, output_suffix=""
):
    handler_limits_str = _make_limits_str(handler_limits)
    pt_limits_str = _make_limits_str(pt_limits)

    user_filtration = ""
    if had_user_subquery:
        user_filtration = textwrap.dedent(
            """
            $user_graph_ids_path = $work_prefix || "/user_graph_ids";
            $user_graph_ids = SELECT * FROM $user_graph_ids_path;

            $request_data =
            SELECT
                *
            FROM
                $request_data
            WHERE
                GraphID in $user_graph_ids;
        """
        )

    return textwrap.dedent(
        """
        USE {yt_proxy};
        PRAGMA yt.ExpirationInterval = "14d";

        $log_interval = "{logs_interval_type}";

        $work_prefix = "{work_prefix}";

        $handler_limits = AsDict({handler_limits_str});
        $processing_type_limits = AsDict({pt_limits_str});

        -- Read request data
        $request_data_logs_root = "{request_data_logs_root}" || $log_interval;
        $request_data_range_path = $work_prefix || "/request_data_range";
        $begin = select some(begin) from $request_data_range_path;
        $end = select some(end) from $request_data_range_path;
        $request_data = (SELECT * FROM RANGE($request_data_logs_root, $begin, $end));

        -- Apply user filtration
        {user_filtration}

        $filtered_processing_types_path = $work_prefix || "/filtered_processing_types_graph_ids";
        $filtered_processing_types = (SELECT * FROM $filtered_processing_types_path);

        -- Handlers limits selection
        $other_handlers_limit = $handler_limits["OTHER"] ?? 0;
        $get_handler_limit = ($handler) -> {{
            RETURN $handler_limits[$handler] ?? $other_handlers_limit;
        }};

        DEFINE ACTION $select_graphids_for_handler($handler) AS
            INSERT INTO @graphids
            SELECT GraphID FROM (
                SELECT GraphID FROM $request_data
                WHERE Handler==$handler
                ORDER BY GraphID DESC
                LIMIT $get_handler_limit($handler)
            );
        END DEFINE;

        $all_handlers_table = $work_prefix || "/unique_handlers";
        $all_handlers = (SELECT AGGREGATE_LIST(Handler) FROM $all_handlers_table);
        EVALUATE FOR $handler IN $all_handlers
            DO $select_graphids_for_handler($handler);
        COMMIT;

        -- Handlers random selection
        DEFINE ACTION $select_random() AS
            INSERT INTO @random_graphids
            SELECT GraphID FROM (
                SELECT GraphID FROM (
                    SELECT GraphID
                    FROM $request_data
                    WHERE Handler IS NOT NULL
                ) AS F
                LEFT ONLY JOIN @graphids AS G
                ON F.GraphID == G.GraphID
                ORDER BY GraphID DESC
                LIMIT $get_handler_limit("RANDOM")
            );
            COMMIT;
            INSERT INTO @graphids SELECT GraphID FROM @random_graphids;
            COMMIT;
        END DEFINE;

        EVALUATE IF DictContains($handler_limits, "RANDOM")
            DO $select_random();

        -- ProcessingType limits selection
        $other_processing_type_limit = $processing_type_limits["OTHER"] ?? 0;
        $get_processing_type_limit = ($processing_type) -> {{
            RETURN $processing_type_limits[$processing_type] ?? $other_processing_type_limit;
        }};

        DEFINE ACTION $select_graphids_for_processing_type($processing_type) AS
            INSERT INTO @graphids
            SELECT GraphID FROM (
                SELECT GraphID FROM $filtered_processing_types
                WHERE ProcessingType == $processing_type
                ORDER BY GraphID desc
                LIMIT $get_processing_type_limit($processing_type)
            );
        END DEFINE;

        $all_processing_types_table = $work_prefix || "/unique_processing_types";
        $all_processing_types = (SELECT AGGREGATE_LIST(ProcessingType) FROM $all_processing_types_table);
        EVALUATE FOR $processing_type IN $all_processing_types
            DO $select_graphids_for_processing_type($processing_type);
        COMMIT;

        -- Write request-data for selected graph ids
        $output = $work_prefix || "/output" || "{output_suffix}";
        INSERT INTO $output WITH TRUNCATE
        SELECT
            F.GraphID as GraphID,
            DENSE_RANK(F.GraphID) over w as `X-Yabs-Nanpu-Req-Id`,
            ResponseNo as `X-Yabs-Nanpu-Request-No`,
            String::Base64Decode(RequestBase64) as Request,
            Unixtime,
            Handler,
            String::Base64Decode(ResponseBase64) as Response,
            Tag,
            HttpStatus
        FROM
            $request_data AS F
        INNER JOIN
            @graphids AS G
        ON
            F.GraphID == G.GraphID
        WINDOW w as
            (ORDER BY F.GraphID)
        ORDER BY
            GraphID;
    """
    ).format(
        yt_proxy=_YT_PROXY,
        logs_interval_type=logs_interval_type,
        work_prefix=work_prefix,
        handler_limits_str=handler_limits_str,
        pt_limits_str=pt_limits_str,
        request_data_logs_root=_REQUEST_DATA_LOGS_ROOT,
        user_filtration=user_filtration,
        output_suffix=output_suffix,
    )
