import json
import logging
import os
import subprocess

from sandbox import sdk2
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt

from sandbox.projects.resource_types import CACHE_DAEMON
from sandbox.projects.common.dolbilka.resources import DPLANNER_EXECUTABLE
from sandbox.projects.common.yabs.server.util.general import try_get_from_vault

from sandbox.projects.yabs.qa.resource_types import (
    BaseBackupSdk2Resource,
    YABS_SERVER_DOLBILKA_PLAN,
    YABS_SERVER_REQUEST_LOG_GZ,
    YABS_SERVER_CACHE_DAEMON_STUB_DATA
)


CACHEDAEMON_DUMP_KEY = 'cachedaemon_dump_res_id'
TESTENV_SWITCH_TRIGGER_PREFIX = 'testenv_switch_trigger'


class GenAmmoAndStubFromYtBinary(BaseBackupSdk2Resource):
    """ yabs/server/tools/gen_sandbox_stub/from_cooked_tables binary """
    releasers = ["YABS_SERVER_SANDBOX_TESTS"]
    release_subscribers = []
    releasable = True


class BuildAmmoAndStubsFromYT(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 32 * 1024
        disk_space = 40960

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        # common parameters
        description = "Generate yabs-server ammo and stubs from pre-cooked tables with compatible format"
        max_restarts = 3
        kill_timeout = 60 * 60 * 4

        # custom parameters
        gen_ammo_and_stub_from_yt_binary = sdk2.parameters.LastReleasedResource(
            "Stub and ammo generator binary",
            resource_type=GenAmmoAndStubFromYtBinary,
            state=(ctr.State.READY, ctr.State.NOT_READY),
            required=True
        )

        cache_daemon_resource = sdk2.parameters.LastReleasedResource(
            'Cache daemon binary',
            resource_type=CACHE_DAEMON,
            state=(ctr.State.READY, ctr.State.NOT_READY),
            required=True
        )

        d_planner_resource = sdk2.parameters.LastReleasedResource(
            'Dolbilka planner binary',
            resource_type=DPLANNER_EXECUTABLE,
            state=(ctr.State.READY, ctr.State.NOT_READY),
            required=True
        )

        # output resource types
        dolbilka_plan_resource_type = sdk2.parameters.String("Resource type for dolbilka_plan", default=YABS_SERVER_DOLBILKA_PLAN.name)
        request_log_gz_resource_type = sdk2.parameters.String("Resource type for request_log_gz", default=YABS_SERVER_REQUEST_LOG_GZ.name)
        cache_daemon_stub_resource_type = sdk2.parameters.String("Resource type for cache_daemon_stub", default=YABS_SERVER_CACHE_DAEMON_STUB_DATA.name)

        input_tables = sdk2.parameters.JSON('Input YT tables', default={})
        spec = sdk2.parameters.JSON('Spec', default={})
        yt_token_vault_name = sdk2.parameters.String("Vault name to get YT token from", default="yabs-cs-sb-yt-token")
        yt_work_prefix = sdk2.parameters.String('YT work prefix', default='', do_not_copy=True)
        testenv_switch_trigger = sdk2.parameters.Bool("Switch Testenv to generated resources", default=False, do_not_copy=True)
        testenv_switch_trigger_value = sdk2.parameters.String('Use following string as testenv switch trigger value, backup date will be used if empty', default='')
        additional_attributes = sdk2.parameters.JSON("Additional attributes", default={})
        resource_ttl = sdk2.parameters.Integer("Output resources ttl", default=30)
        legacy_mode = sdk2.parameters.Bool('Legacy mode', default_value=True)
        with legacy_mode.value[True]:
            key_header = sdk2.parameters.String('Key header for ammo and cachedaemon', default='X-YaBS-Request-Id')

        run_from_nirvana = sdk2.parameters.Bool('Task was launched from Nirvana', default_value=False)
        with run_from_nirvana.value[True]:
            nirvana_input_tables = sdk2.parameters.String('Input YT tables', default='{}')
            nirvana_spec = sdk2.parameters.String('Spec', default='{}')
            nirvana_additional_attributes = sdk2.parameters.String('Spec', default='{}')

        with sdk2.parameters.Output:
            result_spec = sdk2.parameters.JSON('Output generated ammo spec', default={})

    def on_save(self):
        super(BuildAmmoAndStubsFromYT, self).on_save()
        if self.Parameters.run_from_nirvana:
            self.Parameters.input_tables = json.loads(self.Parameters.nirvana_input_tables)
            self.Parameters.spec = json.loads(self.Parameters.nirvana_spec)
            self.Parameters.additional_attributes = json.loads(self.Parameters.nirvana_additional_attributes)
        if self.Parameters.yt_token_vault_name == 'yabs-cs-sb-yt-token':
            self.Requirements.semaphores = ctt.Semaphores(
                acquires=[
                    ctt.Semaphores.Acquire(name='yabs-cs-sb-ammo', weight=1),
                ],
                release=ctt.Status.Group.BREAK + ctt.Status.Group.FINISH
            )
        else:
            self.Requirements.semaphores = ()

    def on_execute(self):
        stubgen_env = self._add_tokens_to_env()

        yt_work_prefix = self.Parameters.yt_work_prefix or '//home/yabs-cs-sandbox/gen_sandbox_stub/{}'.format(self.id)
        requests_out_dir = os.path.abspath('generated_data')

        stubgen_cmdline = [
            unicode(sdk2.ResourceData(self.Parameters.gen_ammo_and_stub_from_yt_binary).path),
            '--tables', json.dumps(self.Parameters.input_tables),
            '--out-dir', requests_out_dir,
            '--d-planner-executable', unicode(sdk2.ResourceData(self.Parameters.d_planner_resource).path),
            '--cachedaemon-executable', unicode(sdk2.ResourceData(self.Parameters.cache_daemon_resource).path),
            '--proxy', 'hahn',
            '--yt-work-prefix', yt_work_prefix,
            '--spec', json.dumps(self.Parameters.spec)
        ]
        if self.Parameters.legacy_mode:
            stubgen_cmdline.extend(('--key-header', self.Parameters.key_header))

        logging.debug('Invoking gen_sandbox_stub executable: %s', ' '.join(stubgen_cmdline))

        with sdk2.helpers.ProcessLog(self, logger='gen_sandbox_stub') as process_log:
            created_ammo_and_stubs_dict = json.loads(subprocess.check_output(stubgen_cmdline, stderr=process_log.stderr, env=stubgen_env))

        logging.info(json.dumps(created_ammo_and_stubs_dict, indent=4))

        result_spec = {}

        for section_name, metadata_dict in created_ammo_and_stubs_dict.iteritems():
            if not metadata_dict:
                logging.error('Failed to generate ammo for "{}"'.format(section_name))
                continue

            result_spec[section_name] = {}
            stub_attrs = self.Parameters.additional_attributes.copy()
            key_header = ','.join(metadata_dict['cachedaemon_key_headers'])
            if self.Parameters.legacy_mode:
                key_header = self.Parameters.key_header
            stub_attrs.update({
                'cache_daemon_res_id': self.Parameters.cache_daemon_resource.id,
                'provided_tags': ' '.join(metadata_dict['cachedaemon_provided_tags']),
                'key_header': key_header,
            })
            if 'ttl' not in stub_attrs:
                stub_attrs.update(ttl=self.Parameters.resource_ttl)
            if self.Parameters.testenv_switch_trigger:
                testenv_switch_trigger_value = self.Parameters.testenv_switch_trigger_value or str(self.id)
                stub_attrs['{}_{}'.format(TESTENV_SWITCH_TRIGGER_PREFIX, section_name)] = testenv_switch_trigger_value
            cachedaemon_data_dir = metadata_dict['cachedaemon_data_dir']
            if not os.path.isdir(cachedaemon_data_dir):
                os.makedirs(cachedaemon_data_dir)
                open(os.path.join(cachedaemon_data_dir, 'empty_stub_flag'), 'a').close()
            stub_resource = sdk2.Resource[self.Parameters.cache_daemon_stub_resource_type](
                self,
                "Cache daemon data for yabs-server tests",
                cachedaemon_data_dir,
                arch='any',
                **stub_attrs
            )
            result_spec[section_name]['stub_resource'] = stub_resource.id

            created_ammo = set()
            if metadata_dict.get('ammo_path'):
                created_ammo.add((metadata_dict['ammo_type'], metadata_dict['ammo_path']))
            if metadata_dict.get('requestlog_path'):
                created_ammo.add(('requestlog', metadata_dict['requestlog_path']))
            if metadata_dict.get('dplan_path'):
                created_ammo.add(('dplan', metadata_dict['dplan_path']))

            for ammo_type, ammo_path in created_ammo:
                if ammo_type == 'dplan':
                    ammo_res_type = self.Parameters.dolbilka_plan_resource_type
                elif ammo_type == 'requestlog':
                    ammo_res_type = self.Parameters.request_log_gz_resource_type

                ammo_attrs = self.Parameters.additional_attributes.copy()
                ammo_attrs.update({
                    'subtype': section_name,
                    CACHEDAEMON_DUMP_KEY: stub_resource.id
                })
                if 'ttl' not in ammo_attrs:
                    ammo_attrs.update(ttl=self.Parameters.resource_ttl)
                if self.Parameters.testenv_switch_trigger:
                    testenv_switch_trigger_value = self.Parameters.testenv_switch_trigger_value or str(self.id)
                    ammo_attrs['{}_{}'.format(TESTENV_SWITCH_TRIGGER_PREFIX, section_name)] = testenv_switch_trigger_value

                ammo_resource = sdk2.Resource[ammo_res_type](
                    self,
                    '{} ammo'.format(section_name),
                    ammo_path,
                    **ammo_attrs
                )

                result_spec[section_name]['ammo_resource'] = ammo_resource.id
                result_spec[section_name]['{}_resource'.format(ammo_type)] = ammo_resource.id
            if metadata_dict['ammo_type'] == 'requestlog':
                result_spec[section_name]['ammo_resource'] = result_spec[section_name]['requestlog_resource']
            elif metadata_dict['ammo_type'] == 'dplan':
                result_spec[section_name]['ammo_resource'] = result_spec[section_name]['dplan_resource']

        self.Parameters.result_spec = result_spec

    def _add_tokens_to_env(self):
        env = os.environ.copy()
        yt_token = try_get_from_vault(self, self.Parameters.yt_token_vault_name)
        logging.info("Token lengths: YT %d", len(yt_token))
        env['YT_TOKEN'] = yt_token
        return env
