import logging
import os
import re
import datetime
import hashlib

import sandbox.common.types.task as ctt
from sandbox import sdk2
import sandbox.projects.common.arcadia.sdk as arcadiasdk

from sandbox.projects.yabs.qa.tasks.YabsServerGetYTRequestData2 import YabsServerGetYtRequestData2
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sdk2 import make_resource_ready
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.generic import new_resource

from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.yabs.qa.utils.general import is_precommit_check

from sandbox.projects.yabs.qa.resource_types import (
    YABS_SERVER_REQUEST_LOG_GZ,
    YABS_SERVER_CACHE_DAEMON_STUB_DATA
)

DEFAULT_SPEC = {
    "bs_func": {
        "make_load": False,
        "mode": "metapartner",
        "limits": [
            {
                "upper": {
                    "RANDOM": 200000
                },
                "lower": {
                    "RANDOM": 500
                }
            }
        ]
    }
}


class BuildCustomAmmo(sdk2.Task):
    class Parameters(YabsServerGetYtRequestData2.Parameters):
        description = "Generate yabs-server ammo"
        yql_directory = "sandbox/projects/yabs/qa/CustomAmmoYqlScripts"
        request_number_limit = sdk2.parameters.Integer("Maximum possible number of requests", default=100000, required=True)
        refresh_time_in_days = sdk2.parameters.Integer("Refresh time in days", default=0, required=True)
        run_only_in_precommit_checks = sdk2.parameters.Bool('Run task only in precommit checks, otherwise exit immediately', default=True)

        with sdk2.parameters.Group('YabsServerGetYtRequestData2 parameters'):
            spec = YabsServerGetYtRequestData2.Parameters.spec(default=DEFAULT_SPEC)

        with sdk2.parameters.Output:
            requestlog_resource = sdk2.parameters.Resource("Requestlog resource", resource_type=YABS_SERVER_REQUEST_LOG_GZ)
            cache_daemon_stub_resource = sdk2.parameters.Resource("Cache daemon stub resource", resource_type=YABS_SERVER_CACHE_DAEMON_STUB_DATA)

    class Context(sdk2.Context):
        ammo_generation_task_id = None
        yql_hash = None

    class Requirements(sdk2.Requirements):
        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(
                    name='yabs-server-collect-custom-ammo-semaphore',
                ),
            ],
            release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
        )

    def _mount_arc(self):
        self.local_arcadia = arcadiasdk.mount_arc_path('arcadia-arc:/#trunk')
        self.local_arcadia_path = self.local_arcadia.__enter__()
        self.yql_directory_path = os.path.join(self.local_arcadia_path, self.Parameters.yql_directory)

    def _read_and_concat_yql_queries(self):
        yql_query = ''
        for index, file in enumerate(os.listdir(self.yql_directory_path)):
            filename = os.path.join(self.yql_directory_path, file)
            with open(filename) as file:
                query = file.read()

                logging.info('Appending yql query: {}'.format(query))

                if index > 0:
                    yql_query = yql_query + '\n union all \n'

                query = query.strip(' \n\t')
                if query[-1] == ';':
                    query = query[:-1]
                yql_query += '(' + query + ')'

                logging.info('Current query after appending: {}'.format(yql_query))
        yql_query += ';'

        self._validate_requests_number(yql_query)
        self.Context.yql_hash = hashlib.md5(yql_query).hexdigest()
        return yql_query

    def _validate_requests_number(self, yql_query):
        logging.info('Validating requests number')
        limits = list(map(int, re.findall(r'limit ([0-9]*)', yql_query, re.IGNORECASE)))
        logging.info('Parsed limits from yql requests: ', limits)

        assert sum(limits) <= self.Parameters.request_number_limit, "Total number of requests exceeded: \n \
                limit={}, sum of limits from yqls={}".format(self.Parameters.request_number_limit, sum(limits))

    def _call_ammo_generation(self, yql_query):
        ammo_generation_task = YabsServerGetYtRequestData2(
            self,
            owner=self.owner,
            gen_ammo_tables_binary=self.Parameters.gen_ammo_tables_binary,
            gen_ammo_and_stub_from_yt_binary=self.Parameters.gen_ammo_and_stub_from_yt_binary,
            cache_daemon_resource=self.Parameters.cache_daemon_resource,
            d_planner_resource=self.Parameters.d_planner_resource,
            spec=self.Parameters.spec,
            yql_query_role=self.Parameters.yql_query_role,
            yt_token_vault_name=self.Parameters.yt_token_vault_name,
            yql_token_vault_name=self.Parameters.yql_token_vault_name,
            pool=self.Parameters.pool,
            enable_page_id_coverage=self.Parameters.enable_page_id_coverage,
            days_interval=self.Parameters.days_interval,
            logs_interval=self.Parameters.logs_interval,
            resource_ttl=self.Parameters.resource_ttl,
            yql_query=yql_query,
        )
        ammo_generation_task.save().enqueue()
        self.Context.ammo_generation_task_id = ammo_generation_task.id

        check_tasks(self, [self.Context.ammo_generation_task_id])

    def _create_and_make_ready_resource(self, path, resource_type, resource_name, kwargs={}):
        open(path, 'a').close()
        resource = new_resource(resource_type, self, resource_name, path, **kwargs)
        make_resource_ready(resource)
        return resource

    def _try_reuse_ammo(self):
        resource = YABS_SERVER_REQUEST_LOG_GZ.find(attrs={'is_custom_ammo': True}).order(-sdk2.Resource.id).first()
        resource_creation_time = 0
        need_to_update_ammo = True

        if resource is not None:
            resource_creation_time = resource.created
            logging.info('Last resource yql request hash:\n {}'.format(resource.yql_hash))
            logging.info('Our query hash:\n {}'.format(self.Context.yql_hash))
            logging.info('Days since last resource creation: {}'.format((datetime.datetime.now(tz=resource_creation_time.tzinfo) - resource_creation_time).days))
            need_to_update_ammo = (datetime.datetime.now(tz=resource_creation_time.tzinfo) - resource_creation_time).days >= self.Parameters.refresh_time_in_days \
                    or resource.yql_hash != self.Context.yql_hash
        else:
            logging.info('Query resource not found, generating a new one and updating ammo')

        if not need_to_update_ammo:
            return resource, sdk2.Resource[resource.cachedaemon_dump_res_id]
        return None, None

    def _create_dummy_ammo_resources(self):
        dummy_ammo_resource = self._create_and_make_ready_resource(
            path='ammo_resource.txt',
            resource_type=YABS_SERVER_REQUEST_LOG_GZ,
            resource_name='Ammo resource',
        )
        dummy_stub_resource = self._create_and_make_ready_resource(
            path='stub_resource.txt',
            resource_type=YABS_SERVER_CACHE_DAEMON_STUB_DATA,
            resource_name='Cache daemon stub resource',
        )
        return dummy_ammo_resource, dummy_stub_resource

    def _get_resources_from_subtask(self):
        result_spec = sdk2.Task[self.Context.ammo_generation_task_id].Parameters.result_spec

        ammo_resource_id = result_spec['bs_func']['requestlog_resource']
        stub_resource_id = result_spec['bs_func']['stub_resource']

        sdk2.Resource[ammo_resource_id].yql_hash = self.Context.yql_hash
        sdk2.Resource[ammo_resource_id].is_custom_ammo = True

        ammo_resource = sdk2.Resource[ammo_resource_id]
        stub_resource = sdk2.Resource[stub_resource_id]

        return ammo_resource, stub_resource

    def _get_resources(self):
        if self.Parameters.run_only_in_precommit_checks and not is_precommit_check(self):
            self.set_info("Task execution skipped because it executes only on precommit checks")
            return self._create_dummy_ammo_resources()

        if self.Context.ammo_generation_task_id is not None:
            return self._get_resources_from_subtask()

        self._mount_arc()
        yql_query = self._read_and_concat_yql_queries()
        ammo, cachedaemon = self._try_reuse_ammo()
        if ammo is None:
            self._call_ammo_generation(yql_query)
        return ammo, cachedaemon

    def on_execute(self):
        ammo_resource, stub_resource = self._get_resources()
        with self.memoize_stage.set_output():
            self.Parameters.requestlog_resource = ammo_resource
            self.Parameters.cache_daemon_stub_resource = stub_resource
