import logging

from sandbox import sdk2
from sandbox.projects.abc.client import AbcClient
from sandbox.sandboxsdk.environments import PipEnvironment


logger = logging.getLogger(__name__)


def st_link(issue):
    return '<a href="https://st.yandex-team.ru/{issue}" target="_blank">{issue}</a>'.format(issue=issue)


def get_issue_list(startrek_client, queue):
    search_query = " AND ".join([
        "Queue: {}".format(queue),
        "Assignee: empty()",
        "Resolution: empty()",
    ])
    return list(startrek_client.issues.find(search_query))


class DutyIssueAssigner(sdk2.Task):

    class Requirements(sdk2.Requirements):
        ram = 4 * 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

        environments = [
            PipEnvironment('startrek_client', use_wheel=True),  # PYPI-101
        ]

    class Parameters(sdk2.Parameters):
        secret = sdk2.parameters.YavSecret("YaV secret", default="sec-01e2dg5tj8bm4wv9csfnm9t1g2")
        duty_slug_to_queue = sdk2.parameters.Dict(
            "Mapping from ABC duty slug in <abc_service_id>:<schedule_slug> format to Startrek queue",
            default={
                "2409:yabs_server_sandbox_tests_duty": "SHMDUTY",
            }
        )

    def on_execute(self):
        from startrek_client import Startrek

        tokens = self.Parameters.secret.data()
        abc_client = AbcClient(tokens['abc_token'])
        startrek_client = Startrek(useragent=self.__class__.__name__, token=tokens['startrek_token'])

        updated_issues = {}
        for slug, queue in self.Parameters.duty_slug_to_queue.items():
            service_id, schedule_slug = slug.split(":")
            service_id = int(service_id)
            on_duty = abc_client.get_current_duty_login(service_id, schedule_slug=schedule_slug)
            logger.debug('Got duty for %s@%d: %s', schedule_slug, service_id, on_duty)
            not_assigned_issues = get_issue_list(startrek_client, queue)
            logger.debug("Got not assigned issues: %s", not_assigned_issues)
            for issue in not_assigned_issues:
                logger.debug("Assigning issue %s to %s", issue.key, on_duty)
                issue.update(assignee=on_duty)
                updated_issues.setdefault(queue, []).append(issue.key)

        if updated_issues:
            self.Parameters.tags += ['UPDATED_{}'.format(q) for q in updated_issues.keys()]
            self.set_info(
                '\n'.join([
                    '{queue} updated issues:\n{issues}'.format(
                        queue=st_link(queue),
                        issues='\n'.join([st_link(issue) for issue in issues])
                    )
                    for queue, issues in updated_issues.items()
                ]),
                do_escape=False
            )
