import logging
import os
import subprocess
import textwrap
import traceback
import datetime

import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.common.errors import TaskFailure, TaskError
from sandbox.common.types import task as task_type
from sandbox.common.types.misc import NotExists
from sandbox.common.types.resource import State
from sandbox.common.utils import get_task_link
from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.yabs.qa.resource_types import YT_ONESHOT_BINARY, YT_ONESHOT_OUTPUT, YT_ONESHOTS_PACKAGE
from sandbox.projects.yabs.qa.utils.general import startrek_hyperlink, get_resource_html_hyperlink, unpack_targz_package
from sandbox.projects.yabs.qa.tasks.BackupYTTables import BackupYTTables, set_table_state
from sandbox.sandboxsdk import environments
from sandbox.sdk2.vcs import svn


logger = logging.getLogger(__name__)
logging.getLogger('requests').setLevel(logging.WARNING)


DEFAULT_YT_PROXY = 'hahn'
DEFAULT_BACKUP_TTL = datetime.timedelta(hours=1)
TESTING_STARTREK_API_ENDPOINT = 'https://st-api.test.yandex-team.ru'

TESTING_YT_TOKEN_VAULT_ITEM = sdk2.VaultItem(owner_or_name='YABS_SERVER_SANDBOX_TESTS',
                                             name='yabs-cs-sb-yt-token')
PRODUCTION_YT_TOKEN_VAULT_ITEM = sdk2.VaultItem(owner_or_name='YABS-YT-ONESHOT-EXECUTOR',
                                                name='robot-yabs-oneshot-yt-token')
STARTREK_TOKEN_VAULT_ITEM = sdk2.VaultItem(owner_or_name='YABS_SERVER_SANDBOX_TESTS',
                                           name='robot-yabs-cs-sb-startrek-token')

ONESHOTS_DIR = 'oneshots_binaries'


class ExecuteYTOneshot(sdk2.Task):
    """Execute YT oneshot"""

    name = 'EXECUTE_YT_ONESHOT'

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        environments = (
            environments.PipEnvironment('startrek_client', use_wheel=True),
            environments.PipEnvironment('yandex-yt', use_wheel=True),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 0
        oneshot_path = sdk2.parameters.String(
            label='Oneshot path',
            description='Example: //arcadia.yandex.ru/arc/trunk/arcadia/yabs/qa/yt_oneshot/example/oneshot.py@4576553')
        oneshot_args = sdk2.parameters.String(label='Oneshot arguments')
        print_tables_only = sdk2.parameters.Bool('Only run "print-tables" command')
        startrek_ticket = sdk2.parameters.String('Oneshot\'s startrek ticket')
        run_in_test_mode = sdk2.parameters.Bool(
            'Run oneshot in test mode',
            description='Use default YT proxy')
        with run_in_test_mode.value[True]:
            run_backup_tables = sdk2.parameters.Bool(
                'Backup tables affected in oneshot and execute oneshot with tables copies',
                default=True
            )
        backup_dir_ttl = sdk2.parameters.Integer(
            'TTL for backups in seconds',
            default=DEFAULT_BACKUP_TTL.total_seconds()
        )
        oneshot_binary_ttl = sdk2.parameters.Integer(
            'Oneshot binary resource TTL in days',
            default=14
        )
        with run_in_test_mode.value[False]:
            yt_proxy = sdk2.parameters.String('YT proxy')
            allow_access_to_production_data = sdk2.parameters.Bool('Allow access to production YT data')
        ypath_prefix = sdk2.parameters.String(
            'YPath prefix',
            description='This parameter is not used when run_in_test_mode and run_backup_tables'
        )
        reuse_oneshot_binary = sdk2.parameters.Bool('Reuse already built oneshot binary', default=False)  # TODO: BSSERVER-13728
        force_data_flush = sdk2.parameters.Bool('Force data flush')

        with sdk2.parameters.Group('For development') as development_block:
            yt_token_vault_name = sdk2.parameters.String('Vault name for YT token')
            oneshot_binary_resource = sdk2.parameters.Resource(
                'YT oneshot binary',
                resource_type=YT_ONESHOT_BINARY,
            )
            oneshots_package = sdk2.parameters.Resource(
                'YT oneshots package',
                resource_type=YT_ONESHOTS_PACKAGE,
            )
            startrek_token_vault_name = sdk2.parameters.String('Vault name for Startrek token')
            use_testing_startrek = sdk2.parameters.Bool('Use testing startrek')

        with sdk2.parameters.Output:
            oneshot_output_resource = sdk2.parameters.Resource('Oneshot output resource')
            oneshot_tables = sdk2.parameters.List('Oneshot tables', value_type=sdk2.parameters.String, default=[])
            backup_dir = sdk2.parameters.String('Directory with backed up and modified oneshot tables')

    class Context(sdk2.Task.Context):
        error_message = None
        oneshot_output = None

    def backup_tables(self, tables):
        """
        Run and wait for completion sandbox task to backup yt tables.
        Return path of YT node with tables backups.

        :param tables: Paths of YT tables to backup
        :return: Path of YT node with tables backups
        """
        with self.memoize_stage.backup_oneshot_tables(commit_on_entrance=False):
            backup_yt_tables_task = BackupYTTables(
                self,
                description='Backup tables for YT oneshot "{}"'.format(self.Parameters.oneshot_path),
                backup_dir_ttl=self.Parameters.backup_dir_ttl,
                tables=tables,
            )
            backup_yt_tables_task.enqueue()
            self.Context.backup_yt_tables_task_id = backup_yt_tables_task.id

        check_tasks(self, [self.Context.backup_yt_tables_task_id], callback=on_backup_tables_finish)

        return sdk2.Task[self.Context.backup_yt_tables_task_id].Parameters.backup_dir

    def on_finish(self, prev_status, status):
        if self.Parameters.startrek_ticket:
            comment = self.generate_startrek_ticket_comment(status)
            self.add_startrek_ticket_comment(comment)

    def on_break(self, prev_status, status):
        if self.Parameters.startrek_ticket:
            comment = self.generate_startrek_ticket_comment(status)
            environments.PipEnvironment('startrek_client', use_wheel=True).prepare()
            self.add_startrek_ticket_comment(comment)

    def on_execute(self):
        from yt.wrapper import YtClient, ypath_join

        with self.memoize_stage.first_run(commit_on_entrance=False):
            if not self.Parameters.print_tables_only:
                # Check that tokens are available
                self.yt_token
                self.startrek_token

            if not any([self.Parameters.oneshot_path, self.Parameters.oneshot_binary_resource, self.Parameters.oneshots_package]):
                raise TaskFailure('There is no oneshot source')

            # Validate `oneshot_path`
            if not self.Parameters.oneshot_binary_resource and self.Parameters.oneshot_path:
                oneshot_path_arcadia_url = svn.Arcadia.parse_url(self.Parameters.oneshot_path)
                logger.debug('oneshot path Arcadia URL: %s', oneshot_path_arcadia_url)
                if oneshot_path_arcadia_url.subpath is None:
                    raise TaskFailure('Failed to parse subpath from oneshot path: {}'
                                      .format(self.Parameters.oneshot_path))

        oneshot_binary_paths = []

        if self.Parameters.oneshot_path or self.Parameters.oneshot_binary_resource:
            if self.Context.oneshot_binary_resource_id is NotExists:
                oneshot_binary_resource = self.get_oneshot_binary_resource()
                self.set_info(
                    'Use oneshot binary from resource {}'
                    .format(get_resource_html_hyperlink(oneshot_binary_resource.id)),
                    do_escape=False
                )
                self.Context.oneshot_binary_resource_id = oneshot_binary_resource.id

            oneshot_binary_resource = sdk2.Resource[self.Context.oneshot_binary_resource_id]
            oneshot_binary_path = str(sdk2.ResourceData(oneshot_binary_resource).path)
            oneshot_binary_paths.append(oneshot_binary_path)

        if self.Parameters.oneshots_package:
            self.set_info(
                'Use oneshot binaries from package {}'
                .format(get_resource_html_hyperlink(self.Parameters.oneshots_package.id)),
                do_escape=False
            )
            package_path = str(sdk2.ResourceData(self.Parameters.oneshots_package).path)
            os.mkdir(ONESHOTS_DIR)
            oneshot_binary_paths += unpack_targz_package(package_path, ONESHOTS_DIR)

        logger.info('Oneshot paths: %s', oneshot_binary_paths)

        with self.memoize_stage.get_oneshot_tables(commit_on_entrance=False):
            oneshot_tables = set()
            for oneshot_binary_path in oneshot_binary_paths:
                oneshot_tables.update(self.get_oneshot_tables(oneshot_binary_path))
            self.Parameters.oneshot_tables = list(oneshot_tables)
            logger.info('Oneshot tables: %s', self.Parameters.oneshot_tables)
            if self.Parameters.print_tables_only:
                return

        ypath_prefix = self.Parameters.ypath_prefix
        if self.Parameters.run_in_test_mode and self.Parameters.run_backup_tables:
            ypath_prefix = self.backup_tables(self.Parameters.oneshot_tables)
            logger.info('YPath prefix: %s', ypath_prefix)
            with self.memoize_stage.set_backup_path(commit_on_entrance=False):
                self.Parameters.backup_dir = ypath_prefix

        for oneshot_binary_path in oneshot_binary_paths:
            oneshot_run_args = [oneshot_binary_path, 'run']
            if ypath_prefix:
                if not ypath_prefix.endswith('/'):
                    ypath_prefix += '/'
                oneshot_run_args += ['--ypath-prefix', ypath_prefix]

            yt_proxy = self.Parameters.yt_proxy
            if self.Parameters.run_in_test_mode:
                yt_proxy = DEFAULT_YT_PROXY
            if yt_proxy:
                oneshot_run_args += ['--proxy', yt_proxy]

            if self.Parameters.oneshot_args:
                oneshot_run_args += ['--oneshot-args', "'{}'".format(self.Parameters.oneshot_args)]
                logger.info('Oneshot arguments: %s', self.Parameters.oneshot_args)

            oneshot_run_command = ' '.join(oneshot_run_args)
            logger.info('Run oneshot: %s', oneshot_run_command)
            try:
                self.Context.oneshot_output = subprocess.check_output(
                    oneshot_run_command,
                    env={'YT_TOKEN': self.yt_token},
                    shell=True,
                    stderr=subprocess.STDOUT,
                )
            except subprocess.CalledProcessError as e:
                logger.exception(e)
                self.Context.oneshot_output = e.output
                self.Context.error_message = 'Oneshot returned non-zero code'
                raise TaskFailure('{error_message}.\n\nOneshot output:\n{oneshot_output}'
                                  .format(error_message=self.Context.error_message,
                                          oneshot_output=self.Context.oneshot_output))
            except Exception as e:
                logger.exception(e)
                self.Context.error_message = 'Failed to execute oneshot binary'
                raise TaskFailure(self.Context.error_message)
            else:
                if self.Parameters.force_data_flush:
                    yt_client = YtClient(proxy=yt_proxy, token=self.yt_token)
                    for table_path in self.Parameters.oneshot_tables:
                        table_backup_path = ypath_join(ypath_prefix, table_path.lstrip('/'))
                        if flush_table(table_backup_path, yt_client):
                            logger.info('Successfully flushed table data: {}'.format(table_backup_path))
                        else:
                            logger.error('Failed to flush table data: {}'.format(table_backup_path))
            finally:
                logger.info('Oneshot output:\n%s', self.Context.oneshot_output)

                self.Parameters.oneshot_output_resource = self.save_oneshot_output(self.Context.oneshot_output)

                self.set_info(
                    '<a href="{resource_link}" target="_blank">Oneshot output</a>'
                    .format(resource_link=self.Parameters.oneshot_output_resource.http_proxy),
                    do_escape=False
                )

    def get_oneshot_binary_resource(self):
        if self.Parameters.oneshot_binary_resource:
            logger.info(
                'Get oneshot binary resource from input parameter: %s',
                self.Parameters.oneshot_binary_resource
            )
            return self.Parameters.oneshot_binary_resource

        if self.Parameters.reuse_oneshot_binary:
            oneshot_parsed_url = svn.Arcadia.parse_url(self.Parameters.oneshot_path)
            oneshot_binary_resource_attrs = dict(
                arcadia_revision=oneshot_parsed_url.revision.replace('r', ''),
                arcadia_branch=oneshot_parsed_url.branch,
                arcadia_tag=oneshot_parsed_url.tag,
                arcadia_trunk=oneshot_parsed_url.trunk,
                oneshot_path=oneshot_parsed_url.subpath,
            )
            logger.info('Find oneshot binary resource with attributes: %s', oneshot_binary_resource_attrs)
            oneshot_binary_resource = YT_ONESHOT_BINARY.find(
                attrs=oneshot_binary_resource_attrs,
                state=State.READY,
            ).first()
            if oneshot_binary_resource is not None:
                logger.info('Found resource %s', oneshot_binary_resource)
                return oneshot_binary_resource
            logger.info('Oneshot binary resource was not found')

        with self.memoize_stage.build_oneshot_binary(commit_on_entrance=False):
            logger.info('Run task to build oneshot binary')
            build_oneshot_binary_task = self.build_oneshot_binary(self.Parameters.oneshot_path)
            self.Context.build_oneshot_binary_task_id = build_oneshot_binary_task.id

        check_tasks(self, [self.Context.build_oneshot_binary_task_id], callback=on_build_oneshot_binary_finish)

        oneshot_binary_resource = YT_ONESHOT_BINARY.find(
            task=sdk2.Task[self.Context.build_oneshot_binary_task_id],
            state=State.READY,
        ).first()
        oneshot_binary_resource.oneshot_path = svn.Arcadia.parse_url(self.Parameters.oneshot_path).subpath

        return oneshot_binary_resource

    @staticmethod
    def get_oneshot_tables(oneshot_binary_path):
        delimiter = ' '
        command = (
            oneshot_binary_path, 'print-tables',
            '--delimiter', delimiter,
        )
        logger.info('Run "%s"', ' '.join(command))
        try:
            oneshot_tables_string = subprocess.check_output(command)
        except Exception as e:
            logger.exception(e)
            raise TaskFailure('Failed to execute oneshot:\n{}'.format(traceback.format_exc()))
        logger.info('Tables affected by oneshot: %s', oneshot_tables_string)

        return oneshot_tables_string.strip().split(delimiter)

    @property
    def yt_token(self):
        try:
            self.__yt_token
        except AttributeError:
            yt_token_vault_item = TESTING_YT_TOKEN_VAULT_ITEM
            if self.Parameters.allow_access_to_production_data:
                yt_token_vault_item = PRODUCTION_YT_TOKEN_VAULT_ITEM
            if self.Parameters.yt_token_vault_name:
                yt_token_vault_item = sdk2.VaultItem(owner_or_name=self.Parameters.yt_token_vault_name)
            logger.info('YT token vault item: %s', yt_token_vault_item)

            self.__yt_token = yt_token_vault_item.data()

        return self.__yt_token

    @property
    def startrek_token(self):
        try:
            self.__startrek_token
        except AttributeError:
            startrek_token_vault_item = STARTREK_TOKEN_VAULT_ITEM
            if self.Parameters.startrek_token_vault_name:
                startrek_token_vault_item = sdk2.VaultItem(owner_or_name=self.Parameters.startrek_token_vault_name)
            logger.info('Startrek token vault item: %s', startrek_token_vault_item)

            self.__startrek_token = startrek_token_vault_item.data()

        return self.__startrek_token

    def build_oneshot_binary(self, oneshot_path):
        """
        Build oneshot binary via YA_MAKE_2 task

        :param oneshot_path: Oneshot directory path. Example: //arcadia.yandex.ru/arc/trunk/arcadia/yabs/qa/yt_oneshot/example
        :return: Enqueued YA_MAKE_2 task
        """
        revision = svn.Arcadia.parse_url(oneshot_path).revision

        logger.debug('revision: %s', revision)
        oneshot_dir = os.path.dirname(oneshot_path)

        logger.debug('oneshot directory: %s', oneshot_dir)
        oneshot_dir_arcadia_url = svn.Arcadia.parse_url(oneshot_dir)

        logger.debug('oneshot directory Arcadia URL: %s', oneshot_dir_arcadia_url)
        # PY2_PROGRAM([progname]) from ya.make
        progname = os.path.splitext(os.path.basename(oneshot_path))[0]
        logger.debug('progname: %s', str(progname))

        checkout_arcadia_from_url = svn.Arcadia.trunk_url(revision=revision)
        if not oneshot_dir_arcadia_url.trunk:
            checkout_arcadia_from_url = svn.Arcadia.branch_url(oneshot_dir_arcadia_url.branch, revision=revision)
        logger.debug('checkout_arcadia_from_url: %s', checkout_arcadia_from_url)

        ya_make_task = sdk2.Task['YA_MAKE_2'](
            self,
            description='Build YT oneshot binary "{}"'.format(oneshot_path),
            checkout_arcadia_from_url=checkout_arcadia_from_url,
            build_system='semi_distbuild',
            targets=oneshot_dir_arcadia_url.subpath,
            arts=os.path.join(oneshot_dir_arcadia_url.subpath, progname),
            result_rt=YT_ONESHOT_BINARY.name,
            result_rd='YT oneshot binary "{}"'.format(oneshot_path),
            result_single_file=True,
            # See details at https://clubs.at.yandex-team.ru/arcadia/20189
            use_aapi_fuse=True,
            aapi_fallback=True,
            use_arc_instead_of_aapi=True,
            ya_yt_token_vault_owner='robot-yabs-cs-sb',
            ya_yt_token_vault_name='yabscs_yt_token',
            sandbox_tags=ctc.Tag.Group.LINUX & ~ctc.Tag.PORTOD,
            result_ttl=str(self.Parameters.oneshot_binary_ttl),
        )
        ya_make_task.enqueue()

        return ya_make_task

    def save_oneshot_output(self, text):
        """Create resource with oneshot output

        :param text: Oneshot output
        :return: Sandbox resource with oneshot output
        """
        oneshot_output_filename = 'oneshot_output.txt'
        with open(oneshot_output_filename, 'w') as oneshot_output_file:
            oneshot_output_file.write(text)
        resource = YT_ONESHOT_OUTPUT(
            self, 'Oneshot output', oneshot_output_filename
        )
        sdk2.ResourceData(resource).ready()
        return resource

    def add_startrek_ticket_comment(self, text):
        if not self.Parameters.startrek_ticket:
            logger.info('Startrek ticket parameter is empty')
            return

        from startrek_client import Startrek
        from startrek_client.exceptions import NotFound

        startrek_client_config = dict(
            useragent=self.__class__.__name__,
            token=self.startrek_token
        )
        if self.Parameters.use_testing_startrek:
            startrek_client_config.update(
                base_url=TESTING_STARTREK_API_ENDPOINT
            )
        startrek_client = Startrek(**startrek_client_config)

        try:
            return startrek_client.issues[self.Parameters.startrek_ticket].comments.create(text=text)
        except NotFound:
            self.set_info('Startrek ticket not found: \'{}\''.format(self.Parameters.startrek_ticket))
            return

    def generate_startrek_ticket_comment(self, status):
        status_color = 'red'
        if status in task_type.Status.Group.SUCCEED:
            status_color = 'green'

        mode = 'Testing' if self.Parameters.run_in_test_mode else 'Production'
        sandbox_task_hyperlink_text = '{mode} oneshot run'.format(mode=mode)
        comment = (
            '{task_hyperlink} finished with !!({task_status_color}){task_status}!!'
            .format(task_hyperlink=startrek_hyperlink(link=get_task_link(self.id), text=sandbox_task_hyperlink_text),
                    task_status_color=status_color,
                    task_status=status)
        )

        if self.Context.oneshot_output is not None:
            comment += textwrap.dedent('''
                <{{ Oneshot output
                %%
                {oneshot_output}
                %%
                }}>
            ''').format(oneshot_output=self.Context.oneshot_output)

        return comment


def default_check_tasks_callback(self, task, failure_message=''):
    if task.status in task_type.Status.Group.BREAK:
        raise TaskError(failure_message)

    if task.status not in task_type.Status.Group.SUCCEED:
        raise TaskFailure(failure_message)


def on_build_oneshot_binary_finish(self, task):
    failure_message = (
        'Failed to build oneshot binary. Task #{task.id} finished with status {task.status}'
        .format(task=task)
    )
    return default_check_tasks_callback(self, task, failure_message)


def on_backup_tables_finish(self, task):
    failure_message = (
        'Failed to backup tables. Task #{task.id} finished with status {task.status}'
        .format(task=task)
    )
    return default_check_tasks_callback(self, task, failure_message)


def flush_table(table_path, yt_client):
    from yt.wrapper import YtError

    initial_state = yt_client.get_attribute(table_path, 'tablet_state')
    if initial_state == 'frozen':
        return True

    try:
        with yt_client.Transaction():
            set_table_state(yt_client, table_path, 'frozen')
            set_table_state(yt_client, table_path, initial_state)
    except YtError as e:
        logger.exception(e)
        return False
    return True
