import logging
import time
from datetime import (
    datetime,
    timedelta,
)

from jinja2 import Environment

from sandbox import sdk2
from sandbox.common.utils import get_task_link
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.common.juggler.jclient import send_events_to_juggler
from sandbox.projects.yabs.qa.solomon.mixin import SolomonTaskMixin, SolomonTaskMixinParameters
from sandbox.projects.yabs.qa.utils.arcadia import get_revision_datetime
from sandbox.projects.release_machine import client
from sandbox.projects.release_machine import rm_notify
from sandbox.projects.release_machine.tasks import base_task as rm_bt
from sandbox.projects.release_machine.tasks.GetLastGoodRevision import (
    GetLastGoodRevision,
    TimeDelta,
)
from sandbox.projects.release_machine.tasks.GetLastGoodRevision.problems import (
    UnresolvedProblem,
    CheckTestIsNotOK,
    RevisionProblem,
    UncheckedInterval,
    IgnoredInterval,
)
from sandbox.projects.yabs.release.duty.schedule import get_current_responsibles


logger = logging.getLogger(__name__)


class InsufficientYabsServerCommits(RevisionProblem):
    template = "Not enough commits to yabs/server before {self.revision}: only {self.commit_count}"

    @property
    def snippet(self):
        return "{self.commit_count} commits to yabs/server in RC".format(self=self)


class HeadTimeDistance(RevisionProblem):
    template = "Revision {self.revision} was commited {self.delay} ago"

    @property
    def snippet(self):
        return "{self.delay}".format(self=self)


class GetLastGoodRevisionMonitor(SolomonTaskMixin, GetLastGoodRevision):
    class Parameters(GetLastGoodRevision.Parameters):
        solomon_parameters = SolomonTaskMixinParameters()

        with sdk2.parameters.Group("Revision distance settings") as distance_settings:
            required_minimal_delay = TimeDelta("Minimal allowed time interval between start revision and desired", description="Interval in seconds or timedelta in HH:MM:SS format", default=0)
            required_max_head_delay = TimeDelta("Maximal allowed time interval between top_limit_revision and desired", description="Interval in seconds or timedelta in HH:MM:SS format", default=0)
            required_minimal_revisions = sdk2.parameters.Integer("Minimal allowed revision interval between start revision and desired", default=0)
            required_yabs_server_commits = sdk2.parameters.Integer("Minimal amount of commits to yabs/server", default=20)

        with sdk2.parameters.Group("Monitoring") as monitoring_settings:
            green_revision_delay_crit_threshold = TimeDelta("Green revision crit threshold", default="48:00:00")
            green_revision_delay_warn_threshold = TimeDelta("Green revision warn threshold", default="24:00:00")

        with sdk2.parameters.Group("Juggler") as juggler_settings:
            juggler_host = sdk2.parameters.String("Juggler host", default="yabs_release")
            juggler_service = sdk2.parameters.String("Juggler service", default="green_revision_delay")

        with sdk2.parameters.Output:
            good_or_stable_revision = sdk2.parameters.Integer("Good or stable revision")

    def get_mention(self, login):
        try:
            mention = "@{}".format(rm_notify.get_mention(
                self,
                person=login,
                component_name=self.Parameters.component_name,
            ))
        except Exception:
            logger.error('Can not fetch tg login for %s', login, exc_info=True)
            mention = "{}@".format(login)
        return mention

    def get_custom_problems(self, revision_candidates):
        custom_problems = {}
        if self.Parameters.required_yabs_server_commits > 0:
            yabs_server_commits = Arcadia.log("arcadia:/arc/trunk/arcadia/yabs/server", revision_from=self.start_revision, revision_to="HEAD")
            yabs_server_revisions = [commit["revision"] for commit in yabs_server_commits]
            logger.info("Got revisions to yabs/server between %s and HEAD: %s", self.start_revision, yabs_server_revisions)
            for r in revision_candidates:
                commit_count = len([rev for rev in yabs_server_revisions if int(rev) < int(r)])
                if commit_count < self.Parameters.required_yabs_server_commits:
                    custom_problems.setdefault(r, []).append(InsufficientYabsServerCommits(r, commit_count=commit_count))

        if self.Parameters.required_max_head_delay > 0:
            top_limit_time = datetime.utcnow()
            for r in revision_candidates:
                r_time = get_revision_datetime(r)
                r_delay = top_limit_time - r_time
                if r_delay.total_seconds() > self.Parameters.required_max_head_delay:
                    custom_problems.setdefault(r, []).append(HeadTimeDistance(r, delay=r_delay - timedelta(microseconds=r_delay.microseconds)))

        return custom_problems

    def generate_message(self, component_info, delay, problems):
        message_template = (
            "Green revision delay is {{ delay }}."
            "{% if n_unresolved_problems > 0 %} {{ n_unresolved_problems }} unresolved TE problems ({{ unresolved_problem_owners }}). {% endif %}"
            "{% if n_other_problems > 0 %} {{ n_other_problems }} other problems ({{ other_problem_owners }}). {% endif %}"
            "Generated by {{ task_link }}"
            " Problems https://beta-testenv.yandex-team.ru/project/yabs-2.0/problems?limit=20&status=unresolved"
        )

        n_unresolved_problems = 0
        unresolved_problem_owners = set()
        n_other_problems = 0
        other_problem_owners = set()

        for problem in problems:
            if isinstance(problem, UnresolvedProblem):
                mention = self.get_mention(problem.owner)
                unresolved_problem_owners.add(mention)
                n_unresolved_problems += 1
            elif not isinstance(problem, CheckTestIsNotOK):
                sandbox_duty_list = get_current_responsibles("SANDBOX")
                sandbox_duty = sandbox_duty_list[0] if sandbox_duty_list else (component_info.notify_cfg__st__assignee if component_info else "undefined")
                mention = self.get_mention(sandbox_duty)
                other_problem_owners.add(mention)
                n_other_problems += 1

        env = Environment()
        template = env.from_string(message_template)
        return template.render(
            delay=str(delay),
            n_unresolved_problems=n_unresolved_problems,
            unresolved_problem_owners=", ".join(unresolved_problem_owners),
            n_other_problems=n_other_problems,
            other_problem_owners=", ".join(other_problem_owners),
            task_link=get_task_link(self.id),
        )

    @staticmethod
    def get_last_scope_revision(component_info, rm_client):
        scopes_response = rm_client.get_scopes(component_name=component_info.name, limit=1)
        scopes = scopes_response.get("branchScopes", [])
        if scopes:
            return scopes[0]["branch"]["baseCommitId"]

        return component_info.first_rev

    def _filter_out_problem(self, revision_problems, ignored_problem_class):
        filtered_revision_problems = {}
        for _revision, _problems in revision_problems.items():
            filtered_revision_problems[_revision] = [_problem for _problem in _problems if not isinstance(_problem, ignored_problem_class)]
        return filtered_revision_problems

    def _get_last_good_revision_or_default(self, revision_problems):
        last_good_revision = self.get_last_good_revision(revision_problems)

        if not last_good_revision:
            rm_client = client.RMClient()
            last_good_revision = self.get_last_scope_revision(self.component_info, rm_client)
            logger.info("No good metric revision found, will use last release revision %s instead", last_good_revision)

        return last_good_revision

    def _get_revision_delay_sensor(self, revision, now_datetime):
        good_revision_datetime = get_revision_datetime(revision)
        logger.info("Now: %s, good_revision %d was commited at %s", now_datetime, revision, good_revision_datetime)
        delay = now_datetime - good_revision_datetime
        return delay

    def send_revision_metrics(self, release_revision_problems):
        revision_problems = self._filter_out_problem(release_revision_problems, HeadTimeDistance)
        good_revision = self._get_last_good_revision_or_default(revision_problems)

        utcnow_datetime = datetime.utcnow()
        now_ts = time.time()
        delay = self._get_revision_delay_sensor(good_revision, utcnow_datetime)

        sensors = [
            {
                "labels": {
                    "sensor": "green_revision_delay_sec",
                },
                "value": delay.total_seconds(),
                "ts": now_ts,
            }
        ]

        for problem_class in (UnresolvedProblem, UncheckedInterval, IgnoredInterval, InsufficientYabsServerCommits):
            _revision_problems_with_ignored_problem_type = self._filter_out_problem(revision_problems, problem_class)
            _metric_revision = self._get_last_good_revision_or_default(_revision_problems_with_ignored_problem_type)
            sensors.append({
                "labels": {
                    "sensor": "conditional_green_revision_delay_sec",
                    "ignored": problem_class.__name__,
                },
                "value": self._get_revision_delay_sensor(_metric_revision, utcnow_datetime).total_seconds(),
                "ts": now_ts,
            })

        self.solomon_push_client.add(sensors)

        last_rev, last_rev_problems = max(revision_problems.items(), key=lambda x: x[0])
        component_info = None
        if self.Parameters.component_name:
            component_info = self.component_info
        message = self.generate_message(component_info, delay, last_rev_problems)
        logger.debug('Message is:\n%s', message)
        if delay > timedelta(seconds=self.Parameters.green_revision_delay_crit_threshold):
            status = "CRIT"
        elif delay > timedelta(seconds=self.Parameters.green_revision_delay_warn_threshold):
            status = "WARN"
        else:
            status = "OK"
        send_events_to_juggler(self.Parameters.juggler_host, self.Parameters.juggler_service, status, message)

    def on_execute(self):
        rm_bt.BaseReleaseMachineTask.on_execute(self)

        release_revision_problems = self.get_revision_problems()
        release_revision = self.get_last_good_revision(release_revision_problems)

        try:
            self.report_problems(release_revision_problems)
        except Exception as exc:
            logger.error(exc)

        if release_revision is not None:
            self.Parameters.good_revision = release_revision
            self.Context.good_revision = release_revision

        self.Parameters.good_or_stable_revision = self._get_last_good_revision_or_default(release_revision_problems)

        self.send_revision_metrics(release_revision_problems)
