import json
import logging

from sandbox import sdk2
from sandbox.projects.yabs.qa.resource_types import YabsFtSmartMetaReport
from sandbox.projects.common.testenv_client.api_client import TestenvApiClient
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.report_utils import ELogChange, ELogFieldChange
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp import YabsServerB2BFuncShootCmp
from sandbox.projects.yabs.qa.tasks.YabsServerCompareCSVersions import YabsServerCompareCSVersions
from sandbox.projects.yabs.qa.tasks.YabsAutoResolveDiffs.process_smart_report import need_autoresolve


logger = logging.getLogger(__name__)


class YabsAutoResolveDiffs(sdk2.Task):
    """ Automatically resolves testenv diffs base on diff metainfo from CMP tasks
    """

    class Parameters(sdk2.Parameters):
        testenv_project = sdk2.parameters.String('Testenv project', default='yabs-2.0')

        logs_data = sdk2.parameters.Bool('Automatically resolve logs_data diffs', default=True)
        with logs_data.value[True]:
            added_logs = sdk2.parameters.Bool('Added logs', default=True)
            removed_logs = sdk2.parameters.Bool('Removed logs', default=False)
            changed_logs = sdk2.parameters.Bool('Changed logs', default=True)  # deprecated

            added_fields = sdk2.parameters.Bool('Added fields', default=True)
            removed_fields = sdk2.parameters.Bool('Removed fields', default=False)
            changed_fields = sdk2.parameters.Bool('Changed fields', default=False)

        with sdk2.parameters.Group('Auth parameters') as auth_parameters:
            yav_secret_id = sdk2.parameters.String('Secret id', required=True)
            yav_secret_version = sdk2.parameters.String('Version of secret')

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 4 * (1 << 10)

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    def allowed_changes(self):
        return {
            ELogChange.added_logs.name: self.Parameters.added_logs,
            ELogChange.removed_logs.name: self.Parameters.removed_logs,
            ELogChange.changed_logs.name: {
                ELogFieldChange.added_fields.name: self.Parameters.added_fields,
                ELogFieldChange.removed_fields.name: self.Parameters.removed_fields,
                ELogFieldChange.changed_fields.name: self.Parameters.changed_fields,
            }
        }

    def need_autoresolve(self, problem, allowed_changes):
        cmp_task_id = problem['diff']['task_id']
        cmp_task = sdk2.Task[cmp_task_id]
        autoresolve = False
        msg = ''
        if isinstance(cmp_task, YabsServerB2BFuncShootCmp):
            smart_metareport_resource = YabsFtSmartMetaReport.find(task_id=cmp_task_id, limit=1).first()
            if not smart_metareport_resource:
                logger.warning('Diff %s (task %s) has no YabsFtSmartMetaReport resource', problem.get('id'), cmp_task_id)
                return False, ''

            with open(str(sdk2.ResourceData(smart_metareport_resource).path)) as smart_report_file:
                smart_report = json.load(smart_report_file)

            if set(smart_report['diff_blocks']) != {'logs_data'}:
                logger.info('Diff %s (task %s) has diff in %s', problem.get('id'), cmp_task_id, smart_report['diff_blocks'])
                return False, ''

            autoresolve, diff_changes = need_autoresolve(smart_report, allowed_changes)

            msg = 'Autoresolved by {}. {}'.format(self.id, '. '.join(diff_changes))
            if len(msg) > 1000:  # Max len for resolve_comment is 1000
                suffix = '...(truncated)'
                msg = msg[:(1000 - len(suffix))] + suffix
        elif isinstance(cmp_task, YabsServerCompareCSVersions):
            autoresolve = True
            msg = 'Autoresolved by {}. {}'.format(self.id, 'CS version changes do not require manual resolve')

        return autoresolve, msg

    @staticmethod
    def get_unresolved_problems(testenv_project, testenv_api_client):
        return testenv_api_client['projects'][testenv_project]['problems'].GET(
            params={
                'status': 'unresolved'
            }
        )

    @staticmethod
    def resolve_problems(testenv_project, problems, testenv_api_client):
        resolve_body = []
        for problem_id, resolve_message in problems:
            resolve_body.append({
                'diff_id': problem_id,
                'patch': {
                    'status': 'resolved',
                    'resolve_comment': resolve_message,
                }
            })

        if not resolve_body:
            logger.info('Nothing to resolve')
            return []

        logger.debug('Will resolve problems: %s', json.dumps(resolve_body, indent=2))
        response = testenv_api_client['projects'][testenv_project]['bulk_update_problems'].PATCH(json=resolve_body)
        logger.debug('Response was: %s', json.dumps(response, indent=2))

        resolved_diffs = [result for result in response if result['result']['status'] == 'SUCCESS']
        unresolved_diffs = [result for result in response if result['result']['status'] != 'SUCCESS']
        logger.info('Resolved diffs: %s', ', '.join([str(result['diff_id']) for result in resolved_diffs]))
        if unresolved_diffs:
            logger.error('Not resolved diffs: %s', '\n'.join(['{}: {}'.format(result['diff_id'], result['result']['reason']) for result in unresolved_diffs]))

        return resolved_diffs

    def on_execute(self):
        secret = sdk2.yav.Secret(self.Parameters.yav_secret_id, self.Parameters.yav_secret_version)
        testenv_token = secret.data()['testenv_token']
        testenv_api_client = TestenvApiClient(token=testenv_token)
        unresolved_problems = self.get_unresolved_problems(self.Parameters.testenv_project, testenv_api_client)

        problems_to_resolve = []
        for problem in unresolved_problems:
            try:
                autoresolve, resolve_message = self.need_autoresolve(problem, self.allowed_changes)
                if autoresolve:
                    problems_to_resolve.append((problem['id'], resolve_message))
            except:
                logger.error('Cannot autoresolve diff_id=%s due to an exception', problem.get('id'), exc_info=True)

        if problems_to_resolve:
            resolved_diffs = self.resolve_problems(self.Parameters.testenv_project, problems_to_resolve, testenv_api_client)
            if resolved_diffs:
                self.set_info("Resolved following diffs in {}: {}".format(self.Parameters.testenv_project, ', '.join([str(result['diff_id']) for result in resolved_diffs])))
