import re
import six
import logging

from sandbox import sdk2
from sandbox.common.utils import Enum
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.testenv_client.api_client import TestenvApiClient

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace_calls, trace_entry_point


logger = logging.getLogger(__name__)


# "Role": "Shoot test name"
EXPECT_DIFF_ROLES = {
    'bs': ['YABS_SERVER_40_FT_BS_SAMPLED', 'YABS_SERVER_40_FT_BS_A_B'],
    'yabs': ['YABS_SERVER_40_FT_YABS_SAMPLED', 'YABS_SERVER_40_FT_YABS_A_B'],
    'bsrank': ['YABS_SERVER_40_FT_BSRANK_SAMPLED', 'YABS_SERVER_40_FT_BSRANK_A_B'],
    'all_roles': [
        'YABS_SERVER_40_FT_BS_SAMPLED', 'YABS_SERVER_40_FT_BS_A_B',
        'YABS_SERVER_40_FT_YABS_SAMPLED', 'YABS_SERVER_40_FT_YABS_A_B',
        'YABS_SERVER_40_FT_BSRANK_SAMPLED', 'YABS_SERVER_40_FT_BSRANK_A_B'
    ],
}


class YabsBinarySearchHelper(sdk2.Task):

    class Parameters(sdk2.Parameters):
        pre_revision = sdk2.parameters.Integer("Pre shoot task revision", required=True)
        test_revision = sdk2.parameters.Integer("Test shoot task revision", required=True)
        find_key = sdk2.parameters.Dict("Attrs to find resource", required=True)
        test_name = sdk2.parameters.String("Test name", required=True)
        jobs_json = sdk2.parameters.JSON("Which jobs should be started for diff in key-job", default={})
        diff_resource_type = sdk2.parameters.String("Meta report with current diff")
        with sdk2.parameters.Group('TestEnv parameters') as auth_parameters:
            testenv_project = sdk2.parameters.String('Testenv project', default='yabs-2.0')
            testenv_token = sdk2.parameters.YavSecret('Testenv token', default="sec-01d6apzcex5fpzs5fcw1pxsfd5")
        with sdk2.parameters.Output():
            report = sdk2.parameters.String("Report about started jobs")

    def set_info_about_diffs(self, revisions_for_info):
        """Setting information about status of revision with diff and previous one. revisions_for_info is list of tuples like (revision_with_diff, previous_revision)"""
        if not revisions_for_info:
            self.set_info("There are no revisions with diff")
            return
        info_data = []
        for curr_revision, prev_revision in revisions_for_info:
            info_data.append("Revision {curr_revision[revision]} with diff has status {curr_revision[status]}. Previous revision {prev_revision[revision]} has status {prev_revision[status]}".format(
                curr_revision=curr_revision, prev_revision=prev_revision)
            )
        self.set_info('\n'.join(info_data))

    def run_test_on_revisions_with_diff(self, job_revisions, revisions_with_diff, testenv_api_client, testenv_project, job):
        revisions_to_run = set()
        revisions_for_info = []
        for i in six.moves.range(1, len(job_revisions)):  # first revision can't give diff
            if job_revisions[i]['revision'] not in revisions_with_diff:
                continue
            revisions_for_info.append((job_revisions[i], job_revisions[i - 1]))
            curr_revision = find_not_checked_revision(job_revisions[i:len(job_revisions) - 1])  # current. Revision len-1 has already completed
            prev_revision = find_not_checked_revision(job_revisions[i - 1:0:-1])  # previous. Revision 0 has already completed
            revisions_to_run.update(filter(lambda x: x is not None, [prev_revision, curr_revision]))
        self.set_info_about_diffs(revisions_for_info)
        if revisions_to_run:
            try:
                with self.memoize_stage["run_job " + job](commit_on_entrance=False, max_runs=3):
                    run_job(testenv_api_client, testenv_project, job, list(revisions_to_run))
            except Exception as e:
                self.Parameters.report = "Can't run job on revisions {}. {}".format(revisions_to_run, e.message)
                raise TaskFailure("Can't run job on revisions {}. {}".format(revisions_to_run, e.message))
            else:
                self.set_info("Run job on revisions {}".format(revisions_to_run))
                self.Parameters.report = "Run job on revisions {}".format(revisions_to_run)
                self.Parameters.tags = ['STARTED_JOB']
        else:
            self.Parameters.report = "There are no revisions to run"
            self.set_info("There are no revisions to run")
            self.Parameters.tags = ['NO_REVISIONS']

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        testenv_token = self.Parameters.testenv_token.data()["testenv_token"]
        testenv_api_client = TestenvApiClient(token=testenv_token)
        job_revisions = get_revisions(
            testenv_api_client,
            self.Parameters.testenv_project,
            self.Parameters.test_name,
            self.Parameters.pre_revision,
            self.Parameters.test_revision,
        )
        revisions_with_precommit_diff = get_precommit_diffs(
            self.Parameters.find_key, job_revisions, self.Parameters.diff_resource_type, self.Parameters.test_name
        )
        if self.Parameters.test_name in self.Parameters.jobs_json:
            jobs = self.Parameters.jobs_json[self.Parameters.test_name]
        else:
            jobs = [self.Parameters.test_name]
        [self.run_test_on_revisions_with_diff(
            job_revisions,
            revisions_with_precommit_diff,
            testenv_api_client,
            self.Parameters.testenv_project,
            job,
        ) for job in jobs]


# HELPER


class TestStatus(Enum):
    OK = "ok"
    EXECUTING = "executing"
    WAIT = "wait_parent"
    NOT_CHECKED = "not_checked"


@trace_calls
def get_job_revisions(testenv_api_client, testenv_project, job, revision_gte, revision_lte):
    return testenv_api_client['projects'][testenv_project]['jobs'][job]['runs'].GET(
        params={
            'revision_gte': revision_gte,
            'revision_lte': revision_lte,
            'show_unchecked': 'true',
            'show_filtered': 'false',
        }
    )


def get_review_id(commit_message):
    review = re.findall(r'REVIEW: (\d+)', commit_message)
    if len(review) == 1:
        return int(review[0])
    return


@trace_calls
def get_revisions(testenv_api_client, testenv_project, job, revision_gte, revision_lte):
    job_revisions = get_job_revisions(testenv_api_client, testenv_project, job, revision_gte, revision_lte)
    revisions_data = []
    for revision in job_revisions:
        review = get_review_id(revision['commit']['message'])
        fixed_revision_marker = re.search(r'\[fix:yabs_server:(\d*)\]', revision['commit']['message'])
        if fixed_revision_marker:
            fixed_revision = int(fixed_revision_marker.group(1))
            job_revisions.extend(get_job_revisions(testenv_api_client, testenv_project, job, fixed_revision, fixed_revision))
            logging.debug('Found [fix:yabs_server] marker with %s revision', fixed_revision)
        revisions_data.append(
            {'revision': int(revision['commit']['revision']), 'status': revision['run']['status'], 'review': review, 'message': revision['commit']['message']}
        )
    revisions_data.sort(key=lambda x: x['revision'])
    logger.debug("Revisions of job: %s", revisions_data)
    return revisions_data


@trace_calls
def get_precommit_diffs(find_key, job_revisions, resource_type, job):
    revisions_with_diff = set()
    for revision in job_revisions:
        checked_tests = set()
        review = revision['review']
        if review is None:
            continue
        expect_diff_marker = re.search(r'\[yabs_expect_diff_on_shm:(\w*)\]', revision['message'])
        if expect_diff_marker:
            roles = expect_diff_marker.group(1).split(',')
            for role in roles:
                if role in EXPECT_DIFF_ROLES and job in EXPECT_DIFF_ROLES[role]:
                    logger.info('Found revision: %s with expect diff marker', revision['revision'])
                    revisions_with_diff.add(revision['revision'])
                    continue
        find_key['arcanum_review_id'] = review
        resources = sdk2.Resource.find(type=resource_type, attrs=find_key).limit(100)
        for resource in resources:
            if not resource or resource.test_name in checked_tests:
                continue
            checked_tests.add(resource.test_name)
            if getattr(resource, 'has_diff') in ('True', True):
                revisions_with_diff.add(revision['revision'])
                logger.info("Revision %s has diff", revision['revision'])
                break
    return revisions_with_diff


# START OF MODE force_run_shoot_tasks
@trace_calls
def run_job(testenv_api_client, testenv_project, job, revisions):
    logger.info("Run job %s on revisions %s", job, revisions)
    testenv_api_client['projects'][testenv_project]['jobs'][job]['runs'].POST(
        data="{{ \"revisions\": {}}}".format(revisions)  # comma-separated list
    )


def find_not_checked_revision(job_revisions):
    for revision in job_revisions:
        if revision['status'] in (TestStatus.OK, TestStatus.EXECUTING, TestStatus.WAIT):
            return  # if the nearest revision was already run, we don't need to do anything
        if revision['status'] == TestStatus.NOT_CHECKED:
            return revision['revision']
    return
# END OF MODE force_run_shoot_tasks
