import json
import logging
from datetime import tzinfo, timedelta, datetime

from sandbox import sdk2
from sandbox.common.rest import Client
from sandbox.common.types.resource import State
from sandbox.common.types.notification import Transport

from sandbox.projects.yabs.qa.solomon.mixin import SolomonTaskMixin, SolomonTaskMixinParameters
from sandbox.projects.yabs.qa.tasks.YabsServerCreateOneShotSpec.spec import YABS_SERVER_HOUSE_SPEC

COMMON_RAM = 1024
COMMON_DISK_SPACE = 100

ZERO = timedelta(0)


class UTC(tzinfo):
    def utcoffset(self, dt):
        return ZERO

    def tzname(self, dt):
        return "UTC"

    def dst(self, dt):
        return ZERO


utc = UTC()


class YabsCollectOneShotData(SolomonTaskMixin, sdk2.Task):
    '''SDK2 task that collects some YABS_SERVER_ONE_SHOT tasks statistics and pushes it to Solomon'''

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = COMMON_RAM
        disk_space = COMMON_DISK_SPACE

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        solomon_parameters = SolomonTaskMixinParameters()

    def on_create(self):
        super(YabsCollectOneShotData, self).on_create()
        if self.Context.copy_of:
            self.Parameters.notifications = [
                notification for notification in self.Parameters.notifications
                if notification.transport != Transport.JUGGLER
            ]

    def get_resource_age(self, resource_id):
        created = sdk2.Resource[resource_id].created
        now = datetime.now(utc)
        logging.debug('Resource #%s created at %s, now %s, age is %s', resource_id, now, now - created)
        return int((now - created).total_seconds())

    def get_task_count_sensors(self):
        sensors = []

        for status in ['SUCCESS', 'FAILURE', 'EXCEPTION', 'TIMEOUT']:
            try:
                sensors.append({
                    'labels': {
                        'sensor': 'task_count',
                        'status': status,
                    },
                    'value': self.rest_client.task.read(
                        type="YABS_SERVER_ONE_SHOT",
                        status=status,
                        limit=1,
                    ).get('total', 0)
                })
            except Exception as exc:
                logging.error("Cannot get '%s' task_count due to %s", status, exc, exc_info=True)

        return sensors

    def get_spec_resources(self):
        (resource_func, resource_load) = (sdk2.Resource.find(
            YABS_SERVER_HOUSE_SPEC,
            attrs={
                attr: 'True',
                'released_spec': 'True'
            },
            state=State.READY,
        ).order(-sdk2.Resource.id).first() for attr in ('good', 'good_with_load'))

        logging.info("Found func_spec #%d, load_spec #%d", resource_func.id, resource_load.id)

        with open(str(sdk2.ResourceData(resource_func).path)) as spec_func_file, \
                open(str(sdk2.ResourceData(resource_load).path)) as spec_load_file:
            spec_func = json.load(spec_func_file)
            spec_load = json.load(spec_load_file)

        return {
            'mysql_func': spec_func['mysql_archive_resource_id'],
            'yt_func': spec_func['cs_input_spec_resource_id'],
            'cs_settings_func': spec_func['cs_settings_archive_resource_id'],
            'ammo_bs_func': spec_func['ft_request_log_resource_id_map']['bs'],
            'ammo_bsrank_func': spec_func['ft_request_log_resource_id_map']['bsrank'],
            'ammo_yabs_func': spec_func['ft_request_log_resource_id_map']['yabs'],
            'stat_bs_release_func': spec_func['stat_bs_release_tar_resource_id'],
            'meta_bs_release_func': spec_func['meta_bs_release_tar_resource_id'],

            'mysql_load': spec_load['mysql_archive_resource_id'],
            'yt_load': spec_load['cs_input_spec_resource_id'],
            'cs_settings_load': spec_load['cs_settings_archive_resource_id'],
            'ammo_bs_load': spec_load['stat_load_request_log_resource_id_map']['bs'],
            'ammo_bsrank_load': spec_load['stat_load_request_log_resource_id_map']['bsrank'],
            'ammo_yabs_load': spec_load['stat_load_request_log_resource_id_map']['yabs'],
            'stat_bs_release_load': spec_func['stat_bs_release_tar_resource_id'],
            'meta_bs_release_load': spec_func['meta_bs_release_tar_resource_id'],
        }

    def get_resource_age_sensors(self, resources_from_spec):
        sensors = []

        for tier, resource_id in resources_from_spec.iteritems():
            try:
                sensors.append(
                    {
                        'labels': {
                            'sensor': 'resource_age',
                            'tier': tier,
                        },
                        'value': self.get_resource_age(resource_id)
                    }
                )
            except Exception as exc:
                logging.error("Cannot get '%s' resource #%d age due to %s", tier, resource_id, exc, exc_info=True)

        return sensors

    def get_version_sensors(self, resources_from_spec):
        from sandbox.projects.yabs.release.version.sandbox_helpers import SandboxHelper
        sandbox_helper = SandboxHelper(sandbox_client=self.server)
        return [
            {
                'labels': {
                    'tier': tier,
                    'sensor': 'version',
                    'version': str(sandbox_helper.get_full_version_from_resource(resources_from_spec[tier])),
                },
                'value': 1
            }
            for tier in ('stat_bs_release_func', 'stat_bs_release_load', 'meta_bs_release_func', 'meta_bs_release_load')
        ]

    def on_execute(self):
        self.rest_client = Client()
        resources_from_spec = self.get_spec_resources()

        sensors_list = []
        sensors_list.extend(self.get_task_count_sensors())
        sensors_list.extend(self.get_version_sensors(resources_from_spec))
        sensors_list.extend(self.get_resource_age_sensors(resources_from_spec))

        logging.info('Got sensors: %s', json.dumps(sensors_list, indent=2))
        self.solomon_push_client.add(sensors_list, cluster='sandbox_one_shot', service='one_shot')
