import itertools
import logging
import os
from enum import Enum

from jinja2 import Environment, FunctionLoader
from sandbox.common.types.task import Status
from sandbox.projects.yabs.qa.clickhouse.http_client import Format

logger = logging.getLogger(__name__)


tasks = "tasks"
tags = "tags"


class TestSet(Enum):
    autorun = 0
    brave = 1  # deprecated
    sampled = 2
    regular = 3

    brave_ft = 4
    brave_sanitize = 5
    brave_performance = 6


class Stages(Enum):
    """ DO NOT CHANGE EXISTING ENUM ITEMS
    It will mess up solomon sensors
    """
    overall = 0  # without (build + setup_ya_make)

    build = 10  # build + setup_ya_make

    base_gen = 20
    ft_shoot = 30
    big_ft_shoot = 40
    load_shoot = 50
    sanitize_shoot = 60


BUILD_TASKS = [
    'BUILD_YABS_SERVER',
    'YABS_SERVER_SETUP_YA_MAKE',
    'YA_PACKAGE',
    'YA_PACKAGE_2',
]
FT_TASKS = [
    'YABS_SERVER_B2B_FUNC_SHOOT_2',
    'YABS_SERVER_B2B_FUNC_SHOOT_CMP',
    'YABS_SERVER_B2B_FUNC_SHOOT_STABILITY',
    'YABS_SERVER_VALIDATE_RESPONSES',
    'YABS_SERVER_VALIDATE_RESPONSES_CMP',
]
LOAD_TASKS = [
    'YABS_SERVER_STAT_PERFORMANCE_BEST_2',
    'YABS_SERVER_STAT_PERFORMANCE_BEST_CMP_2',
]
BASE_GEN_TASKS = [
    'YABS_SERVER_MAKE_BIN_BASES',
    'YABS_SERVER_BASES_REDUCE',
    'YABS_SERVER_BASES_FETCH',
    'YABS_SERVER_REAL_RUN_CS_IMPORT',
    'YABS_SERVER_RUN_CS_IMPORT_WRAPPER',
]
BRAVE_TESTS_TAG = {
    'ft': 'TESTENV-JOB-YABS_SERVER_BRAVE_TESTS_FT',
    'performance': 'TESTENV-JOB-YABS_SERVER_BRAVE_TESTS_PERFORMANCE',
    'sanitize': 'TESTENV-JOB-YABS_SERVER_BRAVE_TESTS_SANITIZE',
}


STAGES_TASKS = {
    TestSet.autorun.name: {
        Stages.build.name: {
            tasks: BUILD_TASKS,
            tags: [
                'TESTENV-JOB-YABS_SERVER_11_BUILD_RELEASE',
                'TESTENV-JOB-YABS_SERVER_12_BUILD_PROFILE',
                'TESTENV-JOB-YABS_SERVER_14_BUILD_SANITIZE_ADDRESS',
                'TESTENV-JOB-YABS_SERVER_15_BUILD_SANITIZE_MEMORY',
                'TESTENV-JOB-YABS_SERVER_20_SETUP_YA_MAKE',
            ]
        },
    },
    TestSet.brave_ft.name: {
        Stages.overall.name: {
            tasks: [
                'YABS_SERVER_RUN_BRAVE_TESTS',
            ],
            tags: [
                BRAVE_TESTS_TAG['ft'],
            ],
        },
        Stages.build.name: {
            tasks: BUILD_TASKS,
            tags: [
                BRAVE_TESTS_TAG['ft'],
            ],
        },
        Stages.ft_shoot.name: {
            tasks: FT_TASKS,
            tags: [
                BRAVE_TESTS_TAG['ft'],
            ],
        },
    },
    TestSet.brave_sanitize.name: {
        Stages.overall.name: {
            tasks: [
                'YABS_SERVER_RUN_BRAVE_TESTS',
            ],
            tags: [
                BRAVE_TESTS_TAG['sanitize'],
            ],
        },
        Stages.build.name: {
            tasks: BUILD_TASKS,
            tags: [
                BRAVE_TESTS_TAG['sanitize'],
            ],
        },
        Stages.sanitize_shoot.name: {
            tasks: ['YABS_SERVER_STAT_PERFORMANCE_SANITIZE'],
            tags: [
                BRAVE_TESTS_TAG['sanitize'],
            ],
        },
    },
    TestSet.brave_performance.name: {
        Stages.overall.name: {
            tasks: [
                'YABS_SERVER_RUN_BRAVE_TESTS',
            ],
            tags: [
                BRAVE_TESTS_TAG['performance'],
            ],
        },
        Stages.build.name: {
            tasks: BUILD_TASKS,
            tags: [
                BRAVE_TESTS_TAG['performance'],
            ],
        },
        Stages.load_shoot.name: {
            tasks: LOAD_TASKS,
            tags: [
                BRAVE_TESTS_TAG['performance'],
            ],
        },
    },
    TestSet.sampled.name: {
        Stages.base_gen.name: {
            tasks: BASE_GEN_TASKS,
            tags: [
                'TESTENV-JOB-YABS_SERVER_30_BASE_GEN_{suffix}'.format(suffix=suffix)
                for suffix in [
                    'COMMON_SAMPLED'
                ] + [
                    '{role}_A_SAMPLED'.format(role=role)
                    for role in ('BS', 'BSRANK', 'YABS')
                ]
            ],
        },
        Stages.ft_shoot.name: {
            tasks: FT_TASKS,
            tags: [
                'TESTENV-JOB-YABS_SERVER_40_FT_{role}_SAMPLED'.format(role=role)
                for role in ('BS', 'BSRANK', 'YABS')
            ] + [
                'TESTENV-JOB-YABS_SERVER_41_VALIDATE_RESPONSES_{role}_SAMPLED'.format(role=role)
                for role in ('BS', 'BSRANK', 'YABS')
            ] + [
                'TESTENV-JOB-YABS_SERVER_402_FT_STABILITY_{role}_SAMPLED'.format(role=role)
                for role in ('BS', 'BSRANK', 'YABS')
            ],
        },
    },
    TestSet.regular.name: {
        Stages.base_gen.name: {
            tasks: BASE_GEN_TASKS,
            tags: [
                'TESTENV-JOB-YABS_SERVER_30_BASE_GEN_{shard}'.format(shard=shard)
                for shard in ['COMMON', 'A', 'B', 'C']
            ],
        },
        Stages.ft_shoot.name: {
            tasks: FT_TASKS,
            tags: [
                'TESTENV-JOB-YABS_SERVER_40_FT_{role}'.format(role=role)
                for role in ('BS', 'BSRANK', 'YABS')
            ] + [
                'TESTENV-JOB-YABS_SERVER_41_VALIDATE_RESPONSES_{role}'.format(role=role)
                for role in ('BS', 'BSRANK', 'YABS')
            ] + [
                'TESTENV-JOB-YABS_SERVER_402_FT_STABILITY_{role}'.format(role=role)
                for role in ('BS', 'BSRANK', 'YABS')
            ],
        },
        Stages.big_ft_shoot.name: {
            tasks: FT_TASKS,
            tags: [
                'TESTENV-JOB-YABS_SERVER_40_FT_BIG_B2B_{role}_OTHER'.format(role=role)
                for role in ('BS', 'BSRANK', 'YABS')
            ] + [
                'TESTENV-JOB-YABS_SERVER_40_FT_{role}_A_B'.format(role=role)
                for role in ('BS', 'BSRANK', 'YABS')
            ],
        },
        Stages.load_shoot.name: {
            tasks: LOAD_TASKS,
            tags: [
                'TESTENV-JOB-YABS_SERVER_40_PERFORMANCE_BEST_{role}{shard}'.format(role=role, shard='_{}'.format(shard) if shard != 'A' else '')
                for role in ('BS', 'BSRANK', 'YABS')
                for shard in ('A', 'B', 'C')
            ]
        },
        Stages.sanitize_shoot.name: {
            tasks: ['YABS_SERVER_STAT_PERFORMANCE_SANITIZE'],
            tags: [
                'TESTENV-JOB-YABS_SERVER_50_SANITIZE_{sanitizer}_{role}{shard}'.format(role=role, shard='_{}'.format(shard) if shard != 'A' else '', sanitizer=sanitizer)
                for role in ('BS', 'BSRANK', 'YABS')
                for shard in ('A', 'B', 'C')
                for sanitizer in ('MEMORY', 'ADDRESS')
            ],
        },
    }
}
TASK_STATUSES = [Status.EXECUTING, Status.ENQUEUED, Status.WAIT_TASK]


def load_template(template_name):
    template_path = os.path.normpath(
        os.path.join(
            os.path.dirname(__file__),
            'templates',
            template_name,
        )
    )
    try:
        import library.python.resource as lpr
    except ImportError:
        with open(template_path, 'r') as f:
            template_text = f.read()
    else:
        template_text = lpr.find(template_path)
    return template_text


def get_template(template_name):
    env = Environment(loader=FunctionLoader(load_template))
    return env.get_template(template_name)


def get_precommit_checks_sensors(clickhouse_client, owners, stages=STAGES_TASKS, statuses=TASK_STATUSES, days=2,
                                 percentiles=(0.50, 0.75, 0.95, 1)):
    sensors = []
    all_task_filters = {
        ('ALL', Stages.overall.name): {
            tasks: [],
            tags: [],
        }
    }
    for test_set, test_set_stages in stages.items():
        for stage, task_filters in test_set_stages.items():
            all_task_filters[(test_set, stage)] = task_filters

        aggregated_filters = [
            task_filters
            for stage_name, task_filters in test_set_stages.items()
        ]
        task_list = list(itertools.chain(*(
            task_filters[tasks]
            for task_filters in aggregated_filters
        )))
        tag_list = list(itertools.chain(*(
            task_filters.get(tags, [])
            for task_filters in aggregated_filters
        )))

        all_task_filters[(test_set, Stages.overall.name)] = {
            tasks: task_list,
            tags: tag_list,
        }
        all_task_filters[('ALL', Stages.overall.name)] = {
            tasks: list(set(all_task_filters[('ALL', Stages.overall.name)][tasks] + task_list)),
            tags: list(set(all_task_filters[('ALL', Stages.overall.name)][tags] + tag_list)),
        }

    logger.debug('Task filters are:\n%s', all_task_filters)

    for (test_set, stage), task_filters in all_task_filters.items():
        try:
            stage_sensors = generate_stage_sensors(
                clickhouse_client, days, owners, task_filters[tasks], task_filters.get(tags, []),
                percentiles, stage, test_set,
            )
        except Exception:
            logger.error("Cannot get stage durations for %s %s", test_set, stage, exc_info=True)

        try:
            statuses_sensors = generate_status_durations(
                clickhouse_client,
                days, owners, task_filters[tasks], statuses, task_filters.get(tags, []),
                percentiles, stage, test_set,
            )
            sensors.extend(stage_sensors)
            sensors.extend(statuses_sensors)
        except Exception:
            logger.error("Cannot get status durations for %s %s", test_set, stage, exc_info=True)

    return sensors


def create_sensor(stage_duration_percentiles, percentiles, stage, test_set, ordered_stage, sensor, **kwargs):
    return [
        {
            'labels': dict(
                percentile='q{}'.format(int(percentile * 100)),
                stage=stage,
                ordered_stage=ordered_stage,
                sensor=sensor,
                test_set=test_set,
                **kwargs
            ),
            'value': value
        }
        for percentile, value in zip(percentiles, stage_duration_percentiles)
        if value is not None
    ]


def generate_stage_sensors(clickhouse_client, days, owners, task_types, tags, percentiles, stage_name, test_set_name):
    query_template = get_template('stage_duration.sql')
    stage_duration_percentiles = get_duration_percentiles(clickhouse_client, query_template, days, owners, task_types, tags, percentiles)

    if not stage_duration_percentiles:
        logger.debug('No durations for %s %s', test_set_name, stage_name)
        return []

    ordered_stage = '{:02d}_{}'.format(Stages[stage_name].value, stage_name)
    return create_sensor(stage_duration_percentiles, percentiles, stage_name, test_set_name, ordered_stage, sensor='duration')


def get_duration_percentiles(clickhouse_client, template, days, owners, task_types, tags, percentiles):
    query = template.render(
        task_types=task_types,
        days_window=days,
        owners=owners,
        tags=tags,
        quantiles=percentiles,
    )

    logger.debug('Rendered query is:\n%s', query)

    meta, duration_percentiles = clickhouse_client.execute_with_format(query, output_format=Format.JSONCompact)

    fields = tuple(field["name"] for field in meta)
    if fields != ('stage_duration_quantiles', ):
        logger.error("Got unexpected data: fields=%s, data_sample=[%s...]", fields, duration_percentiles[0] if duration_percentiles else None)

    logger.info('Got stage duration percentiles %s', duration_percentiles)
    return duration_percentiles[0][0]


def generate_status_durations(clickhouse_client, days, owners, task_types, statuses, tags, percentiles, stage_name, test_set_name):
    query_template = get_template('summary_status_time.sql')
    duration_percentiles_by_status_and_task_type = get_status_duration_percentiles(clickhouse_client, query_template, days, owners, task_types, statuses, tags, percentiles)

    if not duration_percentiles_by_status_and_task_type:
        return []

    sensors = []
    ordered_stage = '{:02d}_{}'.format(Stages[stage_name].value, stage_name)
    for status_name, task_type, status_duration_percentiles in duration_percentiles_by_status_and_task_type:
        new_sensors = create_sensor(status_duration_percentiles, percentiles, stage_name, test_set_name, ordered_stage,
                                    sensor='summary_time', status=status_name, task_type=task_type)
        sensors.extend(new_sensors)
    return sensors


def get_status_duration_percentiles(clickhouse_client, template, days, owners, task_types, statuses, tags, percentiles):
    query = template.render(
        statuses=statuses,
        task_types=task_types,
        days_window=days,
        owners=owners,
        tags=tags,
        quantiles=percentiles,
    )

    logger.debug('Rendered query is:\n%s', query)

    meta, status_duration_percentiles = clickhouse_client.execute_with_format(query, output_format=Format.JSONCompact)
    fields = tuple(field["name"] for field in meta)
    if fields != ('status', 'task_type', 'status_duration_quantiles', ):
        logger.error("Got unexpected data: fields=%s, data_sample=[%s...]", fields, status_duration_percentiles[0] if status_duration_percentiles else None)

    logger.info('Status duration percentiles: %s', status_duration_percentiles)
    return status_duration_percentiles
