import logging
from datetime import datetime
from collections import defaultdict, namedtuple
from itertools import product

import pytz

from sandbox import sdk2
from sandbox.common.types import resource as ctr

from sandbox.projects.yabs.qa.solomon.mixin import SolomonTaskMixin, SolomonTaskMixinParameters
from sandbox.projects.yabs.qa.resource_types import (
    YABS_MYSQL_ARCHIVE_CONTENTS,
    YABS_CS_INPUT_SPEC,
    YABS_SERVER_CACHE_DAEMON_STUB_DATA,
    YABS_SERVER_REQUEST_LOG_GZ,
    YABS_SERVER_DOLBILKA_PLAN,
)


logger = logging.getLogger(__name__)


MetricConfig = namedtuple("MetricConfig", ("resource_attrs", "labels", "resource_types"))
ROLES = ("yabs", "bs", "bsrank")
RESOURCES_BY_TESTS = {
    "func": (
        YABS_SERVER_CACHE_DAEMON_STUB_DATA,
        YABS_SERVER_REQUEST_LOG_GZ,
    ),
    "load": (
        YABS_SERVER_CACHE_DAEMON_STUB_DATA,
        YABS_SERVER_DOLBILKA_PLAN,
    ),
}
CONFIG = [
    MetricConfig(
        resource_attrs={
            "testenv_switch_trigger": None,
        },
        labels={
            "sensor": "data_raw_resource_age",
        },
        resource_types=(
            YABS_MYSQL_ARCHIVE_CONTENTS,
            YABS_CS_INPUT_SPEC,
        ),
    ),
] + [
    MetricConfig(
        resource_attrs={
            "testenv_switch_trigger_{role}_{test}".format(role=role, test=test): None,
        },
        labels={
            "sensor": "ammo_raw_resource_age",
            "role": role,
            "test": test,
        },
        resource_types=RESOURCES_BY_TESTS[test]
    ) for role, test in product(ROLES, RESOURCES_BY_TESTS.keys())
]


class YabsCollectResourceGenerationInfo(SolomonTaskMixin, sdk2.Task):
    """Gathers raw resource info generation
    """
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        disk_space = 100

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        solomon_parameters = SolomonTaskMixinParameters()

        with sdk2.parameters.Group("Resources") as resources:
            limit_resources = sdk2.parameters.Integer("Check no more than N resources of each type", default=3000)

    @staticmethod
    def find_matching_resources_creation_time(attrs, resource_types, max_checked_resources):
        """Searches for set of resources with matching attribute values and returns creation time of the last one
        """
        resource_by_attr = defaultdict(dict)
        pagination_limit = 20
        pagination_offset = 0
        matching_resources_set_found = False
        matching_resources_creation_time = None
        resource_types = set(resource_types)

        while True:  # poorman's do-while loop
            for resource_type in resource_types:
                found_resources = resource_type.find(
                    state=ctr.State.READY,
                    attrs=attrs,
                ).order(-sdk2.Resource.id).limit(pagination_limit).offset(pagination_offset)

                for found_resource in found_resources:
                    attr_values_set = frozenset(
                        (attr_name, getattr(found_resource, attr_name))
                        for attr_name in attrs
                    )
                    resource_by_attr[attr_values_set][resource_type] = max(
                        found_resource.created,
                        resource_by_attr[attr_values_set].get(resource_type, found_resource.created)
                    )
                    matching_resources_set_found = len(resource_types) == len(resource_by_attr[attr_values_set])
                    if matching_resources_set_found:
                        matching_resources_creation_time = max(resource_by_attr[attr_values_set].values())
                        break

                if matching_resources_set_found:
                    break

            if pagination_offset > max_checked_resources:
                logger.warning("Cannot find any more resources %s", list(map(str, resource_types)))
                break

            pagination_offset += pagination_limit

        return matching_resources_creation_time

    def collect_metrics(self, config, limit_resources):
        now = datetime.now(tz=pytz.utc)
        metrics = []
        for cfg in config:
            creation_time = self.find_matching_resources_creation_time(cfg.resource_attrs, cfg.resource_types, limit_resources)
            if creation_time is not None:
                metrics.append({
                    "labels": cfg.labels,
                    "value": (now - creation_time).total_seconds(),
                })
        return metrics

    def on_execute(self):
        metrics = self.collect_metrics(CONFIG, self.Parameters.limit_resources)
        self.solomon_push_client.add(metrics)
