import time
import json
import itertools

from sandbox import sdk2

from sandbox.common.rest import Client
import sandbox.common.types.resource as ctr
from sandbox.common.types.notification import Transport

from sandbox.projects.yabs.qa.solomon.mixin import SolomonTaskMixin, SolomonTaskMixinParameters

from .label_maps import ClusterMap
from sandbox.projects.yabs.qa.resource_types import BaseBackupSdk2Resource


class YabsHostsTaskTypesList(BaseBackupSdk2Resource):
    '''List of YABS tasks observed to execute on YABS-dedicated hosts'''
    pass


def int_mean(a, b):
    return int((a + b) / 2)


def chunk_split(iterable, chunk_size):
    for index in xrange(0, len(iterable), chunk_size):
        yield iterable[index:index + chunk_size]


class YabsCollectSandboxPoolLoad(SolomonTaskMixin, sdk2.Task):
    '''SDK2 task that collects YABS-dedicated SANDBOX hosts info and pushes it to Solomon'''

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024
        disk_space = 100

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        solomon_parameters = SolomonTaskMixinParameters()
        max_hosts_in_single_rest_request = sdk2.parameters.Integer('Maximum number of YABS hosts questioned in a single request to Sandbox REST API', default=1000)

    def on_create(self):
        super(YabsCollectSandboxPoolLoad, self).on_create()
        if self.Context.copy_of:
            self.Parameters.notifications = [
                notification for notification in self.Parameters.notifications
                if notification.transport != Transport.JUGGLER
            ]

    def get_historic_task_types(self):
        task_types_resource = YabsHostsTaskTypesList.find(state=ctr.State.READY).order(-sdk2.Resource.id).first()
        if task_types_resource:
            resource_path = str(sdk2.ResourceData(task_types_resource).path)
            with open(resource_path) as f:
                task_types = set(json.load(f))
        else:
            task_types = set()
        return task_types

    def get_current_task_types(self, full_hosts_data):
        task_types = set(['__IDLE__'])
        for item in full_hosts_data:
            for task_data in item.get('tasks', []):
                task_types.add(task_data['type'])
        return task_types

    def extend_hosts_data(self, full_data, data_chunk, timestamp):
        full_data.extend(data_chunk)
        for item in full_data[len(full_data) - len(data_chunk):]:
            item['timestamp'] = timestamp

    def get_hosts_data(self):
        full_hosts_data = []
        timestamp = int(time.time())
        first_data_chunk = self.rest_client.client.read(tags='YABS', limit=self.Parameters.max_hosts_in_single_rest_request, offset=0)
        self.extend_hosts_data(full_hosts_data, first_data_chunk['items'], timestamp)
        total = first_data_chunk['total']
        for offset in xrange(self.Parameters.max_sensors_per_push, total, self.Parameters.max_sensors_per_push):
            data_chunk = self.rest_client.client.read(tags='YABS', limit=self.Parameters.max_hosts_in_single_rest_request, offset=offset)
            self.extend_hosts_data(full_hosts_data, data_chunk['items'], timestamp)
        return full_hosts_data

    def prepare_sensors_data(self, full_hosts_data):
        task_types = self.get_historic_task_types() | self.get_current_task_types(full_hosts_data)
        with open('task_types.json', 'w') as f:
            json.dump(list(task_types), f)
        YabsHostsTaskTypesList(self, '', 'task_types.json', ttl=2)
        sensors_data = {}
        for host_data in full_hosts_data:
            cluster_list = ClusterMap.get_clusters(host_data)
            for task_type, cluster in itertools.product(task_types, cluster_list):
                if task_type == '__IDLE__':
                    sensor_value = 0 if 'tasks' in host_data else 1
                else:
                    sensor_value = sum(1 for task in host_data.get('tasks', []) if task['type'] == task_type)
                sensors_data.setdefault(cluster, []).append({
                    'labels': {
                        'sensor': task_type,
                        'host': host_data['fqdn']
                    },
                    'ts': host_data['timestamp'],
                    'value': sensor_value
                })
            for cluster in cluster_list:
                sensors_data[cluster].append({
                    'labels': {
                        'sensor': '__TOTAL_HOSTS__',
                        'host': host_data['fqdn'],
                    },
                    'ts': host_data['timestamp'],
                    'value': 1
                })

        return sensors_data

    def on_execute(self):
        self.rest_client = Client()
        hosts_data = self.get_hosts_data()
        sensors_data = self.prepare_sensors_data(hosts_data)
        for cluster, sensors_list in sensors_data.iteritems():
            self.solomon_push_client.add(sensors_list, service='task_load', cluster=cluster)
