import json
import logging

from sandbox import sdk2
from sandbox.common.types.notification import Transport
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.common.testenv_client import TEClient
from sandbox.projects.yabs.qa.solomon.mixin import SolomonTaskMixin, SolomonTaskMixinParameters
from sandbox.projects.yabs.qa.tasks.YabsCollectTestenvData.resources import get_resource_sensors, get_bad_resource_checks, get_broken_tasks
from sandbox.projects.yabs.qa.tasks.YabsCollectTestenvData.tests import get_tests_sensors
from sandbox.projects.yabs.qa.tasks.YabsCollectTestenvData.startrek import create_issue_if_needed


logger = logging.getLogger(__name__)


class YabsCollectTestenvData(SolomonTaskMixin, sdk2.Task):
    '''SDK2 task that collects yabs-2.0 resource delays data and pushes it to Solomon'''

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024
        disk_space = 100
        environments = [
            PipEnvironment('startrek_client', use_wheel=True),  # PYPI-101
            PipEnvironment('jinja2'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        solomon_parameters = SolomonTaskMixinParameters()

        with sdk2.parameters.Group("Specific sensors parameters") as specific_sensor_parameters:
            test_delay_service = sdk2.parameters.String("Service to push job metrics", required=True)
            resource_delay_service = sdk2.parameters.String("Service to push resource metrics", required=True)

        with sdk2.parameters.Group("TestEnv parameters") as testenv_parameters:
            database = sdk2.parameters.String("TestEnv project aka database", required=True)

        with sdk2.parameters.Group("Collector parameters") as collector_parameters:
            testenv_request_timeout = sdk2.parameters.Integer("Timeout in seconds to request testenv", default=300)

    def on_create(self):
        super(YabsCollectTestenvData, self).on_create()
        if self.Context.copy_of:
            self.Parameters.notifications = [
                notification for notification in self.Parameters.notifications
                if notification.transport != Transport.JUGGLER
            ]

    def on_execute(self):
        from startrek_client import Startrek

        secret = sdk2.yav.Secret(self.Parameters.solomon_token_yav_secret_id, self.Parameters.solomon_token_yav_secret_version)
        startrek_client = Startrek(useragent=self.__class__.__name__, token=secret.data()['startrek_token'])

        resource_data = TEClient.get_resources(self.Parameters.database)
        checks = get_bad_resource_checks(resource_data)
        logger.debug('Got checks: %s', json.dumps(checks, indent=2))
        for check, resources in checks.items():
            broken_tasks = get_broken_tasks(self.server, check)
            create_issue_if_needed(startrek_client, check, broken_tasks, resources, queue='SHMDUTY')

        resource_sensors_data = get_resource_sensors(resource_data)
        logger.debug('Got resource sensors data: %s', resource_sensors_data)
        self.solomon_push_client.add(resource_sensors_data, service=self.Parameters.resource_delay_service)

        test_sensors_data = get_tests_sensors(self.Parameters.database)
        logger.debug('Got test sensors data: %s', test_sensors_data)
        self.solomon_push_client.add(test_sensors_data, service=self.Parameters.test_delay_service)
