import json
import logging
import re
import time

from collections import namedtuple

from sandbox.common.types.task import Status
from sandbox.projects.common.testenv_client import TEClient
from sandbox.projects.yabs.qa.tasks.YabsCollectTestenvData.tests import get_tests


logger = logging.getLogger(__name__)
DATABASE_NAME_PATTERN = re.compile(r'database=(?P<database_name>[a-z0-9]*)')
ResourceInfo = namedtuple('ResourceInfo', ('name', 'id'))


def get_resource_sensors(resource_data):
    current_timestamp = int(time.time())
    sensors_data = []
    for resource in resource_data:
        if resource['status'] != 'OK':
            continue
        sensors_data.append({
            'labels': {
                'sensor': 'resource_delay_sec',
                'resource_name': resource['name'],
                'auto_updated': repr(resource['auto_update'] & resource['auto_update_implemented'])
            },
            'ts': current_timestamp,
            'value': current_timestamp - resource['resource_timestamp']
        })
    return sensors_data


def get_check_database(html_string):
    m = DATABASE_NAME_PATTERN.search(html_string)
    if not m:
        return html_string
    return m.group('database_name')


def get_bad_resource_checks(resources):
    bad_resources = [resource for resource in resources if resource['status'] == 'BAD']
    logger.debug(json.dumps(bad_resources, indent=2))
    bad_resources_checks = {}
    for bad_resource in bad_resources:
        check = get_check_database(bad_resource['parallel_check'])
        bad_resources_checks.setdefault(check, []).append(ResourceInfo(bad_resource['name'], bad_resource['resource_id']))

    return bad_resources_checks


def get_broken_tasks(sandbox_client, database):
    try:
        tests = get_tests(database)
    except Exception:
        return []

    sandbox_tasks = TEClient.get_last_sandbox_task_ids(database, tests.keys(), success=False)
    broken_tasks = []
    for sandbox_task in sandbox_tasks:
        if not sandbox_task['task_id']:
            continue
        task_info = sandbox_client.task[sandbox_task['task_id']].read()
        if task_info['status'] in {Status.FAILURE} | Status.Group.BREAK - {Status.STOPPED}:
            logger.info('Task #%d is in %s status', task_info['id'], task_info['status'])
            broken_tasks.append(sandbox_task['task_id'])

    logger.debug('Got tasks: %s', json.dumps(broken_tasks, indent=2))
    return broken_tasks
