import time
import json
import logging
from datetime import tzinfo, timedelta, datetime

from sandbox import sdk2
from sandbox.common.types.resource import State

from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_B2B_BINARY_BASE
from sandbox.projects.yabs.qa.tasks.YabsServerCreateOneShotSpec.spec import YABS_SERVER_HOUSE_SPEC
from sandbox.projects.yabs.qa.tasks.YabsServerCreateABExperimentSpec.spec import YabsServerABExperimentSpec


logger = logging.getLogger(__name__)
ZERO = timedelta(0)


class UTC(tzinfo):
    def utcoffset(self, dt):
        return ZERO

    def tzname(self, dt):
        return "UTC"

    def dst(self, dt):
        return ZERO


utc = UTC()


def get_resource_age(resource_id):
    created = sdk2.Resource[resource_id].created
    now = datetime.now(utc)
    logger.debug('Resource #%s created at %s, now %s, age is %s', resource_id, now, now - created)
    return int((now - created).total_seconds())


def get_spec_resources():
    (resource_func, resource_load) = (sdk2.Resource.find(
        YABS_SERVER_HOUSE_SPEC,
        attrs={
            attr: 'True',
            'released_spec': 'True'
        },
        state=State.READY,
    ).order(-sdk2.Resource.id).first() for attr in ('good', 'good_with_load'))

    logger.info("Found func_spec #%d, load_spec #%d", resource_func.id, resource_load.id)

    with open(str(sdk2.ResourceData(resource_func).path)) as spec_func_file, \
            open(str(sdk2.ResourceData(resource_load).path)) as spec_load_file:
        spec_func = json.load(spec_func_file)
        spec_load = json.load(spec_load_file)

    return {
        'mysql_func': spec_func['mysql_archive_resource_id'],
        'yt_func': spec_func['cs_input_spec_resource_id'],
        'cs_settings_func': spec_func['cs_settings_archive_resource_id'],
        'ammo_bs_func': spec_func['ft_request_log_resource_id_map']['bs'],
        'ammo_bsrank_func': spec_func['ft_request_log_resource_id_map']['bsrank'],
        'ammo_yabs_func': spec_func['ft_request_log_resource_id_map']['yabs'],
        'stat_bs_release_ft': spec_func['stat_bs_release_tar_resource_id'],
        'meta_bs_release_ft': spec_func['meta_bs_release_tar_resource_id'],

        'mysql_load': spec_load['mysql_archive_resource_id'],
        'yt_load': spec_load['cs_input_spec_resource_id'],
        'cs_settings_load': spec_load['cs_settings_archive_resource_id'],
        'ammo_bs_load': spec_load['stat_load_request_log_resource_id_map']['bs'],
        'ammo_bsrank_load': spec_load['stat_load_request_log_resource_id_map']['bsrank'],
        'ammo_yabs_load': spec_load['stat_load_request_log_resource_id_map']['yabs'],
    }, {
        'stat': {
            'ft': spec_func['stat_bs_release_tar_resource_id'],
            'meta_load': spec_load['stat_bs_release_tar_resource_id'],
            'stat_load': spec_load['stat_bs_release_tar_resource_id'],
        },
        'meta': {
            'ft': spec_func['meta_bs_release_tar_resource_id'],
            'meta_load': spec_load['meta_bs_release_tar_resource_id'],
            'stat_load': spec_load['meta_bs_release_tar_resource_id'],
        }
    }


def get_resource_age_sensors(resources_from_spec):
    sensors = []

    for tier, resource_id in resources_from_spec.iteritems():
        if 'bs_release' in tier:
            continue
        try:
            sensors.append(
                {
                    'labels': {
                        'sensor': 'resource_age',
                        'tier': tier,
                    },
                    'value': get_resource_age(resource_id)
                }
            )
        except Exception as exc:
            logger.error("Cannot get '%s' resource #%d age due to %s", tier, resource_id, exc, exc_info=True)

    return sensors


def get_version_sensors(sandbox_rest_client, spec_server_resources, nanny_token):
    from sandbox.projects.yabs.release.version.sandbox_helpers import SandboxHelper
    from sandbox.projects.yabs.release.version.server import ServerVersionHelper

    sandbox_helper = SandboxHelper(sandbox_client=sandbox_rest_client)
    server_version_helper = ServerVersionHelper(nanny_token=nanny_token)

    prod_versions = {}
    for role in ('meta', 'stat'):
        versions = server_version_helper.get_deployed_versions_info('bsfront_production', 'BS_RELEASE_TAR', labels={'server_mode': role})
        logger.debug("Got %s service versions: %s", role, versions)
        prod_server_version, _ = max(versions.items(), key=lambda v: len(v[1]["nanny_services"]))
        prod_versions[role] = prod_server_version

    spec_versions = {
        role: {
            tier: sandbox_helper.get_full_version_from_resource(spec_server_resources[role][tier])
            for tier in ('ft', 'meta_load', 'stat_load')
        }
        for role in ('meta', 'stat')
    }

    sensors = [
        {
            'labels': {
                'tier': tier,
                'role': role,
                'sensor': 'version',
                'version': str(version),
            },
            'value': 1
        }
        for role, role_versions in spec_versions.items()
        for tier, version in role_versions.items()
    ]
    sensors.extend([
        {
            'labels': {
                'tier': tier,
                'role': role,
                'sensor': 'version_diff',
                'status': 'version_differ',
            },
            'value': 1 if version != prod_versions[role] else 0
        }
        for role, role_versions in spec_versions.items()
        for tier, version in role_versions.items()
    ])
    sensors.extend([
        {
            'labels': {
                'tier': tier,
                'role': role,
                'sensor': 'version_diff',
                'status': 'version_equal',
            },
            'value': 1 if version == prod_versions[role] else 0
        }
        for role, role_versions in spec_versions.items()
        for tier, version in role_versions.items()
    ])

    return sensors


def get_ab_experiment_sensors(sandbox_rest_client, nanny_token):
    sensors = []

    spec_attrs = {
        'ft': {'ft': True},
        'stat_load': {'stat_load': True},
        'meta_load': {'meta_load': True},
    }
    now = time.time()

    spec_server_resources = {'meta': {}, 'stat': {}}
    for tier, attrs in spec_attrs.items():
        _find_attrs = dict({'released_spec': True}, **attrs)
        spec_resource = YabsServerABExperimentSpec.find(attrs=_find_attrs, order='-id').limit(1).first()
        if not spec_resource:
            continue
        with open(str(sdk2.ResourceData(spec_resource).path), 'r') as spec_file:
            spec_data = json.load(spec_file)

        for role in ('meta', 'stat'):
            spec_server_resources[role][tier] = spec_data['{}_bs_release_tar_resource_id'.format(role)]

            base_resource_ids = spec_data['{}_binary_base_resource_id_by_tag'.format(role)].values()
            base_resources = YABS_SERVER_B2B_BINARY_BASE.find(id=base_resource_ids).limit(len(base_resource_ids))
            for base_resource in base_resources:
                if hasattr(base_resource, 'mtime'):
                    resource_age = now - float(base_resource.mtime)
                else:
                    resource_age = get_resource_age(base_resource.id)

                sensors.append({
                    'labels': {
                        'base_tag': base_resource.tag,
                        'tier': tier,
                        'sensor': 'baseage',
                        'role': role,
                    },
                    'value': resource_age,
                })

    sensors.extend(get_version_sensors(sandbox_rest_client, spec_server_resources, nanny_token))

    return sensors
