import textwrap

from sandbox import sdk2

from sandbox.common.types import resource as ctr
from sandbox.common.types.notification import Transport
from sandbox.projects.resource_types import CACHE_DAEMON
from sandbox.projects.common.dolbilka.resources import DPLANNER_EXECUTABLE

from sandbox.projects.YabsServerGetSQLArchive import Option as ArchiveMySQLOptions

from sandbox.projects.yabs.qa.resource_types import (
    BS_RELEASE_YT,
    YABS_MYSQL_ARCHIVE_CONTENTS,
    YABS_MYSQL_RESTORE_DESCRIPTION,
    YABS_SERVER_TESTENV_DB_FLAGS,
)
from sandbox.projects.yabs.qa.tasks.BuildAmmoAndStubsFromYT import GenAmmoAndStubFromYtBinary
from sandbox.projects.yabs.qa.tasks.YabsServerArchiveCSSettings import DEFAULT_SETTINGS_UPDATE, DEFAULT_SETTINGS_BLACKLIST
from sandbox.projects.yabs.qa.tasks.YabsServerGetYTRequestData2 import (
    YabsServerGenAmmoTables,
)
from sandbox.projects.yabs.qa.tasks.YabsServerGetYTRequestData2.spec import DEFAULT_SPEC
from sandbox.projects.yabs.qa.tasks.YabsServerCreateYabsHitModelsHamster import YabsServerCreateYabsHitModelsHamster
from sandbox.projects.yabs.qa.tasks.YabsCreateTestenvResources.stages import (
    create_hamsters,
    freeze_saas_state,
    launch_ammo_generation,
    launch_cs_settings_archive_generation,
    launch_mysql_archive_generation,
    launch_one_shot_spec_generation,
    launch_yt_archive_generation,
    publish_resource_spec,
)
from sandbox.projects.yabs.qa.spec.parameters import ShootParameters, MiscResourceParameters, ServerResourceParameters
from sandbox.projects.yabs.qa.spec.stages import get_engine_resources


_STAGES = [
    create_hamsters,
    get_engine_resources,
    launch_mysql_archive_generation,
    launch_yt_archive_generation,
    launch_cs_settings_archive_generation,
    launch_ammo_generation,
    publish_resource_spec,
    launch_one_shot_spec_generation,
]


class YtArchiveParameters(sdk2.Parameters):
    with sdk2.parameters.Group('YT archive parameters') as yt_archive_parameters:
        yt_archive_run_save_input = sdk2.parameters.Bool('Run save input', default=True)
        yt_archive_yt_proxy = sdk2.parameters.String('YT proxy to create input archive', default='hahn')
        yt_archive_archives_root = sdk2.parameters.String('Path to create new archive in', default='//home/yabs-cs-sandbox/input-archive')


class CsSettingsParameters(sdk2.Parameters):
    with sdk2.parameters.Group('CS settings archive parameters') as cs_settings_archive_parameters:
        cs_settings_archive_yt_proxy = sdk2.parameters.String(
            'YT proxy that stores content system production settings',
            required=True,
            default='locke',
        )
        cs_settings_archive_settings_path = sdk2.parameters.String(
            'Path to the yt node with settings',
            required=True,
            default='//home/samogon/yabscs_production/yabs-cs/settings/v5_head',
        )
        cs_settings_archive_yt_token_vault_name = sdk2.parameters.String('Vault name for YT token', required=True, default='yabs-cs-sb-yt-token')
        cs_settings_archive_set_now_unixtime = sdk2.parameters.Bool('Set cs "now_unixtime" setting to current timestamp')
        cs_settings_archive_settings_blacklist = sdk2.parameters.List('List of settings which will be ignored on saving archive', default=DEFAULT_SETTINGS_BLACKLIST)
        cs_settings_archive_settings_update = sdk2.parameters.JSON('Update production settings with custom values', default=DEFAULT_SETTINGS_UPDATE)


class YabsCreateTestenvResources(sdk2.Task):
    """ Creates resources bundle for yabs-2.0 testenv base """

    class Parameters(sdk2.Parameters):
        max_restarts = 2
        # Can't enable `push_tasks_resource` because CreateOneshotSpec task is SDK1
        push_tasks_resource = False

        tokens = sdk2.parameters.YavSecret("YAV secret identifier", default="sec-01d6apzcex5fpzs5fcw1pxsfd5")

        with sdk2.parameters.Group('Common parameters') as common_parameters:
            testenv_switch_trigger = sdk2.parameters.Bool('Resources will be marked for use in testenv', default=False, do_not_copy=True)

            server_resources = ServerResourceParameters()

            generate_mysql_archive = sdk2.parameters.Bool('Generate MySQL archive', default=True)
            generate_yt_archive = sdk2.parameters.Bool('Generate YT archive', default=True)
            generate_cs_settings_archive = sdk2.parameters.Bool('Generate CS settings archive', default=True)
            generate_ammo = sdk2.parameters.Bool('Generate ammo', default=True)

            launch_one_shot_spec_generation = sdk2.parameters.Bool('Launch one-shot spec generation', default=True)

        with generate_mysql_archive.value[True]:
            with sdk2.parameters.Group('MySQL archive parameters') as mysql_archive_parameters:
                mysql_archive_bin_db_list = sdk2.parameters.String('Space separated list of engine bin bases to restore sql tables for', required=True)
                mysql_archive_filter_table = sdk2.parameters.String('Regexp to filter some tables by name', default='.*')
                mysql_archive_backup_date = sdk2.parameters.String('Backup date in YYYYMMDD format. Empty = current - 3h')

                mysql_archive_bs_release_yt = sdk2.parameters.Resource('Resource with dbtool', resource_type=BS_RELEASE_YT)
                mysql_archive_prev_archive_contents_resource = sdk2.parameters.Resource(
                    'Resource with previous archive contents.',
                    resource_type=YABS_MYSQL_ARCHIVE_CONTENTS,
                )
                mysql_archive_additional_restore_description_resource = sdk2.parameters.Resource(
                    'Resource with restore description for additional tables.',
                    resource_type=YABS_MYSQL_RESTORE_DESCRIPTION,
                )
                mysql_archive_additional_restore_description_resource_md5 = sdk2.parameters.String('MD5 of resource with restore description for additional tables.')
                mysql_archive_wait_for_backups = sdk2.parameters.Bool('Wait for table backups for specified date to appear', default=True)
                with mysql_archive_wait_for_backups.value[True]:
                    mysql_archive_wait_for_backups_period = sdk2.parameters.Integer('Sleep between retries in seconds', default=20 * 60)
                    mysql_archive_wait_for_backups_timeout = sdk2.parameters.Integer('Fail task if backups aren\'t present after following period in seconds', default=4 * 60 * 60)

                with sdk2.parameters.CheckGroup('Additional options') as additional_mysql_options:
                    additional_mysql_options.values.no_tables = additional_mysql_options.Value(ArchiveMySQLOptions.NO_TABLES, checked=False)
                    additional_mysql_options.values.no_lm_dumps = additional_mysql_options.Value(ArchiveMySQLOptions.NO_LM_DUMPS, checked=False)

        with generate_yt_archive.value[True]:
            yt_archive_paramters = YtArchiveParameters()

        with generate_cs_settings_archive.value[True]:
            cs_settings_paramters = CsSettingsParameters()

        with generate_ammo.value[True]:
            with sdk2.parameters.Group('Ammo generation parameters') as ammo_parameters:
                ammo_gen_ammo_tables_binary = sdk2.parameters.LastReleasedResource(
                    "Ammo tables generator binary",
                    resource_type=YabsServerGenAmmoTables,
                    state=(ctr.State.READY, ctr.State.NOT_READY),
                )

                ammo_gen_ammo_and_stub_from_yt_binary = sdk2.parameters.LastReleasedResource(
                    "Stub and ammo generator binary",
                    resource_type=GenAmmoAndStubFromYtBinary,
                    state=(ctr.State.READY, ctr.State.NOT_READY),
                )

                ammo_cache_daemon_resource = sdk2.parameters.LastReleasedResource(
                    'Cache daemon binary',
                    resource_type=CACHE_DAEMON,
                    state=(ctr.State.READY, ctr.State.NOT_READY),
                )

                ammo_d_planner_resource = sdk2.parameters.LastReleasedResource(
                    'Dolbilka planner binary',
                    resource_type=DPLANNER_EXECUTABLE,
                    state=(ctr.State.READY, ctr.State.NOT_READY),
                )

                ammo_spec = sdk2.parameters.JSON("Spec", default=DEFAULT_SPEC)
                ammo_format_yql = textwrap.dedent('''
                    SELECT ... as requestID
                    ...
                    FROM {table name without time(//logs/bs-proto-accessstat-log/1h)}
                    ...
                ''')
                ammo_yql_query_description = "Custom YQL request in format: \n" + ammo_format_yql
                ammo_yql_query = sdk2.parameters.String(ammo_yql_query_description, multiline=True)
                ammo_yql_query_role = sdk2.parameters.String("Role of custom requests (yabs, bs, bsrank)")
                ammo_yt_token_vault_name = sdk2.parameters.String("Vault name to get YT token from", default="yabs-cs-sb-yt-token")
                ammo_yql_token_vault_name = sdk2.parameters.String("Vault name to get YQL token from", default="yabs-cs-sb-yql-token")
                ammo_pool = sdk2.parameters.String("YT pool to launch operations in")
                ammo_enable_page_id_coverage = sdk2.parameters.Bool("Enable PageIDs coverage", default=False)
                ammo_days_interval = sdk2.parameters.Integer("Interval in days to get requests", default=1, required=True)
                ammo_logs_interval = sdk2.parameters.String("Logs interval: 1d or 1h", default="1h", required=True)
                ammo_resource_ttl = sdk2.parameters.Integer("Output resources ttl", default=30)

        with sdk2.parameters.Group('Hamster parameters') as hamster_parameters:
            with sdk2.parameters.CheckGroup("Create hamsters for service tags") as hamster_service_tags:
                hamster_service_tags.values.goalnet = hamster_service_tags.Value("goalnet", checked=True)
                hamster_service_tags.values.rsya_hit_models_heavy_01 = hamster_service_tags.Value("rsya_hit_models_heavy_01", checked=True)

            with sdk2.parameters.Group('rsya_hit_models_heavy_01 hamster parameters') as rsya_hit_models_hamster_parameters_group:
                yabs_hit_models_hamster_replicas = YabsServerCreateYabsHitModelsHamster.Parameters.replicas(default=8)

        with sdk2.parameters.Group('SaaS parameters') as saas_parameters:
            freeze_kvrs_saas_state = sdk2.parameters.Bool('Freeze KVRS SaaS state', default=False)

        with launch_one_shot_spec_generation.value[True]:
            with sdk2.parameters.Group('OneShot spec generation parameters') as one_shot_spec_generation_parameters:
                one_shot_spec_release_spec = sdk2.parameters.Bool('Release spec for public use', default=False, do_not_copy=True)
                one_shot_spec_ft_shards = sdk2.parameters.List('FT shards', default=['A'])
                one_shot_spec_ft_shoot_settings_resource = sdk2.parameters.Resource(
                    'FT update parameters resource, will find last ready if empty',
                    resource_type=YABS_SERVER_TESTENV_DB_FLAGS
                )

                one_shot_spec_create_load_spec = sdk2.parameters.Bool('Create load spec', default=True)
                with one_shot_spec_create_load_spec.value[True]:
                    one_shot_spec_add_meta_load = sdk2.parameters.Bool('Add meta in load spec', default=False)
                    with one_shot_spec_add_meta_load.value[True]:
                        one_shot_spec_meta_load_shards = sdk2.parameters.List('META LOAD shards', default=['A', 'B'])
                    one_shot_spec_load_shards = sdk2.parameters.List('LOAD shards', default=['A', 'B', 'C'])

                one_shot_spec_generate_separated_bases = sdk2.parameters.Bool('Generate additional yabs, bs bases', default=True)
                one_shot_spec_generate_not_separated_bases = sdk2.parameters.Bool('Generate bases that have additional (yabs, bs)', default=False)

                with sdk2.parameters.Group('External services') as external_services:
                    oneshot_ext_service_endpoint_resources = ShootParameters.ext_service_endpoint_resources()
                    oneshot_hamster_ext_service_tags = ShootParameters.hamster_ext_service_tags()
                    linear_models_binary_resource = MiscResourceParameters.linear_models_binary_resource()

                one_shot_spec_use_separated_meta_and_stat = sdk2.parameters.Bool('Use separated meta and stat in shoot tasks', default=False)

        with sdk2.parameters.Group("Developer options") as developer_group:
            debug_mode = sdk2.parameters.Bool('Debug mode', default=False)
            with debug_mode.value[True]:
                reusable_stages = sdk2.parameters.CheckGroup(
                    'Reusable stages',
                    choices=[
                        (_stage.__name__, _stage.__name__)
                        for _stage in _STAGES
                    ],
                    default=None
                )

    class Requirements(sdk2.Requirements):
        ram = 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_create(self):
        super(YabsCreateTestenvResources, self).on_create()
        if not self.scheduler:
            self.Parameters.notifications = [
                notification for notification in self.Parameters.notifications
                if notification.transport != Transport.JUGGLER
            ]

    def on_execute(self):
        if self.Parameters.debug_mode:
            self.Context.__reusable_stages = self.Parameters.reusable_stages

        get_engine_resources(self)
        launch_mysql_archive_generation(self)
        launch_cs_settings_archive_generation(self)
        launch_yt_archive_generation(self)
        launch_ammo_generation(self)
        create_hamster_task_ids = create_hamsters(self)
        freeze_saas_state(self)

        publish_resource_spec(self, create_hamster_task_ids=create_hamster_task_ids)

        if self.Parameters.launch_one_shot_spec_generation:
            launch_one_shot_spec_generation(self)
