import itertools
import logging

from sandbox import sdk2

from sandbox.common.types.task import Status
from sandbox.common.types.notification import Transport, JugglerStatus

from sandbox.projects.common.yabs.server.util.general import check_tasks, find_last_ready_resource

from sandbox.projects.YabsServerGetSQLArchive import Option as ArchiveMySQLOptions
from sandbox.projects.yabs.qa.resource_types import (
    YABS_CS_INPUT_SPEC,
    YABS_CS_SETTINGS_ARCHIVE,
    YABS_MYSQL_ARCHIVE_CONTENTS,
    YABS_SERVER_TESTENV_SHARD_MAP,
    YabsServerExtServiceEndpoint,
)
from sandbox.projects.yabs.qa.pipeline.stage import stage
from sandbox.projects.yabs.qa.tasks.YabsServerArchiveCSSettings import YabsServerArchiveCSSettings
from sandbox.projects.yabs.qa.tasks.YabsServerCreateContentSystemInputArchive import (
    YabsServerCreateContentSystemInputArchive,
)
from sandbox.projects.yabs.qa.tasks.YabsServerCreateGoalnetHamster import YabsServerCreateGoalnetHamster
from sandbox.projects.yabs.qa.tasks.YabsServerCreateOneShotSpec import YabsServerCreateOneShotSpec
from sandbox.projects.yabs.qa.tasks.YabsServerCreateYabsHitModelsHamster import YabsServerCreateYabsHitModelsHamster
from sandbox.projects.yabs.qa.tasks.YabsServerFreezeCurrentKvrsSaasSnapshot import (
    YabsServerFreezeCurrentKvrsSaasSnapshot,
)
from sandbox.projects.yabs.qa.tasks.YabsServerGetYTRequestData2 import YabsServerGetYtRequestData2
from sandbox.projects.yabs.qa.utils.general import get_task_html_hyperlink
from sandbox.projects.yabs.qa.utils.resource import get_last_released_resource


logger = logging.getLogger(__name__)


def juggler_notifications(self, process_id):
    notifications = []
    if self.scheduler:
        notifications = [
            sdk2.Notification(
                task_statuses,
                ["host=yabs_server_testing_schedulers&service=yabs_server_{process_id}_status".format(process_id=process_id)],
                Transport.JUGGLER,
                check_status=juggler_status
            )
            for task_statuses, juggler_status in (
                ([Status.FAILURE, Status.Group.BREAK], JugglerStatus.CRIT),
                ([Status.SUCCESS], JugglerStatus.OK),
            )
        ]

    return notifications


@stage(provides='mysql_archive_task_id')
def launch_mysql_archive_generation(self):
    options_list = self.Parameters.additional_mysql_options
    if self.Parameters.testenv_switch_trigger:
        options_list.append(ArchiveMySQLOptions.SWITCH_TESTENV)
    mysql_archive_generation_task = sdk2.Task['YABS_SERVER_GET_SQL_ARCHIVE'](
        self,
        owner=self.owner,
        tags=self.Parameters.tags,
        description=self.Parameters.description,
        bin_db_list=self.Parameters.mysql_archive_bin_db_list,
        filter_table=self.Parameters.mysql_archive_filter_table,
        backup_date=self.Parameters.mysql_archive_backup_date,
        bs_release_yt=self.Parameters.mysql_archive_bs_release_yt,
        prev_archive_contents_resource=self.Parameters.mysql_archive_prev_archive_contents_resource,
        additional_restore_description_resource=self.Parameters.mysql_archive_additional_restore_description_resource,
        additional_restore_description_resource_md5=self.Parameters.mysql_archive_additional_restore_description_resource_md5,
        options_list=' '.join(options_list),
        testenv_switch_trigger_value=str(self.id),
        wait_for_backups=self.Parameters.mysql_archive_wait_for_backups,
        wait_for_backups_period=self.Parameters.mysql_archive_wait_for_backups_period,
        wait_for_backups_timeout=self.Parameters.mysql_archive_wait_for_backups_timeout,
    )
    mysql_archive_generation_task.enqueue()
    self.set_info('Launched mysql_archive generation task {}'.format(get_task_html_hyperlink(mysql_archive_generation_task.id)), do_escape=False)
    return mysql_archive_generation_task.id


@stage(provides='yt_archive_task_id')
def launch_yt_archive_generation(
    self,
    cs_settings_archive_task_id,
    stat_bs_release_yt_resource_id,
):
    check_tasks(self, cs_settings_archive_task_id)

    cs_settings_archive_resource = YABS_CS_SETTINGS_ARCHIVE.find(task_id=cs_settings_archive_task_id, linit=1).first()

    params = {}
    if self.Parameters.testenv_switch_trigger:
        params["testenv_switch_trigger"] = str(self.id)
    yt_archive_generation_task = YabsServerCreateContentSystemInputArchive(
        self,
        owner=self.owner,
        tags=self.Parameters.tags,
        description=self.Parameters.description,
        yt_proxy=self.Parameters.yt_archive_yt_proxy,
        archives_root=self.Parameters.yt_archive_archives_root,
        run_save_input=self.Parameters.yt_archive_run_save_input,
        cs_settings_archive_resource=cs_settings_archive_resource,
        bs_release_yt_resource=stat_bs_release_yt_resource_id,
        __requirements__={
            "tasks_resource": self.Requirements.tasks_resource,
        },
        **params
    )
    yt_archive_generation_task.enqueue()
    self.set_info('Launched yt_archive generation task {}'.format(get_task_html_hyperlink(yt_archive_generation_task.id)), do_escape=False)
    return yt_archive_generation_task.id


@stage(provides='cs_settings_archive_task_id')
def launch_cs_settings_archive_generation(self):
    cs_settings_archive_generation_task = YabsServerArchiveCSSettings(
        self,
        owner=self.owner,
        tags=self.Parameters.tags,
        description=self.Parameters.description,
        archive_yt_proxy=self.Parameters.cs_settings_archive_yt_proxy,
        settings_path=self.Parameters.cs_settings_archive_settings_path,
        yt_token_vault_name=self.Parameters.cs_settings_archive_yt_token_vault_name,
        set_now_unixtime=self.Parameters.cs_settings_archive_set_now_unixtime,
        settings_blacklist=self.Parameters.cs_settings_archive_settings_blacklist,
        settings_update=self.Parameters.cs_settings_archive_settings_update,
        switch_testenv_bases=self.Parameters.testenv_switch_trigger,
        testenv_switch_trigger_value=str(self.id),
    )
    cs_settings_archive_generation_task.enqueue()
    self.set_info('Launched cs_settings_archive generation task {}'.format(get_task_html_hyperlink(cs_settings_archive_generation_task.id)), do_escape=False)
    return cs_settings_archive_generation_task.id


@stage(provides='ammo_task_id')
def launch_ammo_generation(self):
    ammo_generation_task = YabsServerGetYtRequestData2(
        self,
        owner=self.owner,
        tags=self.Parameters.tags,
        description=self.Parameters.description,
        gen_ammo_tables_binary=self.Parameters.ammo_gen_ammo_tables_binary,
        gen_ammo_and_stub_from_yt_binary=self.Parameters.ammo_gen_ammo_and_stub_from_yt_binary,
        cache_daemon_resource=self.Parameters.ammo_cache_daemon_resource,
        d_planner_resource=self.Parameters.ammo_d_planner_resource,
        spec=self.Parameters.ammo_spec,
        format_yql=self.Parameters.ammo_format_yql,
        yql_query_description=self.Parameters.ammo_yql_query_description,
        yql_query=self.Parameters.ammo_yql_query,
        yql_query_role=self.Parameters.ammo_yql_query_role,
        yt_token_vault_name=self.Parameters.ammo_yt_token_vault_name,
        yql_token_vault_name=self.Parameters.ammo_yql_token_vault_name,
        pool=self.Parameters.ammo_pool,
        enable_page_id_coverage=self.Parameters.ammo_enable_page_id_coverage,
        days_interval=self.Parameters.ammo_days_interval,
        logs_interval=self.Parameters.ammo_logs_interval,
        resource_ttl=self.Parameters.ammo_resource_ttl,
        testenv_switch_trigger=self.Parameters.testenv_switch_trigger,
        testenv_switch_trigger_value=str(self.id),
    )
    ammo_generation_task.enqueue()
    self.set_info('Launched ammo generation task {}'.format(get_task_html_hyperlink(ammo_generation_task.id)), do_escape=False)
    return ammo_generation_task.id


@stage(provides="shard_map_resource_id")
def get_shard_map_resource_id(self):
    return find_last_ready_resource(YABS_SERVER_TESTENV_SHARD_MAP, {'testenv_switch_trigger': None}).id


@stage(provides="create_hamster_task_ids")
def create_hamsters(self):
    task_parameters = [
        (
            "rsya_hit_models_heavy_01",
            YabsServerCreateYabsHitModelsHamster,
            dict(
                replicas=self.Parameters.yabs_hit_models_hamster_replicas,
            )
        ),
        (
            "goalnet",
            YabsServerCreateGoalnetHamster,
            dict(
                shard_map_resource=get_shard_map_resource_id(self),
            )
        )
    ]

    create_hamster_task_ids = {}
    for service_tag, task_type, params in task_parameters:
        if service_tag not in self.Parameters.hamster_service_tags:
            logger.debug("Creation of \"%s\" hamster is disabled", service_tag)
            continue

        task = task_type(
            self,
            __requirements__={"tasks_resource": self.Requirements.tasks_resource},
            notifications=juggler_notifications(self, 'create_{}_hamster_endpoint'.format(service_tag)),
            owner=self.owner,
            tags=self.Parameters.tags,
            description=self.Parameters.description,
            testenv_switch_trigger=self.Parameters.testenv_switch_trigger,
            **params
        )

        task.enqueue()
        create_hamster_task_ids[service_tag] = task.id

    info = "Launched hamster creation task:\n"
    for service_tag, task_id in create_hamster_task_ids.items():
        info += "\t{service_tag}: {task_link}\n".format(
            service_tag=service_tag,
            task_link=get_task_html_hyperlink(task_id)
        )
    self.set_info(info, do_escape=False)
    return create_hamster_task_ids


@stage(provides='freeze_saas_state_task_id')
def freeze_saas_state(self):
    """Run task to freeze current KVRS SaaS state

    :return: task id
    :rtype: int
    """
    if not self.Parameters.freeze_kvrs_saas_state:
        logger.debug("Skipping freezing of KVRS SaaS state")
        return

    task = YabsServerFreezeCurrentKvrsSaasSnapshot(
        self,
        __requirements__={"tasks_resource": self.Requirements.tasks_resource},
        owner=self.owner,
        notifications=juggler_notifications(self, 'freeze_saas_state'),
        tags=self.Parameters.tags,
        description=self.Parameters.description,
        testenv_switch_trigger=self.Parameters.testenv_switch_trigger,
    )
    task.enqueue()
    self.set_info("Freeze current SaaS state {}".format(get_task_html_hyperlink(task.id)), do_escape=False)
    return task.id


@stage(provides='resources_spec')
def publish_resource_spec(
    self,
    mysql_archive_task_id,
    yt_archive_task_id,
    cs_settings_archive_task_id,
    ammo_task_id,
    create_hamster_task_ids=None,
):
    check_tasks(self, [
        mysql_archive_task_id,
        yt_archive_task_id,
        cs_settings_archive_task_id,
        ammo_task_id,
    ])

    oneshot_ext_service_endpoint_resources = {
        resource.service_tag: resource.id
        for resource in self.Parameters.oneshot_ext_service_endpoint_resources or []
    }
    hamsters_enabled_for_oneshots = list(itertools.chain.from_iterable(
        self.Parameters.oneshot_hamster_ext_service_tags.values()))
    logger.debug("Hamsters enabled for oneshots: %s", hamsters_enabled_for_oneshots)

    # Use excplicitly specified endpoint resource if available and dont wait for the hamster creating task to finish
    create_hamster_task_ids = create_hamster_task_ids or {}
    create_hamster_tasks_to_wait = {
        service_tag: task_id
        for service_tag, task_id in create_hamster_task_ids.items()
        if service_tag in hamsters_enabled_for_oneshots and service_tag not in oneshot_ext_service_endpoint_resources
    }
    check_tasks(self, list(create_hamster_tasks_to_wait.values()), raise_on_fail=False)

    linear_models_binary_resource = self.Parameters.linear_models_binary_resource or \
        get_last_released_resource("YABS_LINEAR_MODELS_EXECUTABLE")
    resources = {
        'mysql_archive_resource': YABS_MYSQL_ARCHIVE_CONTENTS.find(task_id=mysql_archive_task_id, limit=1).first().id,
        'yt_archive_resource': YABS_CS_INPUT_SPEC.find(task_id=yt_archive_task_id, limit=1).first().id,
        'cs_settings_archive_resource':
            YABS_CS_SETTINGS_ARCHIVE.find(task_id=cs_settings_archive_task_id, limit=1).first().id,
        'ammo_resources': sdk2.Task[ammo_task_id].Parameters.result_spec,
        'linear_models_binary_resource': linear_models_binary_resource.id,
    }

    for service_tag in hamsters_enabled_for_oneshots:
        if service_tag in oneshot_ext_service_endpoint_resources:
            logger.debug(
                "Use explicitly specified endpoint resource %s for hamster \"%s\"",
                oneshot_ext_service_endpoint_resources[service_tag], service_tag)
            continue

        if service_tag in create_hamster_tasks_to_wait:
            create_hamster_task = sdk2.Task[create_hamster_tasks_to_wait[service_tag]]
            if create_hamster_task.status in Status.SUCCESS:
                logger.debug(
                    "Use endpoint resource %s from task %s for hamster \"%s\"",
                    create_hamster_task.Parameters.external_service_spec.id,
                    create_hamster_task.id,
                    service_tag)
                oneshot_ext_service_endpoint_resources[service_tag] = \
                    create_hamster_task.Parameters.external_service_spec.id
                continue

        oneshot_ext_service_endpoint_resources[service_tag] = find_last_ready_resource(
            YabsServerExtServiceEndpoint,
            {'service_tag': service_tag, 'testenv_switch_trigger': None}
        ).id
        logger.debug(
            "Use last ready endpoint resource %s for hamster \"%s\"",
            oneshot_ext_service_endpoint_resources[service_tag], service_tag)

    resources['ext_service_endpoint_resources'] = list(oneshot_ext_service_endpoint_resources.values())

    return resources


@stage(provides='one_shot_spec_generation_task_id')
def launch_one_shot_spec_generation(
    self,
    resources_spec,
    stat_bs_release_tar_resource_id,
    stat_bs_release_yt_resource_id,
    meta_bs_release_tar_resource_id,
    meta_bs_release_yt_resource_id,
):
    one_shot_spec_generation_task = YabsServerCreateOneShotSpec(
        self,
        owner=self.owner,
        tags=self.Parameters.tags,
        description=self.Parameters.description,
        notifications=juggler_notifications(self, 'create_oneshot_spec'),
        stat_bs_release_tar_resource=stat_bs_release_tar_resource_id,
        stat_bs_release_yt_resource=stat_bs_release_yt_resource_id,
        meta_bs_release_tar_resource=meta_bs_release_tar_resource_id,
        meta_bs_release_yt_resource=meta_bs_release_yt_resource_id,
        ft_shards=self.Parameters.one_shot_spec_ft_shards,
        create_load_spec=self.Parameters.one_shot_spec_create_load_spec,
        add_meta_load=self.Parameters.one_shot_spec_add_meta_load,
        meta_load_shards=self.Parameters.one_shot_spec_meta_load_shards,
        stat_load_shards=self.Parameters.one_shot_spec_load_shards,
        generate_separated_bases=self.Parameters.one_shot_spec_generate_separated_bases,
        generate_not_separated_bases=self.Parameters.one_shot_spec_generate_not_separated_bases,
        mysql_archive_resource=resources_spec['mysql_archive_resource'],
        cs_input_spec_resource=resources_spec['yt_archive_resource'],
        cs_settings_archive_resource=resources_spec['cs_settings_archive_resource'],
        linear_models_binary_resource=resources_spec['linear_models_binary_resource'],
        request_log_resource_id_map={
            role: resources_spec['ammo_resources']['{}_func'.format(role)]['requestlog_resource']
            for role in ('bs', 'bsrank', 'yabs')
        },
        ft_request_log_resource_id_map={
            role: resources_spec['ammo_resources']['{}_func'.format(role)]['requestlog_resource']
            for role in ('bs', 'bsrank', 'yabs')
        },
        meta_load_request_log_resource_id_map={
            role: resources_spec['ammo_resources']['{}_meta_load'.format(role)]['requestlog_resource']
            for role in ('bs', 'bsrank', 'yabs')
        },
        stat_load_request_log_resource_id_map={
            role: resources_spec['ammo_resources']['{}_load'.format(role)]['requestlog_resource']
            for role in ('bs', 'bsrank', 'yabs')
        },
        dolbilka_plan_resource_id_map={
            role: resources_spec['ammo_resources']['{}_load'.format(role)]['dplan_resource']
            for role in ('bs', 'bsrank', 'yabs')
        },
        release_spec=self.Parameters.one_shot_spec_release_spec,
        stat_load=True,
        meta_load=True,
        ft_shoot_settings_resource=self.Parameters.one_shot_spec_ft_shoot_settings_resource,
        ext_service_endpoint_resources=resources_spec['ext_service_endpoint_resources'],
        hamster_ext_service_tags=self.Parameters.oneshot_hamster_ext_service_tags,
        use_separated_meta_and_stat=self.Parameters.one_shot_spec_use_separated_meta_and_stat,
        shard_map_resource=get_shard_map_resource_id(self),
    )
    one_shot_spec_generation_task.enqueue()
    self.set_info('Launched OneShot spec generation task {}'.format(get_task_html_hyperlink(one_shot_spec_generation_task.id)), do_escape=False)
    return one_shot_spec_generation_task.id
