import itertools
import logging
import os
import shutil

from sandbox import sdk2
import sandbox.sandboxsdk.paths as sdk_paths
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.errors import TaskFailure
from sandbox.projects.yabs.qa.utils.resource import sync_resource
from sandbox.projects.yabs.qa.resource_types import (
    YabsServerProductionAnalyzerTool,
    YabsServerProductionAnalyzerHtmlReport,
    YabsServerProductionAnalyzerReport,
)

DEFAULT_YT_PROXY = 'hahn'
logger = logging.getLogger(__name__)

DEFAULT_SERVICES = {
    'YABS': {
        'STABLE': 'stable_yabs_frontend_server_yabs_sas',
        'PRESTABLE': 'prestable_yabs_frontend_server_yabs_sas',
        'EXPERIMENT': 'experiment_yabs_frontend_server_yabs_sas',
    },
    'BS': {
        'STABLE': 'stable_yabs_frontend_server_bs_msk_iva',
        'PRESTABLE': 'prestable_yabs_frontend_server_bs_msk_iva',
        'EXPERIMENT': 'experiment_yabs_frontend_server_bs_msk_iva',
    }
}

SERVICE_TYPE = ('STABLE', 'PRESTABLE', 'EXPERIMENT')
REPORT_DIR = 'report'
NANNY_TOKEN_DEFAULT_VAULT_KEY = 'YABS_RELEASER_NANNY_TOKEN'
YT_TOKEN_DEFAULT_VAULT_KEY = 'yabs-cs-sb-yt-token'
DEFAULT_REPORT_RESOURCE_TTL_IN_DAYS = 14


class YabsServerAnalyzeProduction(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        analysis_intervals = sdk2.parameters.List(
            'Analysis intervals',
            description='Two dates in ISO 8601 format joined by "/" symbol, '
                        'e.g. "2019-03-20 07:00:00+03 / 2019-03-20 10:00:00+03"',
            required=True,
        )
        validate_timeouts = sdk2.parameters.Bool('Validate timeouts of requests to external services', default=False)

        with sdk2.parameters.Group('YABS') as yabs_group:
            custom_yabs_services = sdk2.parameters.Bool('Custom', default=False, default_value=False)

            yabs_stable_default = DEFAULT_SERVICES['YABS']['STABLE']
            yabs_testing_default = DEFAULT_SERVICES['YABS']['PRESTABLE']
            with custom_yabs_services.value[False]:
                choices = [
                    (DEFAULT_SERVICES['YABS'][service_type], DEFAULT_SERVICES['YABS'][service_type])
                    for service_type in SERVICE_TYPE
                ]
                yabs_stable = sdk2.parameters.String('Stable', choices=choices, default=yabs_stable_default)
                yabs_testing = sdk2.parameters.String('Testing', choices=choices, default=yabs_testing_default)

            with custom_yabs_services.value[True]:
                yabs_stable_custom = sdk2.parameters.String(label='Stable', default=yabs_stable_default)
                yabs_testing_custom = sdk2.parameters.String(label='Testing', default=yabs_testing_default)

        with sdk2.parameters.Group("BS") as bs_group:
            custom_bs_services = sdk2.parameters.Bool("Custom", default=False, default_value=False)

            bs_stable_default = DEFAULT_SERVICES['BS']['STABLE']
            bs_testing_default = DEFAULT_SERVICES['BS']['PRESTABLE']
            with custom_bs_services.value[False]:
                choices = [
                    (DEFAULT_SERVICES['BS'][service_type], DEFAULT_SERVICES['BS'][service_type])
                    for service_type in SERVICE_TYPE
                ]
                bs_stable = sdk2.parameters.String('Stable', choices=choices, default=bs_stable_default)
                bs_testing = sdk2.parameters.String('Testing', choices=choices, default=bs_testing_default)

            with custom_bs_services.value[True]:
                bs_stable_custom = sdk2.parameters.String(label='Stable', default=bs_stable_default)
                bs_testing_custom = sdk2.parameters.String(label='Testing', default=bs_testing_default)

        with sdk2.parameters.Group("Other parameters") as other_group:
            yt_proxy = sdk2.parameters.String(label='YT proxy', default=DEFAULT_YT_PROXY)
            prod_analyzer_resource = sdk2.parameters.Resource(
                'Resource with production analyzer executable (if not specified use last stable resource)',
                resource_type=YabsServerProductionAnalyzerTool,
            )
            report_resource_ttl = sdk2.parameters.Integer(
                label='Report resource TTL in days',
                default=DEFAULT_REPORT_RESOURCE_TTL_IN_DAYS,
            )

            nanny_token_vault_key = sdk2.parameters.String(label='Nanny token vault key',
                                                           default=NANNY_TOKEN_DEFAULT_VAULT_KEY)
            yt_token_vault_key = sdk2.parameters.String(label='YT token vault key',
                                                        default=YT_TOKEN_DEFAULT_VAULT_KEY)

    def on_execute(self):
        production_analyzer_binary = sync_resource(
            resource=self.Parameters.prod_analyzer_resource,
            resource_type=YabsServerProductionAnalyzerTool
        )

        yt_token = self.get_yt_token()
        nanny_token = self.get_nanny_token()

        if os.path.isdir(REPORT_DIR):
            os.rmdir(REPORT_DIR)
        os.mkdir(REPORT_DIR)

        interval_args = list(itertools.chain(
            *zip(itertools.repeat('--interval'), self.Parameters.analysis_intervals)
        ))
        cmd = [
            production_analyzer_binary,
            '--yabs-stable', (self.Parameters.yabs_stable_custom
                              if self.Parameters.custom_yabs_services
                              else self.Parameters.yabs_stable),
            '--yabs-testing', (self.Parameters.yabs_testing_custom
                               if self.Parameters.custom_yabs_services
                               else self.Parameters.yabs_testing),
            '--bs-stable', (self.Parameters.bs_stable_custom
                            if self.Parameters.custom_bs_services
                            else self.Parameters.bs_stable),
            '--bs-testing', (self.Parameters.bs_testing_custom
                             if self.Parameters.custom_bs_services
                             else self.Parameters.bs_testing),
            '--yt-proxy', self.Parameters.yt_proxy,
            '--yt-token', yt_token,
            '--nanny-token', nanny_token,
            '--out-dir', REPORT_DIR,
            '--debug',
            '--no-color',
        ] + interval_args
        if self.Parameters.validate_timeouts:
            cmd.append('--validate-timeouts')

        logs_dir = sdk_paths.get_logs_folder()
        log_name = 'prod-analyzer'
        out_filename = log_name + '.out'
        err_filename = log_name + '.err'

        logger.info('Run production analyzer: "{}"'.format(' '.join(cmd)))
        with open(os.path.join(logs_dir, out_filename), 'w') as out, \
                open(os.path.join(logs_dir, err_filename), 'w') as err:
            logger.info(
                'Redirect stdout to "{stdout}", stderr to "{stderr}"'
                .format(stdout=out_filename, stderr=err_filename)
            )
            return_code = sp.Popen(cmd, stdout=out, stderr=err).wait()

        if return_code:
            raise TaskFailure(
                'Command "{command}" exited with nonzero code: {code}. See more in {err_filename}'
                .format(
                    command=' '.join(cmd),
                    code=return_code,
                    err_filename=err_filename,
                )
            )

        report_data_path = os.path.join(REPORT_DIR, 'data.json')
        report_resource = YabsServerProductionAnalyzerReport(
            self,
            description='Production analyzer report',
            path=report_data_path,
            ttl=self.Parameters.report_resource_ttl,
        )
        sdk2.ResourceData(report_resource).ready()

        html_report_dir = os.path.join(REPORT_DIR, 'html_report')
        if not os.path.isdir(html_report_dir):
            os.makedirs(html_report_dir)
        shutil.move(os.path.join(REPORT_DIR, 'report.html'), html_report_dir)
        shutil.move(os.path.join(REPORT_DIR, 'plot'), html_report_dir)
        html_report_resource = YabsServerProductionAnalyzerHtmlReport(
            self,
            description='Production analyzer HTML report',
            path=html_report_dir,
            ttl=self.Parameters.report_resource_ttl,
        )
        sdk2.ResourceData(html_report_resource).ready()

    def get_yt_token(self):
        return sdk2.Vault.data(self.Parameters.yt_token_vault_key)

    def get_nanny_token(self):
        return sdk2.Vault.data(self.Parameters.nanny_token_vault_key)
