import datetime
import itertools
import json
import logging
import os
import time
import urlparse
from collections import namedtuple

import dateutil
import pytz
from enum import Enum

from sandbox import sdk2
from sandbox.common.types import task as task_type
from sandbox.common.types.misc import NotExists
from sandbox.projects.abc.client import AbcClient
from sandbox.projects.common.time_utils import datetime_to_timestamp
from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.common.yabs.server.util import send_startrek_report
from sandbox.projects.yabs.qa.resource_types import (
    YabsServerProductionAnalyzerReport,
    YabsServerProductionAnalyzerHtmlReport,
    YabsServerProductionAnalyzerFlameGraphReport
)
from sandbox.projects.yabs.qa.constants import AB_EXPERIMENT_OBSERVATIONS
from sandbox.projects.yabs.qa.tasks.YabsServerAnalyzeProduction import YabsServerAnalyzeProduction
from sandbox.projects.yabs.qa.utils.resource import sync_resource
from sandbox.projects.yabs.qa.utils.general import get_task_html_hyperlink
from sandbox.projects.yabs.qa.utils.datetime_interval import split_datetime_interval, DatetimeInterval, round_ab_time_interval
from sandbox.projects.yabs.qa.utils.subtasks import run_tasks
from sandbox.projects.yabs.qa.template_utils import get_template
from sandbox.projects.yabs.release.common import BaseReleaseTask
from sandbox.projects.yabs.YabsBuildFlameGraph import YabsBuildFlameGraph, YabsFlameGraph
from sandbox.projects.yabs.release.notifications.environment.report_info import TaskInfo
from sandbox.projects.yabs.release.notifications.jns.client import send_message
from sandbox.projects.yabs.release.notifications.jns.helpers import get_logins


logger = logging.getLogger(__name__)

UTC_DAY_BEGINNING = datetime.time(hour=4, tzinfo=pytz.utc)
UTC_DAY_END = datetime.time(hour=18, tzinfo=pytz.utc)
DEFAULT_INITIAL_ANALYSIS_DURATION = datetime.timedelta(hours=5).total_seconds()
DEFAULT_MAX_ANALYSIS_DURATION = datetime.timedelta(hours=9).total_seconds()
DEFAULT_TIME_TO_WAIT = datetime.timedelta(hours=1).total_seconds()
DEFAULT_STARTREK_VAULT_KEY = 'robot-yabs-cs-b2b-startrek-token'
REPORT_FILENAME = 'report.html'

CROSS_MARK_CODE = u'\U0000274c'
CHECK_MARK_CODE = u'\U00002705'
TEMPLATES_DIR = os.path.join(os.path.dirname(__file__), '..', '..', '..', 'release', 'notifications', 'templates')
TEMPLATE_NAME = 'analyzer_report.j2'


class OutOfTime(Exception):
    pass


Status = namedtuple('Status', ('text', 'icon'))


class Events(Enum):
    done = Status("done", u"\U00002705")
    not_done = Status("not done yet, will try again later", u"\U0000231B")
    failed = Status("failed", u"\U0001F6AB")


class ReportData(object):
    __slots__ = (
        'task', 'component_name', 'status', 'tags',
        'versions', 'report_links', 'ab_experiment_links', 'checks', 'analyze_production_task', 'analyze_production_task_status',
    )

    def __init__(
            self,
            task_id=0,
            task_type='SandboxTask',
            component_name='yabs_server',
            status=None,

            yabs_testing_version=None,
            yabs_stable_version=None,
            bs_testing_version=None,
            bs_stable_version=None,
            report_links=(),
            ab_experiment_links=(),
            checks=(),
            analyze_production_task_id=0,
            analyze_production_task_type='SandboxTask',
            analyze_production_task_status='SUCCESS',
    ):

        self.task = TaskInfo(task_type, task_id) if task_id else None

        self.component_name = component_name
        self.status = status
        self.tags = ['analysis', component_name]

        self.versions = {
            'YABS': (yabs_stable_version or 'Unknown', yabs_testing_version or 'Unknown'),
            'BS': (bs_stable_version or 'Unknown', bs_testing_version or 'Unknown'),
        } if any([yabs_stable_version, yabs_testing_version, bs_stable_version, bs_testing_version]) else None
        self.report_links = report_links
        self.ab_experiment_links = ab_experiment_links
        self.checks = checks
        self.analyze_production_task = TaskInfo(analyze_production_task_type, analyze_production_task_id) if analyze_production_task_id else None
        self.analyze_production_task_status = analyze_production_task_status

    def as_dict(self, transport='telegram', mentions=()):
        _dict = {k: getattr(self, k) for k in self.__slots__}
        _dict.update(transport=transport, mentions=mentions)
        return _dict


class YabsServerAnalyzeProductionWrapper(BaseReleaseTask):
    class Requirements(BaseReleaseTask.Requirements):
        cores = 1
        ram = 1024

        class Caches(BaseReleaseTask.Requirements.Caches):
            pass

    class Parameters(BaseReleaseTask.Parameters):

        with sdk2.parameters.Group('Analyzer') as analyzer_group:
            initial_analysis_duration = sdk2.parameters.Integer(
                label='Initial analysis duration (seconds)',
                default=DEFAULT_INITIAL_ANALYSIS_DURATION,
            )
            max_analysis_duration = sdk2.parameters.Integer(
                label='Maximal analysis duration (seconds)',
                default=DEFAULT_MAX_ANALYSIS_DURATION,
            )
            time_to_wait = sdk2.parameters.Integer(
                label='Time to wait between analysis runs (seconds)',
                default=DEFAULT_TIME_TO_WAIT,
            )
            validate_timeouts = sdk2.parameters.Bool(
                label='Validate timeouts of requests to external services',
                default=False
            )
            start_timestamp = sdk2.parameters.Integer(
                label='Start timestamp (UTC)',
                description='Lower limit of analysis start timestamp in UTC',
                default=None,
            )

            bs_stable = sdk2.parameters.String(label='BS stable', default=YabsServerAnalyzeProduction.Parameters.bs_stable_default)
            bs_testing = sdk2.parameters.String(label='BS testing', default=YabsServerAnalyzeProduction.Parameters.bs_testing_default)
            yabs_stable = sdk2.parameters.String(label='YABS stable', default=YabsServerAnalyzeProduction.Parameters.yabs_stable_default)
            yabs_testing = sdk2.parameters.String(label='YABS testing', default=YabsServerAnalyzeProduction.Parameters.yabs_testing_default)

        with sdk2.parameters.Group("Other parameters") as other_group:
            yt_proxy = YabsServerAnalyzeProduction.Parameters.yt_proxy()
            prod_analyzer_resource = YabsServerAnalyzeProduction.Parameters.prod_analyzer_resource()
            report_resource_ttl = YabsServerAnalyzeProduction.Parameters.report_resource_ttl()

        with sdk2.parameters.Output:
            analysis_succeed = sdk2.parameters.Bool(
                'Analysis succeed',
                default=False
            )

    def on_finish(self, prev_status, status):
        self.notify_startrek_on_analysis_end(
            self.Context.analyze_production_task_id,
            self.Context.flame_graph_report_resource_id,
            self.Context.ab_links,
            analysis_is_done=True
        )
        self.notify_telegram_on_analysis_end(
            self.Context.analyze_production_task_id,
            self.Context.flame_graph_report_resource_id,
            self.Context.ab_links,
            analysis_is_done=True
        )

    def on_execute(self):
        with self.memoize_stage.first_run(commit_on_entrance=False):
            self.Context.flame_graph_report_id = 0

            start_timestamp = self.Parameters.start_timestamp
            if not start_timestamp:
                logger.info('Parameter "start timestamp" is not set. Will try to start analysis from current moment.')
                start_timestamp = int(time.time())
            start_datetime = datetime.datetime.fromtimestamp(start_timestamp, tz=pytz.utc)

            logger.info('Start datetime: %s', start_datetime.isoformat())
            self.Context.start_datetime = start_datetime.isoformat()
            self.Context.start_timestamp = start_timestamp

            self.Context.analysis_duration = self.Parameters.initial_analysis_duration
            analysis_intervals = split_datetime_interval(
                start=start_datetime,
                duration=datetime.timedelta(seconds=self.Parameters.initial_analysis_duration),
                day_start_time=UTC_DAY_BEGINNING,
                day_end_time=UTC_DAY_END,
            )
            self.Context.ab_links = get_ab_links(analysis_intervals)
            self.Context.analysis_intervals = map(str, analysis_intervals)

            wait_until_dt = max(analysis_intervals, key=lambda interval_: interval_.end).end
            logger.info('Wait until %s', wait_until_dt)
            wait_until(wait_until_dt)

        if self.Context.analyze_production_task_id in (NotExists, None):
            self.Context.analyze_production_task_id = self.get_analysis_task()
            self.Context.build_flame_graph_task_ids = self.get_build_flame_graph_tasks()

            self.Context.wait_tasks = [
                self.Context.analyze_production_task_id
            ] + self.Context.build_flame_graph_task_ids

            self.Context.all_tasks = self.Context.wait_tasks

            run_tasks(self, self.Context.all_tasks)

        check_tasks(self, self.Context.wait_tasks, raise_on_fail=False)

        analyze_production_task = sdk2.Task[self.Context.analyze_production_task_id]
        if analyze_production_task.status not in task_type.Status.Group.SUCCEED:
            self.Parameters.analysis_succeed = False
            return

        self.Context.flame_graph_report_resource_id = self.build_flame_graph_report()

        production_analysis_report = get_production_analysis_report(analyze_production_task)
        failed_checks = get_failed_checks(production_analysis_report)

        analysis_succeed = True
        if failed_checks:
            logger.info('Failed checks: %s', failed_checks)
            analysis_succeed = False

        failed_clicks = filter(
            lambda item: (
                item['filter'] in (['IsSerp'], ['IsMobileSerp'], ['IsDsp'])
                and item['parameter'] in ('Clicks', 'GoodClicks')
            ),
            failed_checks
        )
        logger.debug('failed_clicks: %s', json.dumps(failed_clicks, indent=2))

        self.Context.enough_clicks = not failed_clicks
        message = 'Got enough clicks' if self.Context.enough_clicks else 'Not enough clicks'
        logging.info(message)
        self.set_info(message)

        if self.Parameters.validate_timeouts:
            self.Context.failed_timeouts = get_failed_timeouts(production_analysis_report)
            logger.info('Failed timeouts: %s', self.Context.failed_timeouts)
            analysis_succeed = analysis_succeed and self.Context.failed_timeouts

        self.Context.important_failed_checks = filter(
            lambda item: (
                item['filter'] in (['IsSerp'], ['IsMobileSerp'], ['IsDsp'])
                and item['parameter'] not in ('Clicks', 'GoodClicks', 'Shows', 'GoodShows')
            ),
            failed_checks
        )

        if analysis_succeed:
            self.Parameters.analysis_succeed = True
            return

        if self.Context.enough_clicks:
            self.Parameters.analysis_succeed = False
            return

        with self.memoize_stage.send_telegram_notification(commit_on_entrance=False):
            self.notify_telegram_on_analysis_end(
                self.Context.analyze_production_task_id,
                self.Context.flame_graph_report_resource_id,
                self.Context.ab_links,
            )

        try:
            self.try_wait_until_next_analysis()
        except OutOfTime:
            self.Parameters.analysis_succeed = False
            return

    def try_wait_until_next_analysis(self):
        last_analysis_intervals = [
            DatetimeInterval.from_str(interval)
            for interval in self.Context.analysis_intervals
        ]
        logger.debug('last_analysis_intervals: %s', last_analysis_intervals)
        last_analysis_duration = sum([interval.duration() for interval in last_analysis_intervals],
                                     datetime.timedelta())
        analysis_duration = last_analysis_duration + datetime.timedelta(seconds=self.Parameters.time_to_wait)
        if analysis_duration > datetime.timedelta(seconds=self.Parameters.max_analysis_duration):
            raise OutOfTime()

        self.Context.analysis_duration = analysis_duration.total_seconds()

        self.Context.analyze_production_task_id = None
        self.Context.build_flame_graph_task_ids = None

        start_datetime = dateutil.parser.parse(self.Context.start_datetime)

        # Explicitly set timezone to UTC because in python-dateutil v2.5 (bundled in sandbox) tzlocal() != tzutc()
        # Remove it when python-dateutil will be upgraded at least to v2.7 https://st.yandex-team.ru/SANDBOX-6326
        start_datetime = start_datetime.replace(tzinfo=pytz.utc)

        analysis_intervals = split_datetime_interval(
            start=start_datetime,
            duration=analysis_duration,
            day_start_time=UTC_DAY_BEGINNING,
            day_end_time=UTC_DAY_END,
        )
        self.Context.analysis_intervals = map(str, analysis_intervals)
        wait_until_dt = max(analysis_intervals, key=lambda interval_: interval_.end).end
        logger.info('Wait until %s', wait_until_dt)
        wait_until(wait_until_dt, simulate_wait=True)

    def handle_production_analysis_task_failure(self, task):
        message = 'Failed to analyze production. Task {link} finished with status {task.status}'\
            .format(link=get_task_html_hyperlink(task.id), task=task)
        message += '\n{}, please check if everything is ok with (pre)stable.'.format(self.get_mention())
        self.set_info(message, do_escape=False)
        self.telegram_notify(
            '{self_link}\n{message}'
            .format(self_link=get_task_html_hyperlink(self.id),
                    message=message)
        )

        self.Parameters.analysis_succeed = False

    def get_recipients(self):
        if self.Context.copy_of:
            return {
                "yachats": {"internal": [{"login": self.author}]},
                "telegram": {"internal": [{"login": self.author}]},
            }
        return {
            "telegram": {"chat_name": ["yabs_server_release_chat"]},
            "yachats": {"chat_name": ["yabs_server_release_chat"]},
        }

    def _notify_event(self, report_data):
        report_template_j2 = get_template(TEMPLATE_NAME, templates_dir=TEMPLATES_DIR)

        html_report = report_template_j2.render(report_data.as_dict(transport='html'))
        self.set_info(html_report, do_escape=False)

        tokens = sdk2.yav.Secret("sec-01fx7jcsjevejnypw63tk26nj3").data()
        try:
            spawn_users = list(set([self.author, AbcClient(tokens['abc_token']).get_current_duty_login(179, schedule_slug='yabs_frontend_duty_first')]))
            for transport, recipient in self.get_recipients().items():
                mentions = get_logins(self, spawn_users, transport=transport)
                report = report_template_j2.render(report_data.as_dict(transport=transport, mentions=mentions))
                send_message(report, tokens['juggler_token'], recipients={transport: recipient})
        except Exception:
            self.set_info('Cannot send notification')
            logger.error('Cannot send notification', exc_info=True)

    def get_report_data(self, analyze_production_task_id, flame_graph_report_resource_id, ab_links, analysis_is_done=False):
        analyze_production_task = sdk2.Task[analyze_production_task_id]
        if analyze_production_task.status not in task_type.Status.Group.SUCCEED:
            return ReportData(
                task_id=self.id,
                task_type=str(self.type),
                status=Events.failed,
                analyze_production_task_id=analyze_production_task_id,
                analyze_production_task_type=analyze_production_task.type,
                analyze_production_task_status=analyze_production_task.status,
            )

        production_analysis_html_report = get_production_analysis_html_report(analyze_production_task)
        html_report_url = urlparse.urljoin(production_analysis_html_report.http_proxy + '/', REPORT_FILENAME)

        flame_graph_report_url = sdk2.Resource[flame_graph_report_resource_id].http_proxy

        production_analysis_report = get_production_analysis_report(analyze_production_task)
        services = production_analysis_report.get('services', {})

        yabs_testing_version = services.get('yabs-testing', {}).get('version')
        yabs_stable_version = services.get('yabs-stable', {}).get('version')
        bs_testing_version = services.get('bs-testing', {}).get('version')
        bs_stable_version = services.get('bs-stable', {}).get('version')

        checks = []
        try:
            if self.Context.enough_clicks:
                checks.append((CHECK_MARK_CODE, 'Enough clicks'))
            else:
                checks.append((CROSS_MARK_CODE, 'Not enough clicks'))
        except AttributeError:
            pass

        try:
            if self.valid_timeouts:
                checks.append((CHECK_MARK_CODE, 'Timeouts validation passed'))
            else:
                checks.append((CROSS_MARK_CODE, 'Timeouts validation failed'))
        except AttributeError:
            pass

        try:
            if self.Context.important_failed_checks:
                checks.append((CROSS_MARK_CODE, 'Statistics is bad'))
            else:
                checks.append((CHECK_MARK_CODE, 'Statistics is good'))
        except AttributeError:
            pass

        return ReportData(
            task_id=self.id,
            task_type=str(self.type),
            component_name=self.Parameters.component_name,
            yabs_testing_version=yabs_testing_version,
            yabs_stable_version=yabs_stable_version,
            bs_testing_version=bs_testing_version,
            bs_stable_version=bs_stable_version,
            status=Events.done if analysis_is_done else Events.not_done,
            report_links=(
                ('Analysis report', html_report_url),
                ('Flame graphs', flame_graph_report_url),
            ),
            ab_experiment_links=ab_links,
            checks=checks,
        )

    def notify_telegram_on_analysis_end(self, analyze_production_task_id, flame_graph_report_resource_id, ab_links, analysis_is_done=False):
        report_data = self.get_report_data(analyze_production_task_id, flame_graph_report_resource_id, ab_links, analysis_is_done=analysis_is_done)
        self._notify_event(report_data)

    def notify_startrek_on_analysis_end(self, analyze_production_task_id, flame_graph_report_resource_id, ab_links, analysis_is_done=False):
        report_data = self.get_report_data(analyze_production_task_id, flame_graph_report_resource_id, ab_links, analysis_is_done=analysis_is_done)
        report_template_j2 = get_template(TEMPLATE_NAME, templates_dir=TEMPLATES_DIR)
        startrek_report = report_template_j2.render(report_data.as_dict(transport='startrek'))

        try:
            self.startrek_notify(startrek_report)
        except Exception as exc:
            logger.exception(exc)

    def get_analysis_task(self):
        analyze_production_task = YabsServerAnalyzeProduction(
            self,
            description=(
                '{analysis_intervals} Run from {self_name}'
                .format(
                    analysis_intervals=self.Context.analysis_intervals,
                    self_name=self.type,
                )
            ),
            analysis_intervals=self.Context.analysis_intervals,
            validate_timeouts=self.Parameters.validate_timeouts,
            prod_analyzer_resource=self.Parameters.prod_analyzer_resource,
            report_resource_ttl=self.Parameters.report_resource_ttl,
            bs_stable=self.Parameters.bs_stable,
            bs_testing=self.Parameters.bs_testing,
            yabs_stable=self.Parameters.yabs_stable,
            yabs_testing=self.Parameters.yabs_testing,
            yt_proxy=self.Parameters.yt_proxy,
        )

        return analyze_production_task.id

    def get_services_bunch(self, role, stage):
        nanny_service = getattr(
            self.Parameters,
            '{}_{}'.format(role, stage)
        )
        return [nanny_service + suffix for suffix in ('_meta', '_stat')]

    def get_build_flame_graph_tasks(self):
        build_flame_graph_task_ids = []
        roles = {'bs': 'metapartner', 'yabs': 'metasearch'}
        for role in ['bs', 'yabs']:
            for program in ['meta', 'stat']:
                role_configs = {}
                for stage in ['stable', 'testing']:
                    start_time = self.Context.start_timestamp
                    end_time = start_time + self.Context.analysis_duration
                    role_configs[stage] = {
                        'nanny_services': self.get_services_bunch(role, stage),
                        'start_time': start_time,
                        'end_time': end_time,
                        'role': roles[role],
                        'program': program
                    }

                build_flame_graph_task = YabsBuildFlameGraph(
                    self,
                    description='{role}/{program}'.format(role=role, program=program),
                    mode='diff',
                    base_parameters=role_configs['stable'],
                    diff_parameters=role_configs['testing'],
                    flame_graph_ttl=self.Parameters.report_resource_ttl,
                    yt_cluster=self.Parameters.yt_proxy,
                )
                build_flame_graph_task_ids.append(build_flame_graph_task.id)

        return build_flame_graph_task_ids

    def build_flame_graph_report(self):
        tasks = map(lambda task_id: sdk2.Task[task_id], self.Context.build_flame_graph_task_ids)
        flame_graphs = []
        for task in tasks:
            if task.status not in task_type.Status.Group.SUCCEED:
                logger.warning('YabsFlameGraph task {task.id} has failed with status {task.status}'.format(
                    task=task
                ))
                continue

            flame_graph_resource = YabsFlameGraph.find(task=task).first()

            if not flame_graph_resource:
                logger.warning('YabsFlameGraph resource for task "{task.id}" not found'.format(
                    task=task
                ))
                continue
            flame_graphs.append((flame_graph_resource.http_proxy, task.Parameters.description))

        report_resource = YabsServerProductionAnalyzerFlameGraphReport(
            self,
            'Flame graph report',
            'flame_graph_report_{}.html'.format(self.Context.flame_graph_report_id),
            ttl=self.Parameters.report_resource_ttl,
        )

        self.Context.flame_graph_report_id += 1

        report_resource_data = sdk2.ResourceData(report_resource)
        report_path = str(report_resource_data.path)
        with open(report_path, 'w') as report_file:
            report_file.write(get_flame_graph_report_html(flame_graphs))
        report_resource_data.ready()

        return report_resource.id

    def startrek_notify(self, message):
        ticket = self.get_release_ticket()
        logger.info('Send comment to startrek ticket %s: "%s"', ticket, message)
        send_startrek_report(
            sdk2.Vault.data(self.Parameters.st_vault_name),
            ticket,
            message,
        )


def wait_until(dt, simulate_wait=False):
    """
    Wait until datetime by raising sdk2.WaitTime

    :param dt: Wait until datetime
    :param simulate_wait: Wait even if `dt` is passed. Useful to re-enqueue the task
    """
    now = datetime.datetime.now(tz=pytz.utc)
    time_to_wait = dt - now

    if time_to_wait > datetime.timedelta():
        logger.debug('Wait for %s until %s', time_to_wait, dt)
        raise sdk2.WaitTime(time_to_wait=time_to_wait.total_seconds())

    if simulate_wait:
        logger.debug('Wait for 10 seconds to re-enqueue the task')
        raise sdk2.WaitTime(time_to_wait=10)


def get_failed_checks(analyzer_output):
    result = []
    for item in analyzer_output['items']:
        for param, param_data in item['data'].iteritems():
            for key, value in param_data.iteritems():
                if not value['ok']:
                    result.append({
                        'filter': item['filter'],
                        'table': item['table'],
                        'parameter': param,
                        'key': key,
                        'value': value['value'],
                        'note': value['note']
                    })
    return result


def get_failed_timeouts(analyzer_output):
    result = []
    for item in analyzer_output['timeouts_validation']:
        failed = filter(
            lambda obj: not obj['success'],
            item['data']
        )
        result.append(map(
            lambda obj: {
                'ext_tag': obj['ext_tag'],
                'handler': obj['handler'],
                'role': item['role'],
            },
            failed
        ))
    return list(itertools.chain(*result))


def get_production_analysis_report(analyze_production_task):
    report_resource = YabsServerProductionAnalyzerReport\
        .find(task=analyze_production_task)\
        .first()
    report_path = sync_resource(resource=report_resource)
    with open(report_path) as report_file:
        return json.load(report_file)


def get_production_analysis_html_report(analyze_production_task):
    return YabsServerProductionAnalyzerHtmlReport\
        .find(task=analyze_production_task)\
        .first()


def get_flame_graph_report_html(flame_graphs):
    HTML_TEMPLATE = """
    <!DOCTYPE html>
    <html>
        <head>
            <title>Flame Graph report</title>
        </head>
        <body>
            <ul>
                {flame_graph_list}
            </ul>
        </body>
    </html>
    """
    FLAME_GRAPH_TEMPLATE = '<li><a href="{link}">{description}</a></li>'
    flame_graph_list = '\n'.join([FLAME_GRAPH_TEMPLATE.format(link=link, description=description) for link, description in flame_graphs])
    return HTML_TEMPLATE.format(flame_graph_list=flame_graph_list)


def get_ab_links(analysis_intervals):
    links = {}
    for analysis_interval in analysis_intervals:
        start_interval, end_interval = round_ab_time_interval(analysis_interval)
        ts_start, ts_end = int(datetime_to_timestamp(start_interval)), int(datetime_to_timestamp(end_interval))
        date_start, date_end = start_interval.strftime('%Y%m%d'), end_interval.strftime('%Y%m%d')
        links['{} - {}'.format(start_interval.isoformat(' '), end_interval.isoformat(' '))] = build_ab_stat_links(date_start, date_end, ts_start, ts_end)
    return links


def build_ab_stat_links(date_start, date_end, ts_start, ts_end):
    links = {
        observation_name:
        'https://ab.yandex-team.ru/observation/{}/calc/bs/{}/{}'
        '?granularity=half&groups=clean_frauds_staff&aggr=fast&cache=1'
        '&ts_start={}&ts_end={}#colored=true'.format(observation_id, date_start, date_end, ts_start, ts_end)
        for observation_id, observation_name in AB_EXPERIMENT_OBSERVATIONS.items()}
    return links
