# -*- coding: utf-8 -*-
import json
import datetime
import time

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.yabs.qa.resource_types import YABS_CS_SETTINGS_ARCHIVE


SETTINGS_PATH = 'settings.json'
DEFAULT_SETTINGS_BLACKLIST = frozenset([
    'enable_direct_context_archiving',
    'resources_archive_path',
    'run_vanilla',
    'enable_dynamic_resource_export',
    'enable_smart_resource_export',
    'enable_fs_smart_resource_export',
    'enable_wordstat_export',
    'direct_price_export_link',
    'publish_lifecycle_events',
    'use_job_tracer',
    'write_generation_info',
    'reduce_max_failed_job_count',
    'use_premkdb',
])


DEFAULT_SETTINGS_UPDATE = {
    'sys_const_update': [{
        'value': {
            'SharedRtbDspCacheTtlMs': 0,
        }
    }],
}


class YabsServerArchiveCSSettings(sdk2.Task):
    name = 'YABS_SERVER_ARCHIVE_CS_SETTINGS'
    description = 'Creates sandbox resource with content system current production settings'

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        environments = (
            PipEnvironment('yandex-yt', use_pre=True, use_wheel=True),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = datetime.timedelta(minutes=30).total_seconds()

        with sdk2.parameters.Group('Settings location') as settings_location:
            yt_proxy = sdk2.parameters.String(
                'YT proxy that stores content system production settings',
                required=True,
                default='locke',
            )
            settings_path = sdk2.parameters.String(
                'Path to the yt node with settings',
                required=True,
                default='//home/samogon/yabscs_production/yabs-cs/settings/v5_head',
            )
            dummy_mode = sdk2.parameters.Bool("Do not save settings to resource")

        with sdk2.parameters.Group('Testenv options') as testenv_options:
            switch_testenv_bases = sdk2.parameters.Bool('Switch testenv bases to new resource', default=False, do_not_copy=True)
            testenv_switch_trigger_value = sdk2.parameters.String('Use following string as testenv switch trigger value, backup date will be used if empty', default='')

        with sdk2.parameters.Group('Yt token params') as yt_token_params:
            yt_token_vault_name = sdk2.parameters.String('Vault name for YT token', required=True, default='yabs-cs-sb-yt-token')

        with sdk2.parameters.Group('Patch settings') as specific_options:
            set_now_unixtime = sdk2.parameters.Bool('Set cs "now_unixtime" setting to current timestamp')
            settings_blacklist = sdk2.parameters.List('List of settings which will be ignored on saving archive', required=False, default=DEFAULT_SETTINGS_BLACKLIST)
            settings_update = sdk2.parameters.JSON('Update production settings with custom values', required=False, default=DEFAULT_SETTINGS_UPDATE)

    def yt_token(self):
        return sdk2.Vault.data(
            self.Parameters.yt_token_vault_name,
        )

    def on_execute(self):
        import yt.wrapper.client

        settings = {}

        proxy = self.Parameters.yt_proxy

        if not self.Parameters.dummy_mode:
            token = self.yt_token()

            client = yt.wrapper.client.Yt(proxy=proxy, config={"token": token})
            settings_list = json.loads(client.get(self.Parameters.settings_path, format='json'))
            settings = settings_list['settings']

        settings_blacklist = self.Parameters.settings_blacklist or []
        for setting in settings_blacklist:
            settings.pop(setting, None)

        if self.Parameters.settings_update:
            settings.update(self.Parameters.settings_update)

        if self.Parameters.set_now_unixtime:
            settings['now_unixtime'] = [{'value': int(time.time())}]

        with open(SETTINGS_PATH, 'w') as out:
            json.dump(settings, out, indent=2, sort_keys=True)

        settings_archive_res = YABS_CS_SETTINGS_ARCHIVE(
            self,
            'Archive with content system production settings',
            SETTINGS_PATH,
            arch='any',
            yt_proxy=proxy,
        )
        if self.Parameters.switch_testenv_bases:
            testenv_switch_trigger_value = self.Parameters.testenv_switch_trigger_value or datetime.date.today().strftime('%Y%m%d')
            settings_archive_res.testenv_switch_trigger = testenv_switch_trigger_value

        sdk2.ResourceData(settings_archive_res).ready()
