from __future__ import division

import sys
import traceback

import json
import logging
import multiprocessing
import os
import posixpath
import re
import shutil
import textwrap
import time
import uuid

from collections import namedtuple
from cProfile import Profile
from pstats import Stats

from enum import Enum
from sandbox import sdk2

from sandbox.sandboxsdk import environments

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.common.errors import TaskError, TaskFailure
from sandbox.common.fs import get_unique_file_name, make_folder
from sandbox.common.types.task import ReleaseStatus
from sandbox.common.types.resource import RestartPolicy

from sandbox.projects.common.yabs.server import requestlog
from sandbox.projects.common.yabs.server.util import pack_utils
from sandbox.projects.common.yabs.server.util.general import try_get_from_vault, check_tasks
from sandbox.projects.common.yabs.server.util import truncate_output_parameters
from sandbox.projects.yabs.base_bin_task import BaseBinTaskMixin, base_bin_task_parameters
from sandbox.projects.yabs.qa.hamster.record import ENDPOINTS_TABLE_PROXY
from sandbox.projects.yabs.qa.hamster.testenv import get_ok_testenv_hamster_endpoints
from sandbox.projects.yabs.qa.hamster.utils import (
    get_bad_hamsters,
    get_hamster_url,
    shoot_task_enabled_hamsters,
)

from sandbox.projects.yabs.qa.constants import DEBUG_COOKIE
from sandbox.projects.yabs.qa.sut.bases_provider.adapters.sandbox import BaseStateSandboxAdapter
from sandbox.projects.yabs.qa.sut.solomon_stats import (
    dump_solomon_stats,
    create_saas_solomon_stats_validation_report,
    validate_saas_solomon_stats,
)
from sandbox.projects.yabs.qa.sut.components.linear_models import YabsLinearModels
from sandbox.projects.yabs.qa.sut.metastat.adapters.sandbox import YabsMetaSandboxAdapter, YabsStatSandboxAdapter
from sandbox.projects.yabs.qa.sut.metastat.adapters.sandbox.parameters import YabsSUTParameters
from sandbox.projects.yabs.qa.ammo_module.requestlog.adapters.yabs_specific.sandbox import (
    AmmoRequestlogModuleYabsSpecificSandboxAdapter,
)
from sandbox.projects.yabs.qa.ammo_module.requestlog import EAmmoFileType
from sandbox.projects.yabs.qa.dolbilo_module.simple.adapters.sandbox import DolbiloModuleSandboxAdapter
from sandbox.projects.yabs.qa.errorbooster.decorators import track_errors
from sandbox.projects.yabs.qa.mutable_parameters import MutableParameters
from sandbox.projects.yabs.qa.resource_types import (
    BaseBackupSdk2Resource,
    YabsResponseDumpUnpacker,
    YabsServerResponseDump,
    YABS_SERVER_REQUEST_LOG_GZ,
    YABS_SERVER_FT_DOLBILKA_PLAN,
    YABS_SERVER_TESTENV_DB_FLAGS,
)
from sandbox.projects.yabs.qa.response_tools.unpacker import (
    uc_compress,
    find_unpacker_resource,
    get_unpacker_executable_path,
)
from sandbox.projects.yabs.qa.utils.base import get_max_unpacking_workers
from sandbox.projects.yabs.qa.utils.general import (
    get_json_md5,
    get_resource_html_hyperlink,
    get_task_html_hyperlink,
    html_hyperlink,
    is_precommit_check,
)
from sandbox.projects.yabs.qa.utils.resource import json_from_resource
from sandbox.projects.yabs.qa.utils.task_hooks import get_shoot_requirements
from sandbox.projects.yabs.qa.utils.shoot import check_proto, get_original_request_id
from sandbox.projects.yabs.qa.tasks.duration_measure_task import BaseDurarionMeasureTask, BaseDurarionMeasureTaskParameters
from sandbox.projects.yabs.qa.tasks.YabsServerShmResourceGC import DEFAULT_HAMSTER_TTL
from sandbox.projects.yabs.qa.tasks.YabsServerShmResourceGC.hamster import ping_active_hamster_endpoints
from sandbox.projects.yabs.qa.tasks.YabsServerShmResourceGC.saas import ping_active_saas_freeze_data, SNAPSHOTS_TABLE_PROXY
from sandbox.projects.yabs.qa.tasks.YabsServerShmResourceGC.kvrs_saas import ping_active_kvrs_saas_freeze_data
from sandbox.projects.yabs.qa.bases.sample_tables.parameters import SamplingStrategyParameter
from sandbox.projects.yabs.qa.tasks.YabsServerUploadShootResultToYt import (
    YabsServerUploadShootResultToYt,
    YTUploadParameters,
    get_uc_executable_path
)
from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace, trace_calls, trace_entry_point
from sandbox.projects.yabs.sandbox_task_tracing.wrappers import subprocess
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.generic import enqueue_task, new_resource
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.projects.common.yql import run_query
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sdk2 import make_resource_ready, new_resource_data

from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShoot2.utils import (
    BAD_CHECK_STATES_DO_NOT_RESHOOT,
    dump_and_upload_coverage,
    generate_semaphores,
    update_shoot_statuses,
    dump_access_counters,
)


logger = logging.getLogger(__name__)


LoggedProcess = namedtuple('LoggedProcess', ['process', 'process_log_context'])
CountHttpRequest = namedtuple('CountHttpRequest', ['request_string', 'sort_key'])

FINALLY_BAD_CHECK_STATES = ['BadHTTP', 'BadProto', 'BadJSON', 'BadExts', 'NoYabsHTTP']
EDebugOutputMode = Enum(
    value='EDebugOutputMode',
    names=[
        ('json', 1),
        ('proto-binary', 2),
        ('proto_binary', 2)
    ]
)

QUERIES_FOR_LOG_CHECK = {
    "yabs": textwrap.dedent("""
        $all = (
            SELECT COUNT(*)
            FROM `{prefix}/primary_ammo/0/hit`
            WHERE `PageID` == 2
        );
        $success = (
            SELECT COUNT(*)
            FROM (
                SELECT
                    a.`BannerCount` AS BannerCount,
                    a.`PageID` AS PageID,
                    a.`RequestID` AS  RequestID
                FROM `{prefix}/primary_ammo/0/hit` as a
            )
            WHERE BannerCount != 0 AND PageID == 2
        );

        SELECT IF (CAST($success AS DOUBLE) / CAST($all AS DOUBLE) > 0.1, true, false);
    """),
}


class YabsBadRequestsID(BaseBackupSdk2Resource):
    """ Resource with list of bad requests id """
    auto_backup = True
    restart_policy = RestartPolicy.DELETE


class YabsServerExtTagsDump(BaseBackupSdk2Resource):
    """ Resource with list of failed tags per bad requests ids """
    auto_backup = True
    restart_policy = RestartPolicy.DELETE


class YabsServerShootTaskRequirements(sdk2.Parameters):
    with sdk2.parameters.Group('Requirements settings (use these instead of requirements tab!)') as requirements_settings:
        shard_space = sdk2.parameters.Integer(
            'Binary base space required for single shard (will account to either disk or ramdrive requirement), GB',
            default_value=200,
        )
        generic_disk_space = sdk2.parameters.Integer(
            'Generic disk space, GB',
            default_value=180
        )
        ram_space = sdk2.parameters.Integer(
            'Ram space, GB',
            default_value=240,
        )
        use_explicit_ramdrive_size = sdk2.parameters.Bool(
            'Use explicit ramdrive size',
            default_value=False,
        )


class YabsServerB2BFuncShoot2ParametersBase(sdk2.Task.Parameters):
    with sdk2.parameters.Group('Yabs-server module settings') as yabs_server_module_settings:
        server_module_parameters = YabsSUTParameters()
        meta_module_parameters = YabsMetaSandboxAdapter.get_init_parameters_class()()
        stat_module_parameters = YabsStatSandboxAdapter.get_init_parameters_class()()

    with sdk2.parameters.Group('Ammo generation module settings') as ammo_module_settings:
        ammo_module_parameters = AmmoRequestlogModuleYabsSpecificSandboxAdapter.get_init_parameters_class()()

    with sdk2.parameters.Group('Shoot module settings') as shoot_module_settings:
        shoot_module_parameters = DolbiloModuleSandboxAdapter.get_init_parameters_class()()

    with sdk2.parameters.Group('Primary ammo validation settings') as primary_ammo_validation_settings:
        ignored_ext_tags = sdk2.parameters.List(
            'Ignored ext tags',
            description='Disable response status check for following tags',
            default=['pcode_renderer'],
        )
        maximum_bad_response_ratio = sdk2.parameters.Float(
            'Maximum bad response ratio',
            default_value=0.1,
        )

    with sdk2.parameters.Group('Secondary count links settings') as general_group:
        shoot_count_links = sdk2.parameters.Bool(
            'Enable secondary count links shoot sessions',
            default_value=True,
        )
        with shoot_count_links.value[True]:
            all_count_links = sdk2.parameters.Bool(
                'Add click count links to shoot',
                default_value=True,
            )
            secondary_count_links_limit = sdk2.parameters.Integer(
                'Maximum number of secondary count links to shoot',
                default=700000,  # TODO BSSERVER-8932
            )
            with sdk2.parameters.Group('Secondary ammo validation settings') as secondary_ammo_validation_settings:
                ignored_secondary_count_links_ext_tags = sdk2.parameters.List(
                    'Ignored ext tags in secondary count links shoot',
                    default=['bigb_rf_balancer'],
                )
                maximum_secondary_count_links_bad_response_ratio = sdk2.parameters.Float(
                    'Maximum secondary count links bad response ratio',
                    default=0.3,
                )

    with sdk2.parameters.Group('Override parameters') as override_parameters:
        update_parameters_resource = sdk2.parameters.Resource(
            'Resource with JSON-dumped parameter update dict',
            resource_type=YABS_SERVER_TESTENV_DB_FLAGS,
        )
        update_parameters_resources = sdk2.parameters.Resource(
            'Resources with JSON-dumped parameter update dict (new version)',
            resource_type=YABS_SERVER_TESTENV_DB_FLAGS,
            multiple=True,
        )

    with sdk2.parameters.Group('Additional info') as info:
        sampling_strategy = SamplingStrategyParameter('Data sampling strategy')


class YabsServerB2BFuncShoot2Parameters(YabsServerB2BFuncShoot2ParametersBase):
    """
    This inheritance is needed to override parameters, that come from other classes.
    Like headers_update_dict from AmmoRequestlogYabsSpecificParameters, which default we need to override.
    """
    headers_update_dict = YabsServerB2BFuncShoot2ParametersBase.headers_update_dict(
        default_value={
            "x-yabs-debug-options-json": json.dumps({
                "business": False,
                "trafaret": False,
                "logs": True,
                "trace": False,
                "keywords": False,
                "mx_zero_features": False,
                "ext_http_entities": True,
                "mx": False,
                "exts": True,
                "debug_log": False,
                # Extra fields
                "match_log": False,
                "filter_log": False,
                "force_event_log": False,  # since we extract count links manually and need valid money - we don't need unnecessary event logs
                "include_binary_ext_http_requests_override": ["bscount", "bscount_fbs2"],
            })
        }
    )


class YabsServerB2BFuncShoot2(BaseBinTaskMixin, BaseDurarionMeasureTask, sdk2.Task):

    '''
    New task for functional b2b tests of yabs-server service.
    '''

    name = 'YABS_SERVER_B2B_FUNC_SHOOT_2'

    class Parameters(YabsServerB2BFuncShoot2Parameters):

        tokens = sdk2.parameters.YavSecret("YAV secret identifier", default="sec-01d6apzcex5fpzs5fcw1pxsfd5")

        _base_bin_task_parameters = base_bin_task_parameters(
            release_version_default=ReleaseStatus.STABLE,
            resource_attrs_default={"task_bundle": "yabs_server_shoot"},
        )

        requirements = YabsServerShootTaskRequirements()

        with sdk2.parameters.Group('Secrets') as secrets:
            yql_token_vault_name = sdk2.parameters.String(
                'YQL token vault name',
                default='yabs-cs-sb-yql-token',
            )

        with sdk2.parameters.Group('Misc settings'):
            reshoot_threads = sdk2.parameters.JSON(
                'Reshoot threads',
                default=[4] * 3,
            )
            dump_parser_resource = sdk2.parameters.Resource(
                'Dump parser resource',
                resource_type=YabsResponseDumpUnpacker,
            )
            response_dumps_ttl = sdk2.parameters.Integer(
                'TTL for YABS_SERVER_RESPONSES_DUMP resource, in days',
                default=YabsServerResponseDump.ttl,
            )
            use_base_state_generator = sdk2.parameters.Bool(
                'Unpack bases before init modules',
                default_value=True
            )
            do_collect_coverage = sdk2.parameters.Bool('Collect gcov coverage, requires gcov build', default=False)
            run_only_in_precommit_checks = sdk2.parameters.Bool('Run task only in precommit checks, otherwise exit immediately', default=False)
            allow_reuse_of_this_task = sdk2.parameters.Bool('Allow other tasks with same parameters reuse results of this task', default=False)
            get_access_counters = sdk2.parameters.Bool('Load access counters, requires build with \'struct_access_counter\' codegen', default=False)
            use_latest_hamsters = sdk2.parameters.Bool(
                "Use latest hamster resources",
                description="Get OK hamster resources from testenv",
                default=False,
            )

        yt_upload_parameters = YTUploadParameters()

        with sdk2.parameters.Output:
            merged_input_parameters = sdk2.parameters.JSON(
                'Merged input parameters',
                default={},
            )
            uploaded_logs_to_yt_prefix = sdk2.parameters.String('YT path with uploaded logs')
            upload_to_yt_task_id = sdk2.parameters.Integer('Upload task id')
            shoot_statuses = sdk2.parameters.JSON('Statuses of each shooting', default={})
            final_statuses = sdk2.parameters.JSON('Statuses of primary shooting', default={})

            input_parameters_hash = sdk2.parameters.String('Input parameters hash to reuse shoot results in stability runs')

        duration_parameters = BaseDurarionMeasureTaskParameters()

    class Requirements(sdk2.Task.Requirements):
        client_tags = (ctc.Tag.LINUX_XENIAL | ctc.Tag.YABS) & ctc.Tag.SSD
        environments = [
            environments.PipEnvironment('yql', version='1.2.91'),
            environments.PipEnvironment('yandex-yt'),
        ]

    def on_enqueue(self):
        super(YabsServerB2BFuncShoot2, self).on_enqueue()

        if self.Context.shoot_finished or (self.Parameters.run_only_in_precommit_checks and not is_precommit_check(self)):
            self.Requirements.client_tags = ctc.Tag.LINUX_XENIAL
            return

        shoot_requirements = get_shoot_requirements(self.server, self.Parameters)
        self.Context.shoot_requirements = dict(shoot_requirements.__dict__)

        self.Requirements.ram = shoot_requirements.ram
        self.Requirements.ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, shoot_requirements.ramdrive_size, None)
        self.Requirements.disk_space = shoot_requirements.disk_space

        self.Context.unpacking_workers = get_max_unpacking_workers(shoot_requirements.bin_bases_unpacked_size, shoot_requirements.ram, shoot_requirements.ramdrive_size)

        semaphores = generate_semaphores(self.enabled_hamsters)
        if semaphores:
            self.Requirements.semaphores = ctt.Semaphores(
                acquires=semaphores,
                release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH, ctt.Status.Group.WAIT)
            )

        with self.memoize_stage.calc_input_parameters_hash:
            try:
                self.Parameters.input_parameters_hash = calc_input_parameters_hash(self.server, self.id)
            except Exception:
                ex_type, ex, tb = sys.exc_info()
                self.Context.calc_parameters_hash_exception = ''.join(traceback.format_exception(ex_type, ex, tb))

    '''
    Properties
    '''

    @property
    def dump_parser_resource(self):
        if self.MutableParameters.dump_parser_resource:
            dump_parser_resource = self.MutableParameters.dump_parser_resource
        else:
            dump_parser_resource = find_unpacker_resource(
                global_key=self.MutableParameters.meta_server_resource.global_key,
                global_key_type=self.MutableParameters.meta_server_resource.global_key_type,
                build_mode="release",
            )
        return dump_parser_resource

    @property
    def dump_parser_path(self):
        return get_unpacker_executable_path(self.dump_parser_resource)

    enabled_hamsters = property(shoot_task_enabled_hamsters)

    '''
    Getters
    '''
    def get_base_yt_path(self):
        if self.MutableParameters.upload_to_yt_prefix:
            return '{}/{}'.format(self.MutableParameters.upload_to_yt_prefix.rstrip('/'), self.Context.upload_to_yt_task_id)
        else:
            raise TaskFailure('YT prefix requested but upload to YT was not (probable task bug)')

    @property
    def uc_executable_path(self):
        if not hasattr(self, '_uc_executable_path'):
            self._uc_executable_path = get_uc_executable_path()
        return self._uc_executable_path

    '''
    Methods for outgoing data publication
    '''

    @trace_calls
    def prepare_dplan_resource(self, dplan_path, shoot_index, content_type):
        dplan_resource = new_resource(
            YABS_SERVER_FT_DOLBILKA_PLAN,
            self,
            'Dplan resource',
            dplan_path,
            shoot_index=shoot_index,
            content_type=content_type,
            cachedaemon_dump_res_id=self.MutableParameters.cache_daemon_stub_resource.id,
        )
        self.upload_futures.append(self.thread_pool.submit(make_resource_ready, dplan_resource))

    @trace_calls
    def prepare_dump_resource(self, dump_path, shoot_index, content_type, base_revision, pack_type='lz4hc', debug_cookie=DEBUG_COOKIE):
        compressed_dump_path = uc_compress(self, dump_path, pack_type=pack_type)
        shoot_reports_yt_node = posixpath.join(self.Parameters.upload_to_yt_prefix, str(self.id)) if self.Parameters.upload_to_yt_prefix else ''
        dump_resource = new_resource(
            YabsServerResponseDump,
            self,
            'Dump resource',
            compressed_dump_path,
            shoot_index=shoot_index,
            pack_codec=pack_type,
            content_type=content_type,
            debug_cookie=debug_cookie,
            debug_mode='proto_binary',
            dump_parser_id=self.dump_parser_resource.id,
            shoot_reports_yt_node=shoot_reports_yt_node,
            ammo_type=self.MutableParameters.meta_mode,
            ttl=self.Parameters.response_dumps_ttl,
        )
        self.upload_futures.append(self.thread_pool.submit(make_resource_ready, dump_resource))

    @trace_calls
    def prepare_ammo_resource(self, ammo_path, shoot_index, content_type):
        gzipped_ammo_path = pack_utils.gzip_file(ammo_path)
        ammo_resource = new_resource(
            YABS_SERVER_REQUEST_LOG_GZ,
            self,
            'Ammo resource',
            gzipped_ammo_path,
            shoot_index=shoot_index,
            content_type=content_type,
            cachedaemon_dump_res_id=self.MutableParameters.cache_daemon_stub_resource.id,
        )
        self.upload_futures.append(self.thread_pool.submit(make_resource_ready, ammo_resource))

    @trace_calls
    def prepare_bad_requests_dump(self, bad_requests_path, ext_tags_path):
        make_resource_ready(new_resource(YabsBadRequestsID, self, 'List of bad RequestIDs', bad_requests_path))
        if os.path.exists(ext_tags_path):
            make_resource_ready(new_resource(YabsServerExtTagsDump, self, 'List of failed tags per bad RequestIDs', ext_tags_path))
        else:
            logging.debug("File with ext failed tags doesn't exist")

    '''
    Methods concerning secondary count links shoot session
    '''

    @trace_calls
    def generate_count_links_ammo(self, dump_path):
        count_links_ammo_path = str(uuid.uuid4()) + '.ammo'
        cmdline = [
            self.dump_parser_path,
            '--shoot-count-proto',
            '-r', count_links_ammo_path,
            dump_path,
            '-j', str(multiprocessing.cpu_count()),
        ]
        if self.MutableParameters.all_count_links:
            cmdline.append('--all-count-links')
        with sdk2.helpers.ProcessLog(self, 'generate_count_ammo') as process_log_context:
            process_log_context.logger.propagate = True
            subprocess.check_call(cmdline, stdout=process_log_context.stdout, stderr=subprocess.STDOUT)
        return count_links_ammo_path

    @trace_calls
    def shoot_with_reshoot(self, ammo_module, shoot_module, server, filtered_ext_tags, content_type):
        result_dump_paths = []
        for shoot_index in range(len(self.MutableParameters.reshoot_threads) + 1):
            is_last = shoot_index == len(self.MutableParameters.reshoot_threads)
            dump_path, bad_requests_path, statuses, need_reshoot = self.shoot_and_check_proto(
                ammo_module, shoot_module, server,
                filtered_ext_tags,
                content_type=content_type,
                shoot_index=shoot_index,
                is_last=is_last,
            )
            result_dump_paths.append(dump_path)
            if not need_reshoot:
                break
        return result_dump_paths, bad_requests_path, statuses

    @trace_calls
    def shoot_and_check_proto(self, ammo_module, shoot_module, server, ext_tags, content_type, shoot_index=0, is_last=False):
        stage_info = dict(content_type=content_type, shoot_index=shoot_index)
        hamster_ext_service_tags = self.MutableParameters.hamster_ext_service_tags
        if self.MutableParameters.use_separate_linear_models_service and self.MutableParameters.linear_models_binary_resource:
            hamster_ext_service_tags.extend(list(YabsLinearModels.service_tags))

        with self.stage_duration("shoot_{content_type}_{shoot_index}".format(**stage_info)), trace('shoot', info=stage_info):
            dump_path = self.shoot(
                ammo_module, shoot_module, server,
                content_type=content_type,
                shoot_index=shoot_index,
            )

            bad_requests_path, bad_response_parser_output, statuses_file = check_proto(
                self.dump_parser_path,
                dump_path,
                ext_tags,
                hamster_ext_service_tags=hamster_ext_service_tags,
            )

        statuses = self.get_bad_requests_statuses(
            bad_requests_path,
            bad_response_parser_output,
            statuses_file,
            BAD_CHECK_STATES_DO_NOT_RESHOOT + (FINALLY_BAD_CHECK_STATES if is_last else [])
        )
        self.shoot_statuses.setdefault(content_type, []).append(statuses)

        need_reshoot = True
        if content_type == 'primary_ammo':
            self.final_statuses = update_shoot_statuses(self.final_statuses, statuses)
        if not is_last:
            if not ammo_module.update_ammo(bad_requests_path, content_type=content_type):
                logging.info('Finished {} reshoot session on attempt {}'.format(content_type, shoot_index))
                statuses = self.get_bad_requests_statuses(bad_requests_path, bad_response_parser_output, statuses_file, FINALLY_BAD_CHECK_STATES)
                need_reshoot = False
        else:
            need_reshoot = False

        return dump_path, bad_requests_path, statuses, need_reshoot

    @trace_calls
    def merge_count_ammo(self, count_ammo_paths):
        request_id_header_pattern = re.compile(r'x-yabs-request-id: (\d+)', re.IGNORECASE)

        added_original_request_ids = set()
        request_list = []
        for count_ammo_path in reversed(count_ammo_paths):
            current_dump_original_request_ids = set()
            with open(count_ammo_path) as count_ammo_file:
                for request, _ in requestlog.iterate(count_ammo_file, header_separator='\t| '):
                    request_id = None
                    request_id_match = request_id_header_pattern.search(request)
                    if request_id_match:
                        request_id = request_id_match.group(1)

                    original_request_id = get_original_request_id(request_id)
                    if not original_request_id:
                        logging.debug('Failed to get original request id from request:\n%s', request)
                        continue

                    if original_request_id in added_original_request_ids:
                        continue
                    current_dump_original_request_ids.add(original_request_id)

                    request_list.append(CountHttpRequest(request, int(request_id)))

            added_original_request_ids |= current_dump_original_request_ids

        request_list.sort(key=lambda item: item.sort_key)

        merged_count_ammo_path = str(uuid.uuid4()) + '.ammo'
        with open(merged_count_ammo_path, 'w') as merged_count_ammo_file:
            for request in request_list:
                requestlog.write_stpd(merged_count_ammo_file, request.request_string, '<timestamp>', '<tag>')
        return merged_count_ammo_path

    '''
    Methods concerning reshoot sessions
    '''

    @trace_calls
    def get_bad_requests_statuses(self, bad_requests_path, bad_response_parser_output, statuses_file, bad_states_to_count):
        if self.total_responses_count is None:
            self.total_responses_count = self.get_total_count(bad_response_parser_output)
        self.total_bad_responses_count += sum(self.get_status_count(bad_response_parser_output, status) for status in bad_states_to_count)
        logging.info(
            'Total unrecoverable bad responses: %d, limit %d',
            self.total_bad_responses_count,
            self.total_responses_count * self.bad_response_ratio
        )
        if self.total_bad_responses_count >= self.total_responses_count * self.bad_response_ratio and \
                self.total_responses_count * self.bad_response_ratio > 0:
            self.prepare_bad_requests_dump(bad_requests_path, bad_requests_path + '_tags.txt')
            raise TaskFailure(
                'Bad response status check failed: bad_responses_count = %s, total = %s' %
                (self.total_bad_responses_count, self.total_responses_count)
            )
        with open(statuses_file) as f:
            statuses = json.load(f)

        return statuses

    def get_status_count(self, output, status):
        match = re.search(r'Status: {status}\s+Count: (\d+)'.format(status=status), output)
        return int(match.group(1)) if match else 0

    def get_total_count(self, output):
        match = re.search(r'Total:\s+(\d+)', output)
        return int(match.group(1))

    '''
    Main execution sequence methods
    '''

    def init_parameters(self):
        custom_fields = self.server.task[self.id].custom.fields.read()
        custom_fields_dict = {item['name']: item for item in custom_fields}
        input_parameters_dict = {item['name']: item['value'] for item in custom_fields if not item['output']}
        for update_parameters_resource in (self.Parameters.update_parameters_resources or []):
            update_parameters_resource_path = str(new_resource_data(update_parameters_resource).path)
            with open(update_parameters_resource_path, 'r') as f_out:
                update_parameters_dict = json.load(f_out)
            input_parameters_dict.update(update_parameters_dict)
        input_parameters_dict = truncate_output_parameters(input_parameters_dict, type(self).Parameters)
        self.MutableParameters = MutableParameters.__from_dict__(input_parameters_dict)
        try:
            self.Parameters.merged_input_parameters = self.MutableParameters.__dict__
        except Exception:
            logging.exception('Failed to set output parameter (likely set in a previous run')
        for key, value in self.MutableParameters.__dict__.items():
            if key in custom_fields_dict and custom_fields_dict[key]['type'] == 'resource':
                if value is None:
                    continue
                if isinstance(value, int):
                    self.MutableParameters.__dict__[key] = sdk2.Resource[value]
                elif isinstance(value, list):
                    self.MutableParameters.__dict__[key] = [sdk2.Resource[res_id] for res_id in value]
                else:
                    raise ValueError('Failed to cast value {} to sdk2.Resource'.format(value))

    def on_create(self):
        if self.Context.copy_of:
            merged_input_parameters = sdk2.Task[self.Context.copy_of].Parameters.merged_input_parameters
            for key, value in merged_input_parameters.iteritems():
                try:
                    setattr(self.Parameters, key, value)
                except AttributeError:
                    pass
            self.Parameters.update_parameters_resources = None

    @trace_calls
    def ping_resources_in_use(self):
        from yt.wrapper import YtClient
        yt_token = try_get_from_vault(self, 'yt_token_for_report_upload')

        yt_client_hamster_endpoints = YtClient(proxy=ENDPOINTS_TABLE_PROXY, token=yt_token)
        yt_client_saas_shapshots = yt_client_hamster_endpoints if ENDPOINTS_TABLE_PROXY == SNAPSHOTS_TABLE_PROXY else YtClient(proxy=SNAPSHOTS_TABLE_PROXY, token=yt_token)

        now = int(time.time())

        failures = []
        if self.Parameters.ext_service_endpoint_resources:
            try:
                active_endpoint_resources = {
                    (endpoint_resource_id, service_tag): json_from_resource(endpoint_resource_id)
                    for service_tag, endpoint_resource_id in self.enabled_hamsters.items()
                }
                logger.debug("Active hamster endpoints: %s", active_endpoint_resources)
                ping_active_hamster_endpoints(yt_client_hamster_endpoints, active_endpoint_resources, now,
                                              DEFAULT_HAMSTER_TTL)
            except Exception:
                logging.error('Got exception while trying to ping hamster_endpoints', exc_info=True)
                failures.append('hamster_endpoints')
        if self.Parameters.saas_freeze_data_resource:
            try:
                ping_active_saas_freeze_data(yt_client_saas_shapshots, [self.Parameters.saas_freeze_data_resource], now)
            except Exception:
                logging.error('Got exception while trying to ping saas_freeze_data', exc_info=True)
                failures.append('saas_freeze_data')
        if self.Parameters.saas_frozen_topology:
            try:
                ping_active_kvrs_saas_freeze_data(yt_client_saas_shapshots, [self.Parameters.saas_frozen_topology.id], now)
            except Exception:
                logging.error('Got exception while trying to ping saas_frozen_topology', exc_info=True)
                failures.append('saas_frozen_topology')
        if failures:
            self.set_info('Failed to ping resources in use: [{}], see logs for more info'.format(', '.join(failures)))

    @track_errors
    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        if self.Parameters.run_only_in_precommit_checks and not is_precommit_check(self):
            self.set_info("Task execution skipped because it executes only on precommit checks")
            return

        input_parameters_hash = calc_input_parameters_hash(self.server, self.id)
        logger.debug('Input parameters hash is %s', input_parameters_hash)

        profiler = Profile()
        profiler.enable()
        self.init_parameters()
        self.ping_resources_in_use()

        with self.memoize_stage.set_input_archive_info, trace('set_input_archive_info'):
            base_resource_id = [
                resource_id
                for resource_id in set(self.Parameters.meta_binary_base_resources + self.Parameters.stat_binary_base_resources)
                if hasattr(sdk2.Resource[resource_id], 'input_spec')
            ][0]
            base_resource = sdk2.Resource[base_resource_id]

            input_spec_resource_id = base_resource.input_spec
            input_spec_resource_data = new_resource_data(sdk2.Resource[input_spec_resource_id])
            with open(str(input_spec_resource_data.path), 'r') as input_spec_file:
                input_spec_data = json.load(input_spec_file)
                input_archive_root_yt_path = input_spec_data['__SANDBOX_AUXILLARY_DATA__']['archive_root']

            input_archive_yt_path = posixpath.join(input_archive_root_yt_path, 'yt_tables')
            if getattr(base_resource, 'sampling_strategy') == 'sampled':
                input_archive_yt_path = posixpath.join(input_archive_root_yt_path, 'shallow_copy', base_resource.sampling_parameters_hash, 'yt_tables')

            self.set_info(
                'Input tables for binary bases are <a href="https://yt.yandex-team.ru/hahn/navigation?path={yt_path}" target="_blank">here</a>'.format(
                    yt_path=input_archive_yt_path,
                ),
                do_escape=False,
            )

        with self.memoize_stage.find_active_hamsters(commit_on_entrance=False):
            if self.Parameters.use_latest_hamsters:
                ok_testenv_hamster_endpoints = get_ok_testenv_hamster_endpoints()
                self.MutableParameters.ext_service_endpoint_resources = [
                    ok_testenv_hamster_endpoints[service_tag][0]
                    for service_tag, _ in self.enabled_hamsters.items()
                ]
                self.set_info(
                    "Updated hamster resources: {}".format(
                        self.MutableParameters.ext_service_endpoint_resources))

        with self.memoize_stage.validate_hamsters(commit_on_entrance=False):
            secrets = self.Parameters.tokens.data()
            not_ready_hamsters = get_bad_hamsters(
                self.id, secrets["nanny_token"], secrets["yp_token"],
                task_parameters=self.MutableParameters
            )
            error_message = ""
            for service_tag, resource_id in not_ready_hamsters.items():
                error_message += "  {hamster_link} {resource_link}\n".format(
                    hamster_link=html_hyperlink(get_hamster_url(resource_id), service_tag),
                    resource_link=get_resource_html_hyperlink(resource_id),
                )

            if error_message:
                error_message = "Not ready hamsters:\n" + error_message

                if self.Context.copy_of:
                    error_message += "\nMost likely you have cloned a task that is too old. Try to clone newer one.\n"

                error_message += \
                    "\nTo use latest hamsters clone task and enable 'Use latest hamster resources' parameter.\n"

                self.set_info(error_message, do_escape=False)
                raise TaskFailure("Hamsters validation failed")

        with self.memoize_stage.shoot(commit_on_entrance=False), trace('shoot'):
            self.shoot_threads = [self.MutableParameters.shoot_threads] + self.MutableParameters.reshoot_threads
            logging.debug('Threads per shoot - %s', self.shoot_threads)

            import concurrent.futures
            self.thread_pool = concurrent.futures.ThreadPoolExecutor(max_workers=len(self.shoot_threads) * 3)
            self.upload_futures = []

            self.shoot_statuses = {}
            try:
                self.total_bad_responses_count = 0
                self.total_responses_count = None
                self.final_statuses = {}
                self.bad_response_ratio = self.MutableParameters.maximum_bad_response_ratio
                with self.stage_duration("create_modules"), trace("create_modules"):
                    stat_adapter = YabsStatSandboxAdapter(
                        self.MutableParameters,
                        task_instance=self,
                        work_dir="stat_adapter",
                    )
                    meta_adapter = YabsMetaSandboxAdapter(
                        self.MutableParameters,
                        task_instance=self,
                        work_dir="meta_adapter",
                    )
                    base_state_adapter = BaseStateSandboxAdapter(
                        self.MutableParameters,
                        task_instance=self,
                        base_dir=stat_adapter.get_base_dir(),
                        transport_resource_path=stat_adapter.get_transport_resource_path(),
                    )

                    base_state = None
                    if self.MutableParameters.use_base_state_generator:
                        base_state_generator = base_state_adapter.create_module(
                            unpack_workers=self.Context.unpacking_workers,
                        )
                        base_state = base_state_generator.get_bases(set(self.MutableParameters.stat_binary_base_resources + self.MutableParameters.meta_binary_base_resources))

                    stubs_cachedaemon_data_dir = get_unique_file_name(os.path.join(str(self.ramdrive.path), 'stubs'), 'cache_daemon_data')
                    cachedaemon = stat_adapter.create_cachedaemon(
                        cachedaemon_data_dir=stubs_cachedaemon_data_dir,
                        threads_per_service={'bscount': self.Parameters.shoot_threads * len(self.Parameters.stat_shards)},
                    )
                    stat_module = stat_adapter.create_module(
                        cachedaemon=cachedaemon,
                        shared_base_state=base_state,
                        max_unpacking_workers=self.Context.unpacking_workers
                    )
                    meta_module = meta_adapter.create_module(
                        cachedaemon=cachedaemon,
                        stat_instances=stat_module.get_instances(),
                        shared_base_state=stat_module.shared_base_state,
                        max_unpacking_workers=self.Context.unpacking_workers
                    )

                if self.Parameters.use_default_debug_cookie:
                    debug_cookie = DEBUG_COOKIE
                else:
                    cookies = meta_module.get_debug_cookies()
                    logging.info('Got debug cookies: %s', cookies)
                    debug_cookie = cookies[0]

                with self.stage_duration("ammo"), trace("ammo"):
                    ammo_adapter = AmmoRequestlogModuleYabsSpecificSandboxAdapter(self.MutableParameters, self, debug_cookie=debug_cookie)
                    ammo_module = ammo_adapter.create_module()
                shoot_adapter = DolbiloModuleSandboxAdapter(self.MutableParameters, self)
                shoot_module = shoot_adapter.create_module()

                ext_tags = meta_module.get_ext_tag_set().union(stat_module.get_ext_tag_set())
                filtered_primary_ext_tags = ext_tags - set(self.Parameters.ignored_ext_tags)
                filtered_secondary_ext_tags = ext_tags - set(self.Parameters.ignored_secondary_count_links_ext_tags)

                with cachedaemon, stat_module as stat, meta_module as server:
                    primary_dump_paths, bad_requests_path, statuses = self.shoot_with_reshoot(
                        ammo_module, shoot_module, server,
                        filtered_ext_tags=filtered_primary_ext_tags,
                        content_type='primary_ammo',
                    )
                    with self.memoize_stage.set_final_statuses, trace('set_final_statuses'):
                        self.Parameters.final_statuses = json.dumps(self.final_statuses)
                    if self.MutableParameters.shoot_count_links and self.MutableParameters.meta_mode != 'bsrank':
                        count_ammo_paths = map(self.generate_count_links_ammo, primary_dump_paths)

                        with self.stage_duration("secondary_ammo"), trace("secondary_ammo"):
                            ammo_module.reload_ammo(self.merge_count_ammo(count_ammo_paths), EAmmoFileType.plain, update_ammo=True, content_type='secondary_count_links')

                        shoot_module.adapter.parameters.shoot_request_limit = self.Parameters.secondary_count_links_limit
                        self.total_responses_count = None
                        self.total_bad_responses_count = 0
                        self.bad_response_ratio = self.MutableParameters.maximum_secondary_count_links_bad_response_ratio

                        _, count_bad_requests_path, _ = self.shoot_with_reshoot(
                            ammo_module, shoot_module, server,
                            filtered_ext_tags=filtered_secondary_ext_tags,
                            content_type='secondary_count_links',
                        )
                        with open(bad_requests_path, 'a') as f_out, open(count_bad_requests_path) as f_in:
                            shutil.copyfileobj(f_in, f_out)
                        if os.path.exists(count_bad_requests_path + '_tags.txt'):
                            with open(bad_requests_path + '_tags.txt', 'a') as f_out, open(count_bad_requests_path + '_tags.txt') as f_in:
                                shutil.copyfileobj(f_in, f_out)
                    if self.Parameters.do_collect_coverage:
                        dump_and_upload_coverage(self)

                    servers = list(stat_module._stats.values()) + [meta_module.get_server_backend_object()]

                    solomon_stats_dir = get_unique_file_name(meta_adapter.get_logs_dir(), "solomon_stats")
                    make_folder(solomon_stats_dir)
                    for server in servers:
                        dump_solomon_stats(server, solomon_stats_dir)

                    if self.Parameters.saas_frozen_topology:
                        for server in servers:
                            self.set_info(
                                create_saas_solomon_stats_validation_report(
                                    server.name,
                                    validate_saas_solomon_stats(server.name, solomon_stats_dir)
                                )
                            )

                    if self.Parameters.get_access_counters:
                        access_counters_dir = get_unique_file_name(meta_adapter.get_logs_dir(), "access_counters")
                        make_folder(access_counters_dir)
                        for server in servers:
                            dump_access_counters(server, access_counters_dir)

                self.Context.shoot_finished = True

                self.prepare_bad_requests_dump(bad_requests_path, bad_requests_path + "_tags.txt")
                if self.Parameters.meta_store_access_log:
                    meta_module._meta.check_access_log(
                        ext_sharded={},
                        quantiles={90, 99},
                        error_rate_thr=1.,
                        request_error_rate_thr=1.,
                        ext_error_rate_thr=1.,
                    )
                if self.Parameters.stat_store_access_log:
                    for stat in stat_module._stats.values():
                        stat.check_access_log(
                            ext_sharded={},
                            quantiles={90, 99},
                            error_rate_thr=1.,
                            request_error_rate_thr=1.,
                            ext_error_rate_thr=1.,
                        )
            finally:
                concurrent.futures.wait(self.upload_futures)
                if self.MutableParameters.upload_to_yt_prefix:
                    self.Context.upload_to_yt_task_id = enqueue_task(YabsServerUploadShootResultToYt(
                        self,
                        description='Upload shoot result of task {}'.format(get_task_html_hyperlink(self.id)),
                        tags=self.Parameters.tags,
                        owner=self.owner,
                        shoot_task=self.id,
                        dump_parser_resource=self.dump_parser_resource.id,
                        upload_to_yt_prefix=self.MutableParameters.upload_to_yt_prefix,
                        write_ext_responses=self.MutableParameters.write_ext_responses,
                        write_ext_requests=self.MutableParameters.write_ext_requests,
                        yt_write_response_headers_content_types=self.MutableParameters.yt_write_response_headers_content_types,
                        yt_node_ttl=self.Parameters.yt_node_ttl,
                    )).id
                    self.set_info('Run task {}'.format(get_task_html_hyperlink(self.Context.upload_to_yt_task_id)), do_escape=False)
                    self.set_info('Shoot results will be available at {} for {} days as soon as task {} will finish upload'.format(
                        html_hyperlink(
                            link='https://yt.yandex-team.ru/hahn/?page=navigation&path={}'.format(self.get_base_yt_path()),
                            text='link',
                        ),
                        self.Parameters.yt_node_ttl,
                        get_task_html_hyperlink(self.Context.upload_to_yt_task_id)
                    ), do_escape=False)
                    with self.memoize_stage.set_yt_upload_output:
                        self.Parameters.uploaded_logs_to_yt_prefix = self.get_base_yt_path()
                        self.Parameters.upload_to_yt_task_id = self.Context.upload_to_yt_task_id

            profiler.disable()
            with open(os.path.join(str(self.log_path()), 'profile.log'), 'w') as f:
                profile_stats = Stats(profiler, stream=f)
                profile_stats.sort_stats('cumulative', 'ncalls').print_stats()

            with self.memoize_stage.set_shoot_statuses:
                self.Parameters.shoot_statuses = json.dumps(self.shoot_statuses, sort_keys=True, indent=4)

        with self.memoize_stage.check_logs(commit_on_entrance=False, commit_on_wait=False), trace('check_logs'):
            if self.Context.testenv_resource_check:
                check_tasks(self, self.Context.upload_to_yt_task_id)
                self.check_correctness_of_logs_with_yql()

    def get_table_size(self, table):
        row_count = len(table.rows)
        if row_count > 0:
            col_count = len(table.rows[0])
        else:
            col_count = 0
        return row_count, col_count

    @trace_calls
    def check_correctness_of_logs_with_yql(self):
        try:
            query = QUERIES_FOR_LOG_CHECK[self.MutableParameters.meta_mode]
        except KeyError:
            logging.info("No log validation for role %s", self.MutableParameters.meta_mode)
            return

        prefix = self.get_base_yt_path()
        query = query.format(prefix=prefix)
        self.set_info('YQL query for output log check: {}'.format(query))
        yql_token = try_get_from_vault(self, self.Parameters.yql_token_vault_name)

        request = run_query(
            query_id='check correctness of output logs',
            query=query,
            yql_token=yql_token,
            db='hahn',
            wait=False,
            syntax_version=1)
        self.set_info(html_hyperlink(link=request.share_url, text='Responses validation YQL operation'), do_escape=False)

        for table in request.get_results(wait=True):
            table.fetch_full_data()
            row_count, col_count = self.get_table_size(table)
            if (row_count != 1) or (col_count != 1):
                raise TaskError(
                    'Wrong output format of query_for_log_check (YQL query for output log check).\n' +
                    'Expected: rows = 1, cols = 1\n' +
                    'Actual: rows = {}, cols = {}'.format(row_count, col_count)
                )

            if bool(table.rows[0][0]) is False:
                self.set_info('YQL query for output log check has returned false.')
                raise TaskFailure('YQL query for output log check has failed.')

    @trace_calls
    def shoot(self, ammo_module, shoot_module, server, content_type, base_revision=0, shoot_index=0, debug_cookie=DEBUG_COOKIE):
        self.prepare_ammo_resource(
            ammo_module.get_ammo_path(),
            shoot_index=shoot_index,
            content_type=content_type,
        )
        self.prepare_dplan_resource(
            ammo_module.get_dplan_path(),
            shoot_index=shoot_index,
            content_type=content_type,
        )
        dplan_path = ammo_module.get_dplan_path()
        dump_path = shoot_module.shoot_and_watch(
            server,
            dplan_path,
            shoot_threads=self.shoot_threads[shoot_index],
        )
        self.prepare_dump_resource(
            dump_path,
            shoot_index=shoot_index,
            content_type=content_type,
            base_revision=base_revision,
            debug_cookie=debug_cookie,
        )
        return dump_path


def calc_input_parameters_hash(rest_client, task_id, parameters_class=YabsServerB2BFuncShoot2Parameters):
    custom_fields = rest_client.task[task_id].custom.fields.read()
    raw_input_parameters = {item['name']: item['value'] for item in custom_fields if not item['output']}
    hashable_parameters = {}
    for parameter_name, parameter_value in raw_input_parameters.items():
        if not hasattr(parameters_class, parameter_name):
            continue
        if isinstance(parameter_value, list):
            hashable_parameters[parameter_name] = list(sorted(parameter_value))
        else:
            hashable_parameters[parameter_name] = parameter_value

    logger.info('Hashable parameters are: %s', json.dumps(hashable_parameters, indent=2, sort_keys=True))

    return get_json_md5(hashable_parameters)
