import copy
import logging
import subprocess
import time
import requests
import os
import json

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.projects.yabs.qa.hamster.config import CAPACITY as HAMSTER_CAPACITY
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_GCDA_COVERAGE


BAD_CHECK_STATES_DO_NOT_RESHOOT = ['BadYabsHTTPCode', 'BadPhantomHTTPCode']


def update_shoot_statuses(final_statuses, statuses):
    res = copy.deepcopy(final_statuses)
    for key, value in statuses.items():
        if key == "OK" or key in BAD_CHECK_STATES_DO_NOT_RESHOOT:
            res[key] = res.setdefault(key, 0) + value
        else:
            res[key] = value
    return res


COVERAGE_FILENAME = "yabs_gcdas.tar.gz"


def dump_and_upload_coverage(task):
    logging.info("Dumping coverage...")
    subprocess.check_output(["pkill", "-USR1", "yabs-server"])
    time.sleep(10)
    subprocess.check_output([
        "tar",
        "zcf",
        COVERAGE_FILENAME,
        "-C", "gcda-root/",
        "--transform", "s#place.*"
                       "("
                       "(build_root/.{4}/[0-9a-f]{6})"  # local build, looks like /place/sandbox-data/build_cache/yb/build_root/qzws/000427/ads/... # noqa
                       "|"
                       "(-[0-9a-f]+/[0-9a-f]{2}/[^/]+/[0-9a-f]{4})"  # distbuild, looks like /place/db-0/key_0/srvs/worker_bee400196fbaffcdaa36a81c398c9668/srvdata/build/1/ff413fcb-7db2c031-58cdb6ea-d21ce-0/83/TQsuTVEvi2E_FeyJtnIBXw-2/22dc/yabs/... # noqa
                       "/"
                       ")##x",
        "place"
        ], stderr=subprocess.STDOUT)
    coverage_resource = YABS_SERVER_GCDA_COVERAGE(task, "Yabs .gcda coverage", COVERAGE_FILENAME)
    sdk2.ResourceData(coverage_resource).ready()


def dump_access_counters(server, access_counters_dir):
    logging.info("Dumping access counters...")

    access_counters_url = "http://localhost:{}/access_counters".format(server.listen_port)
    try:
        access_counters_data = requests.get(access_counters_url)
        access_counters_data.raise_for_status()
    except requests.exceptions.RequestException as e:
        logging.error("Got error when dumping access counters: {}".format(e))
    else:
        access_counters_filename = "{}.json".format(server.name)
        access_counters_path = os.path.join(access_counters_dir, access_counters_filename)
        with open(access_counters_path, "w") as f:
            json.dump(access_counters_data.json(), f, indent=4)


def generate_semaphores(hamsters, capacity=None):
    """Generate semaphores to limit the number of tasks that perform requests hamsters.
    It is used to limit the load on the hamsters.

    :param hamsters: Dictionary of hamster service tags and their endpoint resource id.
    :type hamsters: dict[str, int]
    :param capacity: Dictionary of hamster service tags and their capacity,
        defaults to sandbox.projects.yabs.qa.hamster.config.CAPACITY
    :type capacity: dict, optional
    :return: List of objects, representing semaphore acquisition.
    :rtype: list[sandbox.common.types.task.Semaphore.Acquire]
    """
    capacity = capacity or HAMSTER_CAPACITY
    return [
        ctt.Semaphores.Acquire(
            name="yabs_server_hamster_{tag}_{resource_id}".format(
                tag=service_tag,
                resource_id=resource_id,
            ),
            weight=1,
            capacity=capacity[service_tag]
        )
        for service_tag, resource_id in hamsters.items()
        if service_tag in capacity
    ]
