# -*- coding: utf-8 -*-

import datetime
import json
import logging
import os
import posixpath
import re
import subprocess
import tempfile
from collections import defaultdict, OrderedDict, Counter
from cProfile import Profile
from pstats import Stats

from six.moves import StringIO

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.common import config
from sandbox.common import utils as common_utils
from sandbox.common.types import task as task_type
from sandbox.common.types.misc import NotExists
from sandbox.common.errors import SandboxException, TaskFailure
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
from sandbox.projects.common.yabs.server.util.general import check_tasks, CustomAssert, try_get_from_vault
from sandbox.projects.common.yabs.server.util import truncate_output_parameters
from sandbox.projects.yabs.base_bin_task import BaseBinTaskMixin, base_bin_task_parameters
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShoot2 import (
    EDebugOutputMode,
    YabsBadRequestsID,
    YabsServerB2BFuncShoot2Parameters,
    YabsServerExtTagsDump,
)
from sandbox.projects.yabs.qa.tasks.YabsServerUploadShootResultToYt import get_shoot_task_dumps
from sandbox.projects.yabs.qa.tasks.base_compare_task.parameters import BaseCompareTaskParameters
from sandbox.projects.yabs.qa.tasks.base_compare_task.task import BaseCompareTask
from sandbox.projects.yabs.qa.errorbooster.decorators import track_errors
from sandbox.projects.yabs.qa.resource_types import (
    BaseBackupSdk2Resource,
    YABS_REPORT_RESOURCE,
    UCTool,
    YabsResponseDumpUnpacker,
    YabsFtSmartMetaReport
)
from sandbox.projects.yabs.qa.response_tools.unpacker import uc_decompress, get_unpacker_executable_path
from sandbox.projects.yabs.qa.utils.general import (
    diff_dict,
    html_hyperlink,
    makedirs_except,
    startrek_hyperlink,
    is_precommit_check,
)
from sandbox.projects.yabs.qa.utils.resource import sync_resource
from sandbox.projects.yabs.qa.mutable_parameters import MutableParameters
from sandbox.projects.yabs.qa.ammo_module.requestlog.adapters.yabs_specific.sandbox.parameters import StringifiedList
from sandbox.projects.yabs.qa.hamster.utils import calculate_enabled_hamsters
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.analyze_logs import (
    Agg, GroupBy, Func,
    hitlogid_bannerid_eventcost_rank, hitlogid_bannerid_billcost_rank,
    hitlogid_bannerid_clicks, hitlogid_bannerid_shows,
)
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.compare import DEFAULT_COMPARISON_FLAGS, REPORT_CHUNKS_COUNT, ComparisonFlag, compare_multiple
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.compare_dict import compare_dicts
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.ignored_count_requests import get_ignored_count_request_ids
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.operations import avg, group, size, to_dict
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.trafaret_rank_diff import compare_trafaret_ranks
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.profile import (
    PROFILE_TMP_DIR,
    SORT_PROFILE_STATS_BY,
    update_profile_stats,
)

from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.report_utils import make_logs_statistics_report_data, make_smart_report_data
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.yt_report import upload_failed_test_ids_to_yt
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.report import (
    CmpReport, CmpReportLinksIndex, CmpSmartMetaReport, CmpCliReport,
    ECmpReportFiles, create_comparison_metareport,
)
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.default_paint import (
    DEFAULT_BASE64_CAPTURE_JSON,
    DEFAULT_IGNORED_LOG_FIELDS_SET,
    DEFAULT_IGNORED_LOG_FIELDS_STR,
    DEFAULT_IGNORED_RECORD_FIELDS_SET,
    DEFAULT_IGNORED_RECORD_FIELDS_STR,
    DEFAULT_USER_SUBSTITUTES_JSON,
    LOG_TRANSFORMATION,
    RESPONSE_TRANSFORMATION,
    EXT_TRANSFORMATION,
    EXT_IGNORES,
)
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.query import (
    get_query_yt,
    get_query_chyt
)

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace, trace_calls, trace_entry_point
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.generic import enqueue_task, new_resource
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sdk2 import make_resource_ready


logger = logging.getLogger(__name__)


JSON_AMMO_TYPES = {'yabs', 'bs'}
TEMPLATES_DIR = os.path.join(os.path.dirname(__file__), 'templates')
TTL_FOR_DUMP_WITH_DIFF = 7


def split_stripped(str, separator):
    """Returns the array of stripped substrings"""
    return [s.strip() for s in str.split(separator)]


def prolongate_shoot_task_dumps_ttl(task, ttl):
    for resource in get_shoot_task_dumps(task.id):
        logging.debug("Set TTL={} days for resource #{}".format(ttl, resource.id))
        resource.ttl = ttl


def get_hashes(path):
    hash_dict = defaultdict(str)
    with open(os.path.join(path, "hashes"), "rb") as f:
        for line in f:
            test_id, hash = line.split(':')
            hash_dict[test_id] = hash
    return hash_dict


class YabsFtReport(BaseBackupSdk2Resource):
    """ Resource with diff report """

    auto_backup = True
    restart_policy = ctr.RestartPolicy.DELETE


class YabsFtReportIndex(BaseBackupSdk2Resource):
    """ Resource with diff report index (needed for TestEnv and checks) """

    auto_backup = True
    restart_policy = ctr.RestartPolicy.DELETE


class YabsFtCliReport(BaseBackupSdk2Resource):
    """Resource with diff report, formatted to be human readable in terminal"""

    auto_backup = True
    restart_policy = ctr.RestartPolicy.DELETE


class YabsFtExtTagsDiff(BaseBackupSdk2Resource):
    """Resource with ext tags diff"""

    auto_backup = True
    restart_policy = ctr.RestartPolicy.DELETE


class YabsLogsStatDiff(BaseBackupSdk2Resource):
    """Resource with log statistics diff"""

    auto_backup = True
    restart_policy = ctr.RestartPolicy.DELETE


class YabsServerB2BFuncShootCmp(BaseBinTaskMixin, BaseCompareTask):
    """SDK2 task for comparing two FT_SHOOT result dumps"""

    name = 'YABS_SERVER_B2B_FUNC_SHOOT_CMP'
    non_identical_test_sets = False

    events_log_aggregation_config = {
        'RealCost': [sum, avg],
        'BillCost': [sum, avg],
        'BidCorrectionRaw': [avg],
        'ABConversionCostCoef': [avg],
        # tuple means product
        ('BillCost', 'Rank'): [sum, avg],
        'EventCost': [sum, avg],
        ('EventCost', 'Rank'): [sum, avg],
        'Rank': [avg],
        GroupBy(by=('HitLogID', 'BannerID'), aggs=Agg('EventCost', 'max'), name='MaxEventCost'): [sum, avg],
        GroupBy(by=('HitLogID', 'BannerID'), aggs=Agg('BillCost', 'max'), name='MaxBillCost'): [sum, avg],
        GroupBy(by=('ProductType'), aggs=Agg('ProductType', 'count'), name='ProductType'): [to_dict],
        Func(function=hitlogid_bannerid_eventcost_rank,
             name='<b>dot(MaxEventCost, MaxRank)</b>',
             used_fields=('HitLogID', 'BannerID', 'EventCost', 'Rank')): [sum, avg],
        Func(function=hitlogid_bannerid_billcost_rank,
             name='<b>dot(MaxBillCost, MaxRank)</b>',
             used_fields=('HitLogID', 'BannerID', 'BillCost', 'Rank')): [sum, avg],
        Func(function=hitlogid_bannerid_clicks,
             name='<b>Clicks</b>',
             used_fields=('HitLogID', 'BannerID', 'CounterType')): [size],
        Func(function=hitlogid_bannerid_shows,
             name='<b>Shows</b>',
             used_fields=('HitLogID', 'BannerID', 'CounterType')): [size],
    }

    statistics_aggregation_config = {
        'event': events_log_aggregation_config,
        'event_bad': events_log_aggregation_config,
        'factors': {
            'BM_PCTR': [avg],
            'BM_CTR': [avg],
            'CTR': [avg],
            'PCTR': [avg],
        },
        'dsp': {
            'PartnerPrice': [sum, avg],
            'Price': [sum, avg],
        },
        'hit': {
            'GluedCompetitorsNumber': [min, max, avg],
            'PPCGot': [min, max, avg],
            'PMatchFailBits': [group],
            'MetaCtrCalculation': [min, max, avg],
            'DirectBannersOnMeta': [min, max, avg],
            'CreativeBannersOnMeta': [min, max, avg],
            'MobileAppBannersOnMeta': [min, max, avg],
        },
    }

    ignore_compare_input_parameters = [
        'binary_base_resources',
        'linear_models_data_resource',
        'server_resource',
        'response_dumps_ttl',
        'stat_binary_base_resources',
        'meta_binary_base_resources',
        'stat_server_resource',
        'meta_server_resource',
        'saas_freeze_data_resource',
        'hamster_ext_service_tags',
        'ext_service_endpoint_resources',
    ]
    critical_compare_input_parameters = [
        'cache_daemon_stub_resource',
        'requestlog_resource',
    ]

    class Requirements(sdk2.Task.Requirements):
        disk_space = 250 * 1024
        cores = 32
        ram = 100 * 1024  # 100 GiB
        client_tags = ctc.Tag.SSD
        environments = (
            environments.PipEnvironment('demjson'),
            environments.PipEnvironment('numpy'),
            environments.PipEnvironment('jinja2'),
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('futures'),
        )

    class Context(sdk2.Task.Context):
        has_diff = None
        do_binary_search = True
        ammo_type = None
        short_report_text = None
        short_report_link = None
        cloned_shoot_task_ids = {'pre': None, 'test': None}
        merged_shoot_parameters_from = [None, None]
        on_save_logs = []

    class Parameters(BaseCompareTaskParameters):
        kill_timeout = int(datetime.timedelta(hours=2).total_seconds())  # TODO: igorock@ profile and fix BSSERVER-14083 BSSERVER-6297
        push_tasks_resource = True

        _base_bin_task_parameters = base_bin_task_parameters(
            release_version_default=task_type.ReleaseStatus.STABLE,
            resource_attrs_default={"task_bundle": "yabs_server_shoot"},
        )

        with sdk2.parameters.Group('Common settings') as common_settings:
            unpacker_resource = sdk2.parameters.Resource(
                'Unpacker resource',
                resource_type=YabsResponseDumpUnpacker,
            )
            uc_resource = sdk2.parameters.Resource('UC tool resource', resource_type=UCTool)
            template_resource = sdk2.parameters.Resource('Template resource', resource_type=YABS_REPORT_RESOURCE)

            new_paints = sdk2.parameters.Bool('Use new paints', description='Docs at https://nda.ya.ru/3Vn77g', default=True)
            with new_paints.value[False]:
                user_substitutes = sdk2.parameters.JSON(
                    'User regex substitutes to apply before diff', default=DEFAULT_USER_SUBSTITUTES_JSON
                )
                log_ignores = sdk2.parameters.String(
                    'Comma-separated ignored log fields '
                    '(can also specify log before colon, '
                    'like "hit:HitLogID", and whole logs, like "access_stat:*")',
                    default=DEFAULT_IGNORED_LOG_FIELDS_STR,
                    multiline=True)
            with new_paints.value[True]:
                user_substitutes_new = sdk2.parameters.JSON('Response transformations', default=RESPONSE_TRANSFORMATION)
                log_ignores_new = sdk2.parameters.JSON('Logs transformations', default=LOG_TRANSFORMATION)

            base64_capture = sdk2.parameters.JSON(
                'Regex patterns to capture and decode base64 entities',
                default=DEFAULT_BASE64_CAPTURE_JSON)
            skip_record_fields = sdk2.parameters.String(
                'Comma-separated ignored record fields',
                default=DEFAULT_IGNORED_RECORD_FIELDS_STR)

            new_exts_blocks = sdk2.parameters.Bool(
                'Use additional exts (external services requests) comparison',
                default=True)
            with new_exts_blocks.value[True]:
                ext_substitutes = sdk2.parameters.JSON(
                    'Ext transformations',
                    description='Docs at https://nda.ya.ru/t/x3L45ccZ3W57oo',
                    default=EXT_TRANSFORMATION)
                ext_ignores = sdk2.parameters.JSON(
                    'Disable diff blocks for ext tags',
                    description='Docs at https://nda.ya.ru/t/x2bfj8Ku3W57ou',
                    default=EXT_IGNORES)

            excluded_ext_tags = sdk2.parameters.List(
                'External tags to exclude from result',
                default=[
                    'daemon_tvm_client',
                    'bigb_rf_balancer',
                    'bscount_fbs',
                    'bscount_fbs2',
                    'pcode_renderer',  # TODO: remove after https://st.yandex-team.ru/BSSERVER-16541
                    'get_tsar',  # TODO: remove after fixes https://st.yandex-team.ru/SHMDUTY-317
                    'get_saas_data',  # TODO: remove after fixes https://st.yandex-team.ru/SHMDUTY-317
                ],
            )
            parallel_jobs = sdk2.parameters.Integer(
                'Number of jobs to be used in paralleled operations. Leaving "None" will set to #ncpu for host',
                default=None,
            )
            unpacker_threads = sdk2.parameters.Integer('Threads for unpacker', default=32)
            report_ttl = sdk2.parameters.Integer(
                'TTL for b2b report',
                description='Default ttl is 42 days. It should be 550 days for big tests',
                default=42)
            max_filter_smart_report = sdk2.parameters.Integer(
                'Max number of smart reports in diff viewer',
                default=7,
                required=True
            )
            with sdk2.parameters.Group("Comparison settings") as comparison:
                parsing_timeout = sdk2.parameters.Integer("Timeout in seconds for parsing single JSONP response", default=4)
                failed_test_yt_prefix = sdk2.parameters.String('Prefix in YT to upload failed test IDs to', default='//home/yabs-cs-sandbox/reports')

                with sdk2.parameters.CheckGroup("Comparison flags") as comparison_flags:
                    for name, flag in DEFAULT_COMPARISON_FLAGS.items():
                        comparison_flags.values[name] = comparison_flags.Value(flag.description, checked=flag.value)

                with sdk2.parameters.CheckGroup("Find diff in blocks") as diff_blocks:
                    diff_blocks.values.response_data = diff_blocks.Value("Response", checked=True)
                    diff_blocks.values.headers = diff_blocks.Value("Headers", checked=True)
                    diff_blocks.values.logs_data = diff_blocks.Value("Logs", checked=True)
                    diff_blocks.values.code = diff_blocks.Value("Code", checked=True)
                    diff_blocks.values.exts_query_data = diff_blocks.Value("Exts queries", checked=True)
                    diff_blocks.values.exts_headers_data = diff_blocks.Value("Exts headers", checked=True)
                    diff_blocks.values.exts_request_data = diff_blocks.Value("Exts request entities", checked=True)
                    diff_blocks.values.exts_base_info = diff_blocks.Value("Exts base info (status, code)", checked=True)
                    diff_blocks.values.exts_response_data = diff_blocks.Value("Exts response", checked=False)
                    diff_blocks.values.exts_response_hash_data = diff_blocks.Value("Exts response hash", checked=False)

                compare_request_id_whitelist = StringifiedList('Compare only selected request_ids', default=[])
                compare_request_id_blacklist = StringifiedList('Ignore selected request_ids')

                diff_context = sdk2.parameters.Integer('Extra context lines to show with diff', default=3)

                use_search_adapter_for_uniformat = sdk2.parameters.Bool('Use search adapter for unified answer', default=False)
                use_hashes = sdk2.parameters.Bool('Use hashes to compare tests', default=True)

            batch_mode = sdk2.parameters.Bool('Use batch mode (rerun shoot tasks with updated parameters, then compare)', default_value=False)
            with batch_mode.value[True]:
                shoot_task_parameters = YabsServerB2BFuncShoot2Parameters()

            check_products = sdk2.parameters.Bool('Check products', default=True)
            compare_statuses = sdk2.parameters.Bool('Compare statuses of shootings', default=False)
            compare_enabled_hamsters = sdk2.parameters.Bool('Compare enabled hamsters', default=True)
            run_only_in_precommit_checks = sdk2.parameters.Bool('Run task only in precommit checks, otherwise exit immediately', default=False)
            ttl_for_dump_with_diff = sdk2.parameters.Integer('Increase response_dumps ttl for compared tasks is CMP has diff', default=TTL_FOR_DUMP_WITH_DIFF)

        with sdk2.parameters.Group('Dirty hacks') as dirty_hacks:
            # TODO: igorock@ remove after BSEFFECTIVE-298
            volkswagen_test = sdk2.parameters.Bool('Force has_diff=false', description='Temporarily BSEFFECTIVE-298', default=False, do_not_copy=True)

        with sdk2.parameters.Output():
            log_stat_diff_resource_id = sdk2.parameters.Integer("log_statistics diff resource ID")

    def on_save(self):
        super(YabsServerB2BFuncShootCmp, self).on_save()
        if self.pre_task and self.test_task and [self.pre_task.id, self.test_task.id] != self.Context.merged_shoot_parameters_from:
            pre_task_parameters = self.pre_task.Parameters
            test_task_parameters = self.test_task.Parameters
            for param_name, _ in pre_task_parameters:
                try:
                    if hasattr(test_task_parameters, param_name) and getattr(pre_task_parameters, param_name) == getattr(test_task_parameters, param_name):
                        try:
                            setattr(self.Parameters, param_name, getattr(pre_task_parameters, param_name))
                        except AttributeError:
                            self.Context.on_save_logs.append('Got attribute error on setting shoot task param {}'.format(param_name))
                except ValueError:
                    self.Context.on_save_logs.append('Got ValueError on getting attribute {}'.format(param_name))
            self.Context.merged_shoot_parameters_from = [self.pre_task.id, self.test_task.id]

    def construct_log_ignore_json(self):
        ignore_dict = {'by_log': set(), 'by_field': set(), 'by_log_and_field': {}}
        for item in split_stripped(self.Parameters.log_ignores, ','):
            subitem_list = split_stripped(item, ':')
            if len(subitem_list) == 1:
                ignore_dict['by_field'].add(subitem_list[0])
            elif len(subitem_list) == 2:
                if subitem_list[1] == '*':
                    ignore_dict['by_log'].add(subitem_list[0])
                else:
                    ignore_dict['by_log_and_field'].setdefault(subitem_list[0], set()).add(subitem_list[1])
            else:
                raise SandboxException('Invalid log ignores')
        ignore_dict['by_log'] = list(ignore_dict['by_log'])
        ignore_dict['by_field'] = list(ignore_dict['by_field'])
        for item in ignore_dict['by_log_and_field']:
            ignore_dict['by_log_and_field'][item] = list(ignore_dict['by_log_and_field'][item])
        return json.dumps(ignore_dict)

    @track_errors
    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        self.start_time = datetime.datetime.now()
        if self.Parameters.run_only_in_precommit_checks and not is_precommit_check(self):
            self.set_info("Task execution skipped because it executes only on precommit checks")
            self.Context.has_diff = False
            self.Parameters.has_diff = False
            return

        if self.Context.on_save_logs:
            logging.warning('Got following errors in serverside hooks:')
            while self.Context.on_save_logs:
                logging.warning(self.Context.on_save_logs.pop())
            logging.warning('-----')
        if self.Parameters.batch_mode:
            CustomAssert(self.pre_task and self.test_task, 'Both pre and test task ids need to be specified for batch mode', TaskFailure)
            self.on_execute_batch()
        else:
            with self.memoize_stage.execute_cmp(commit_on_entrance=False), trace('execute_cmp'):
                self.on_execute_cmp()
                if self.Parameters.compare_statuses:
                    self.compare_statuses()

    @trace_calls
    def on_execute_batch(self):
        '''
        Clones shoot sessions that were provided to the task and reruns them with updated input parameters.
        After that, runs compare logic on those new sessions in a separate CMP-task.
        '''
        tasks_to_check = []
        for item in ('pre', 'test'):
            if self.Context.cloned_shoot_task_ids[item]:
                tasks_to_check.append(self.Context.cloned_shoot_task_ids[item])
            else:
                shoot_task = getattr(self, '{}_task'.format(item))
                if not shoot_task:
                    raise SandboxException('You need to specify {} task'.format(item))
                shoot_params = MutableParameters.__from_parameters__(shoot_task.Parameters)
                for param_name, param_value in self.Parameters.shoot_task_parameters:
                    default_param_value = getattr(type(self).Parameters, param_name).default_value
                    if param_value != default_param_value:
                        setattr(shoot_params, param_name, param_value)
                shoot_task_type = type(shoot_task)
                cloned_shoot_task = enqueue_task(shoot_task_type(
                    self,
                    description='Updated {} run, copy of #{} from CMP task #{}'.format(item, shoot_task.id, self.id),
                    **truncate_output_parameters(dict(shoot_params), shoot_task_type.Parameters)
                ))
                tasks_to_check.append(cloned_shoot_task)
                self.Context.cloned_shoot_task_ids[item] = cloned_shoot_task.id
        check_tasks(self, tasks_to_check)
        cmp_params = self.get_self_params()
        cmp_params.update(
            {
                'pre_task': self.Context.cloned_shoot_task_ids['pre'],
                'test_task': self.Context.cloned_shoot_task_ids['test'],
                'batch_mode': False,
            }
        )
        cmp_subtask = enqueue_task(self.type(self, description='CMP subtask of task #{}'.format(self.id), **cmp_params))
        self.set_info('Comparison delegated to <a href="{task_url}" target="_blank">another CMP task</a>'.format(
            task_url=common_utils.get_task_link(cmp_subtask.id)),
            do_escape=False,
        )

    def get_self_params(self):
        params_data = self.server.task[self.id].custom.fields.read()
        return {item['name']: item['value'] for item in params_data if item['value'] is not None}

    @trace_calls
    def on_execute_cmp(self):
        terminate_task = self.check_tasks_parameters()
        if terminate_task:
            return

        if self.Parameters.compare_enabled_hamsters:
            with self.memoize_stage.compare_enabled_hamsters(commit_on_entrance=False):
                enabled_hamsters_diff_string = self.compare_enabled_hamsters()
                if enabled_hamsters_diff_string:
                    self.set_info("Enabled hamsters differ:\n\n{}".format(enabled_hamsters_diff_string))
                    self.Parameters.has_diff = True
                    self.Context.has_diff = True
                    return

        profiler = Profile()
        profiler.enable()
        os.makedirs(PROFILE_TMP_DIR)

        self.dump_info = {'pre': {}, 'test': {}}
        bad_requests_resources = {'pre': '', 'test': ''}

        user_substitutes = ''
        if self.Parameters.new_paints:
            user_substitutes_json = self.Parameters.user_substitutes_new
        else:
            user_substitutes_json = self.Parameters.user_substitutes
        if user_substitutes_json:
            user_substitutes = json.dumps(user_substitutes_json)

        if self.Parameters.new_paints:
            log_ignores = json.dumps(self.Parameters.log_ignores_new)
        else:
            log_ignores = self.construct_log_ignore_json()

        base64_capture = json.dumps(self.Parameters.base64_capture) if self.Parameters.base64_capture else ''
        dump_resources = {}
        bad_requests_resources = {}
        ext_tags_resources = {}
        for shoot_task_type, shoot_task in [('pre', self.pre_task), ('test', self.test_task)]:
            dump_resources[shoot_task_type] = get_shoot_task_dumps(shoot_task.id)
            if not dump_resources[shoot_task_type]:
                raise SandboxException('Could not find a dump result from input task')
            self.merge_dump_resource_attrs(dump_resources[shoot_task_type], self.dump_info[shoot_task_type], ('base_revision', 'ammo_type', 'debug_cookie'))
            self.prepare_aggregates(dump_resources[shoot_task_type], shoot_task_type, user_substitutes, base64_capture, self.Parameters.skip_record_fields, log_ignores)
            try:
                bad_requests_resources[shoot_task_type] = sync_resource(resource=YabsBadRequestsID.find(task=shoot_task, state=ctr.State.READY).first())
            except Exception:
                logging.exception("Can't find resource with bad requests id for %s", shoot_task_type)
                bad_requests_resources[shoot_task_type] = None
            try:
                ext_tags_resources[shoot_task_type] = sync_resource(resource=YabsServerExtTagsDump.find(task=shoot_task, state=ctr.State.READY).first())
            except Exception:
                logging.exception("Can't find resource with failed tags for %s", shoot_task_type)
                ext_tags_resources[shoot_task_type] = None

        pre_task = self.pre_task
        bsrank = pre_task.Parameters.meta_mode == 'bsrank'

        def concat_requests_list(bad_requests_resources):
            ids = set()
            for item in ('pre', 'test'):
                if bad_requests_resources[item] is not None:
                    with open(bad_requests_resources[item]) as input_file:
                        ids |= set(int(line) for line in input_file if line)
            return ids

        bad_requests_ids = concat_requests_list(bad_requests_resources)
        logging.info("Do not compare %d requests", len(bad_requests_ids))
        logging.debug("Do not compare requests: %s", ','.join(str(item) for item in bad_requests_ids))

        ext_tags = {'pre': {}, 'test': {}}
        for item in ('pre', 'test'):
            if ext_tags_resources[item] is not None:
                with open(ext_tags_resources[item]) as input_file:
                    for line in input_file:
                        if not line:
                            continue
                        ext_tags[item].update(json.loads(line))
        ext_tags_diff_path = 'ext_tags_diff.txt'
        with open(ext_tags_diff_path, 'a') as f:
            f.write(compare_dicts(ext_tags['pre'], ext_tags['test'])[-1])
        self._create_report_resource(
            resource_class=YabsFtExtTagsDiff, description='Ext tags diff', path=ext_tags_diff_path
        )

        if self.dump_info['pre']['ammo_type'] != self.dump_info['test']['ammo_type']:
            raise SandboxException('Dumps to compare have different ammo types')
        self.Context.ammo_type = self.dump_info['pre']['ammo_type']

        report = CmpReport()
        cli_report = CmpCliReport()

        comparison_flags = {
            name: ComparisonFlag('', name in self.Parameters.comparison_flags) for name in DEFAULT_COMPARISON_FLAGS
        }
        logging.debug('Comparison flags: %s', comparison_flags)

        tempfile.tempdir = 'tmp'
        comparison_result, web_report_index, statistics_result, total_tests = self.compare(
            'pre', 'test', report, cli_report, bad_requests_ids, comparison_flags, bsrank
        )

        if comparison_result is None:
            return

        if self.pre_task.Parameters.uploaded_logs_to_yt_prefix and self.test_task.Parameters.uploaded_logs_to_yt_prefix:
            query_params = {
                'pre_id': self.Parameters.pre_task.Parameters.upload_to_yt_task_id,
                'test_id': self.Parameters.test_task.Parameters.upload_to_yt_task_id,
                'cmp_id': self.id,
                'yt_prefix': self.Parameters.failed_test_yt_prefix,
            }

            self.set_info(
                '<a href="https://wiki.yandex-team.ru/bannernajakrutilka/server/sandbox/how-to-select-shm-results/" '
                'target="_blank">How to select SHM results</a>\n{}\n{}'.format(
                    get_query_yt(query_params),
                    get_query_chyt(query_params)
                ),
                do_escape=False
            )

        diff_info = []
        if statistics_result.diff_abs:
            for product in statistics_result.diff_abs['event']['ProductType.to_dict'].keys():
                if statistics_result.pre['event']['ProductType.to_dict'].get(product, 0) != statistics_result.test['event']['ProductType.to_dict'].get(product, 0):
                    diff_info.append("{}: {} -> {}".format(
                        product,
                        statistics_result.pre['event']['ProductType.to_dict'].get(product, 0),
                        statistics_result.test['event']['ProductType.to_dict'].get(product, 0))
                    )
        if not diff_info:
            self.set_info('Count of products has no diff')
        else:
            self.set_info('\n'.join(['Count of product has diff'] + diff_info))

        with self.memoize_stage.set_has_diff():
            if self.Parameters.volkswagen_test:
                self.Context.has_diff = False
                self.Parameters.has_diff = False
            else:
                self.Context.has_diff = bool(comparison_result.test_failures_count)
                self.Parameters.has_diff = bool(comparison_result.test_failures_count)

        if self.Context.has_diff:
            try:
                for task in [self.pre_task, self.test_task]:
                    prolongate_shoot_task_dumps_ttl(task, self.Parameters.ttl_for_dump_with_diff)
            except Exception as e:
                logging.error('Failed to prolongate shoot tasks dumps TTL: %s', e)
                self.set_info('Failed to prolongate shoot tasks dumps TTL. See details at common.log')

        log_stat_diff_resource_id = self._create_logs_stat_diff_resource(
            statistics_result.pre,
            statistics_result.test,
            statistics_result.diff_in_percents,
            statistics_result.diff_abs
        )
        smart_report_data = make_smart_report_data(comparison_result)
        logs_statistics_report_data = make_logs_statistics_report_data(
            statistics_result.pre,
            statistics_result.test,
            statistics_result.diff_in_percents,
            statistics_result.diff_abs
        )
        smart_meta_report = CmpSmartMetaReport()
        smart_meta_report.add_smart_meta_report(smart_report_data)
        report.add_smart_report(smart_report_data)
        report.add_clusterized_smart_reports(comparison_result.smart_reports)
        report.add_aggregated_logs_statistics(logs_statistics_report_data)
        if self.bsrank_results is not None:
            report.add_bsrank_results(self.bsrank_results)
        html_report, st_report, report_index_resource_url = self.finalize_report_and_upload(
            report, cli_report, smart_meta_report, total_tests, comparison_result.test_failures_count, len(web_report_index)
        )
        self.Context.short_report_text = '{failures} out of {total_tests} tests failed'.format(
            failures=comparison_result.test_failures_count, total_tests=total_tests
        )
        self.Context.short_report_link = report_index_resource_url

        with self.memoize_stage.do_report():
            self.set_info(html_report, do_escape=False)
            self.Parameters.st_report = st_report

        profiler.disable()
        profile_stats_buffer = StringIO()
        profile_stats = Stats(profiler, stream=profile_stats_buffer)
        nfiles = update_profile_stats(profile_stats)
        profile_stats.sort_stats(*SORT_PROFILE_STATS_BY).print_stats()
        # Trim useless file stats from the beginning of the file
        trimmed_profile_stats = '\n'.join(profile_stats_buffer.getvalue().split('\n')[nfiles:])
        profile_result_filename = os.path.join(str(self.log_path()), 'profile.log')
        with open(profile_result_filename, 'w') as f:
            f.write(trimmed_profile_stats)

        with self.memoize_stage.set_log_stat_diff_output_param(commit_on_entrance=False), trace('set_log_stat_diff_output_param'):
            self.Parameters.log_stat_diff_resource_id = log_stat_diff_resource_id

    def merge_dump_resource_attrs(self, dump_resources, attr_dict, merge_attr_list):
        for attribute in merge_attr_list:
            if len({getattr(dump_resource, attribute) for dump_resource in dump_resources}) > 1:
                raise SandboxException('Provided resources from the same group have different "{}" attributes'.format(attribute))
            attr_dict[attribute] = getattr(dump_resources[0], attribute)

    @trace_calls
    def prepare_aggregates(self, dump_resources, destination_path, user_substitutes, base64_capture, skip_record_fields, log_ignores):
        makedirs_except(destination_path)
        unpacked_dump_path = destination_path + '_dump'
        dump_parser_paths_dict = {}

        unpacker_path = None
        if self.Parameters.unpacker_resource:
            logging.debug("Unpacker resource: #%d", self.Parameters.unpacker_resource.id)
            unpacker_path = get_unpacker_executable_path(self.Parameters.unpacker_resource)

        for dump_resource in dump_resources:
            logging.debug("Dump resource #%s", dump_resource.id)

            try:
                dump_debug_mode = EDebugOutputMode[dump_resource.debug_mode] if dump_resource.debug_mode else EDebugOutputMode.json
            except AttributeError:
                dump_debug_mode = EDebugOutputMode.json

            packed_dump_path = sync_resource(resource=dump_resource)

            if self.Parameters.unpacker_resource is None:
                try:
                    dump_parser_id = dump_resource.dump_parser_id
                    if not dump_parser_id:
                        dump_parser_id = None
                except AttributeError:
                    dump_parser_id = None
                logging.debug("Unpacker resource: #%d", dump_parser_id)

                if dump_parser_id not in dump_parser_paths_dict:
                    dump_parser_paths_dict[dump_parser_id] = get_unpacker_executable_path(sdk2.Resource[dump_parser_id])
                unpacker_path = dump_parser_paths_dict[dump_parser_id]

            logging.debug("Decompress dump archive from %s to %s", packed_dump_path, unpacked_dump_path)
            uc_decompress(
                self,
                source_path=packed_dump_path,
                destination_path=unpacked_dump_path,
                uc_resource_id=self.Parameters.uc_resource,
                threads=self.Parameters.unpacker_threads,
            )

            logging.debug("Unpack dump from %s to %s", unpacked_dump_path, destination_path)
            self.unpack_dump(
                unpacked_dump_path,
                destination_path,
                unpacker_path,
                dump_debug_mode,
                user_substitutes,
                base64_capture,
                skip_record_fields,
                log_ignores,
                self.Parameters.new_paints,
                self.Parameters.excluded_ext_tags,
                self.Parameters.new_exts_blocks,
                self.Parameters.ext_substitutes,
                self.Parameters.ext_ignores,
                self.Parameters.use_search_adapter_for_uniformat,
                self.Parameters.use_hashes,
            )
            os.remove(unpacked_dump_path)

    def compare_enabled_hamsters(self):
        """Compare hamsters enabled in pre and test tasks.

        :return: String with diff between enabled hamsters in pre and test tasks.
        :rtype: str
        """
        pre_hamsters = calculate_enabled_hamsters(
            self.pre_task.Parameters.hamster_ext_service_tags,
            [int(resource.id) for resource in self.pre_task.Parameters.ext_service_endpoint_resources],
        )
        test_hamsters = calculate_enabled_hamsters(
            self.test_task.Parameters.hamster_ext_service_tags,
            [int(resource.id) for resource in self.test_task.Parameters.ext_service_endpoint_resources],
        )
        logging.debug("Enabled hamsters: %s", {"pre": pre_hamsters, "test": test_hamsters})

        if pre_hamsters == test_hamsters:
            return ""

        return diff_dict(pre_hamsters, test_hamsters)

    @trace_calls
    def unpack_dump(self, dump_path, destination_path, unpacker_path, dump_debug_mode, user_substitutes,
                    base64_capture, skip_record_fields, log_ignores, new_paints, excluded_ext_tags, new_exts_blocks,
                    ext_substitutes, ext_ignores, use_search_adapter_for_uniformat, use_hashes):
        makedirs_except(destination_path)
        if dump_debug_mode == EDebugOutputMode.json:
            convert_command = '--convert'
        elif dump_debug_mode == EDebugOutputMode.proto_binary:
            convert_command = '--convert-proto'
        else:
            raise SandboxException('Unknown debug mode for response converter')
        command_call_list = [unpacker_path, convert_command, '-r', destination_path, dump_path]
        if user_substitutes:
            command_call_list.extend(('--substitutes-json', user_substitutes))
        if log_ignores:
            command_call_list.extend(('--log-ignores-json', log_ignores))
        if base64_capture:
            command_call_list.extend(('--base64-capture-json', base64_capture))
        if skip_record_fields:
            command_call_list.extend(('--skip-record-fields', skip_record_fields))
        if new_paints:
            command_call_list.append('--new-paints')
        if use_hashes:
            command_call_list.append('--writeHashes')
        if use_search_adapter_for_uniformat:
            command_call_list.append('--uniformat')
        if excluded_ext_tags:
            command_call_list.extend(('--exclude-ext-tags', ','.join(excluded_ext_tags)))
        if new_exts_blocks:
            command_call_list.append('--new-exts-blocks')
            if ext_substitutes:
                command_call_list.extend(('--ext-substitutes-json', json.dumps(ext_substitutes)))
            if ext_ignores:
                command_call_list.extend(('--ext-ignores-json', json.dumps(ext_ignores)))

        command_call_list.extend(('-j', str(self.Parameters.unpacker_threads)))
        logger.debug("Unpacker command: %s", " ".join(command_call_list))
        with sdk2.helpers.ProcessLog(self, logger='dolbilo2json_{}'.format(dump_path)) as pl:
            subprocess.check_call(command_call_list, stdout=pl.stdout, stderr=pl.stdout)

    @trace_calls
    def compare(self, pre_path, test_path, report, cli_report, bad_requests_ids, comparison_flags, bsrank=False):
        pre_dumps = set(os.listdir(pre_path))
        test_dumps = set(os.listdir(test_path))

        pre_dumps -= set(['hashes'])
        test_dumps -= set(['hashes'])

        ignored_count_request_ids = get_ignored_count_request_ids(pre_dumps, test_dumps, bad_requests_ids)

        logging.debug("Do not compare secondary count requests: %s", ignored_count_request_ids)

        if pre_dumps ^ test_dumps:
            logging.warning(
                'Non-identical response sets, '
                'make sure you are diffing the right tasks!\n'
                'Pre response count: %s\n'
                'Test response count: %s',
                len(pre_dumps),
                len(test_dumps),
            )
            logging.warning('Request-ID\'s unique to pre: %s', ', '.join(pre_dumps - test_dumps))
            logging.warning('Request-ID\'s unique to test: %s', ', '.join(test_dumps - pre_dumps))
            self.non_identical_test_sets = True

        template_path = sync_resource(resource=self.Parameters.template_resource, resource_type=YABS_REPORT_RESOURCE)
        report.prepare(template_path=template_path)
        bad_test_ids_union = set(bad_requests_ids | ignored_count_request_ids)

        filtered_test_ids = set(test_id for test_id in pre_dumps & test_dumps if int(test_id) not in bad_test_ids_union)
        if self.Parameters.compare_request_id_whitelist:
            filtered_test_ids &= set(self.Parameters.compare_request_id_whitelist)
        if self.Parameters.compare_request_id_blacklist:
            filtered_test_ids -= set(self.Parameters.compare_request_id_blacklist)

        assert len(
            filtered_test_ids
        ), 'We\'ve got zero tests to compare, check shoot tasks or compare_request_id_* parameters!'

        total_tests = len(filtered_test_ids)
        if self.Parameters.use_hashes:
            logging.debug("Read from {}".format(os.path.join(pre_path, "hashes")))
            pre_hashes = get_hashes(pre_path)
            logging.debug("Read from {}".format(os.path.join(test_path, "hashes")))
            test_hashes = get_hashes(test_path)

            zero_diff = set(filter(lambda test_id: test_hashes[test_id] == pre_hashes[test_id], filtered_test_ids))
            # filter after hashes
            filtered_test_ids -= zero_diff
            logging.debug("Responses with equal hashes: {}".format(len(zero_diff)))
            logging.debug("Responses with different hashes: {}".format(len(filtered_test_ids)))

        comparison_result, web_report_index, statistics_result = compare_multiple(
            pre_path=pre_path,
            test_path=test_path,
            test_ids=filtered_test_ids,
            report=report,
            cli_report=cli_report,
            statistics_aggregation_config=self.statistics_aggregation_config,
            n_jobs=self.Parameters.parallel_jobs,
            diff_blocks=self.Parameters.diff_blocks,
            comparison_flags=comparison_flags,
            timeout=self.Parameters.parsing_timeout,
            new_paints=self.Parameters.new_paints,
            diff_context=self.Parameters.diff_context,
            finish_time=self.start_time + datetime.timedelta(seconds=self.Parameters.kill_timeout),
        )

        self.bsrank_results = (
            compare_trafaret_ranks(
                pre_path,
                test_path,
                pre_dumps & test_dumps,
                bad_requests_ids,
                [[], ['TrafaretID'], ['TrafaretID', 'Position']],
                self.Parameters.parallel_jobs,
                REPORT_CHUNKS_COUNT,
            )
            if bsrank
            else None
        )

        report_results = {
            'search': OrderedDict([
                ('pre.code', list(comparison_result.pre_codes)),
                ('test.code', list(comparison_result.test_codes)),
                ('handler', list(comparison_result.handlers)),
                ('tags', list(comparison_result.diff_tags)),
                (
                    'filter_smart_report',
                    [
                        key
                        for key, value in
                        Counter({
                            report: len(test_ids)
                            for report, test_ids in comparison_result.filter_smart_reports.items()
                        }).most_common(self.Parameters.max_filter_smart_report)
                    ]
                )
            ]),
            'meta': [
                {'title': title, 'value': value}
                for title, value in self.construct_metadata(comparison_result).iteritems()
            ],
            'results': web_report_index,
        }

        logging.info(comparison_result.unique_changed_logs)
        report.add_final_report_results(report_results)

        if comparison_result.failed_test_ids:
            report.add_failed_test_ids(comparison_result.failed_test_ids)

            if self.Parameters.failed_test_yt_prefix:
                try:
                    from yt.wrapper import YtClient

                    yt_client = YtClient(token=try_get_from_vault(self, 'yt_token_for_report_upload'), proxy='hahn')
                    yt_report_dir = '{}/{}'.format(self.Parameters.failed_test_yt_prefix, self.id)
                    failed_test_ids_table_path = upload_failed_test_ids_to_yt(comparison_result, yt_client, yt_report_dir)
                    self.set_info(
                        '<a href="https://yt.yandex-team.ru/hahn/?page=navigation&path={}">Failed test IDs uploaded to Hahn</a>'
                        .format(failed_test_ids_table_path),
                        do_escape=False
                    )
                except Exception:
                    logging.exception('Failed to upload failed test ids to YT')
                    self.set_info('Failed to upload failed test ids to YT')

        return comparison_result, web_report_index, statistics_result, total_tests

    @trace_calls
    def _create_logs_stat_diff_resource(self, statistics_pre, statistics_test, statistics_diff_in_percents, statistics_diff_abs):
        report = defaultdict(dict)
        for log_name, log_content in statistics_diff_in_percents.iteritems():
            for log_field in log_content:
                diff_in_percents = statistics_diff_in_percents[log_name][log_field]
                diff_abs = statistics_diff_abs[log_name][log_field]
                if diff_in_percents is None or diff_in_percents == 0:
                    continue
                elif isinstance(diff_in_percents, dict):
                    diff_in_percents = {key: value for key, value in diff_in_percents.items() if value}
                    if not diff_in_percents:
                        continue

                    diff_abs = {key: value for key, value in diff_abs.items() if value}

                log_field_key = re.sub('</?b>', '', str(log_field))
                report[log_name][log_field_key] = {
                    'pre': statistics_pre.get(log_name, dict()).get(log_field, 0),
                    'test': statistics_test.get(log_name, dict()).get(log_field, 0),
                    'diff_in_percents': diff_in_percents,
                    'diff_abs': diff_abs
                }

        file_path = 'logs_stat_diff.json'
        with open(file_path, 'w') as f:
            json.dump(report, f, indent=2)

        resource = self._create_report_resource(
            resource_class=YabsLogsStatDiff, description='Logs statistics diff', path=file_path
        )

        return resource.id

    @trace_calls(save_arguments=(2, 'description'))
    def _create_report_resource(self, resource_class, description, path, **kwargs):
        resource = new_resource(resource_class, task=self, description=description, path=path, ttl=self.Parameters.report_ttl, **kwargs)
        make_resource_ready(resource)
        return resource

    @trace_calls
    def finalize_report_and_upload(self, report, cli_report, smart_meta_report, test_total_count, test_failures_count, web_report_index_count):
        self._create_report_resource(
            resource_class=YabsFtSmartMetaReport,
            description='Smart meta report resource',
            path=smart_meta_report.get_local_path(),
            test_name=self.Parameters.test_name,
            testenv_db=self.Context.testenv_database,
            meta_mode=self.Context.ammo_type,
            arcanum_review_id=self.Context.arcanum_review_id if self.Context.arcanum_review_id is not NotExists else self.Parameters.arcanum_review_id,
            has_diff=self.Context.has_diff,
        )
        cli_report_resource = self._create_report_resource(
            resource_class=YabsFtCliReport,
            description='Terminal-viewable report resource, download and read README.md inside',
            path=cli_report.compress()
        )

        report.compress()
        report_resource = self._create_report_resource(
            resource_class=YabsFtReport,
            description='Report resource',
            path=report.get_local_path(),
        )
        report_url = self.get_resource_url(self.id, report.get_local_path(), report_resource.id)

        def report_file_url(rel_path):
            return posixpath.join(report_url, rel_path)

        report_links = [
            ("Diff viewer", report_file_url(ECmpReportFiles.report_index_html)),
            ("Smart report", report_file_url(ECmpReportFiles.smart_report)),
            ("Clusterized smart report", report_file_url(ECmpReportFiles.clusterized_smart_reports)),
            ("Logs statistics", report_file_url(ECmpReportFiles.logs_statistics_html)),
            ("Full cli report", "https://sandbox.yandex-team.ru/resource/{resource_id}/view".format(resource_id=cli_report_resource.id)),
        ]
        if self.bsrank_results is not None:
            report_links.append(("Trafaret rank report", report_file_url('bsrank_results.html')))

        task_execution_report_html = create_comparison_metareport(
            test_total_count, web_report_index_count, test_failures_count, report_links,
            line_sep='<br>',
            warning='ATTENTION: NON-IDENTICAL RESPONSE SETS!!!\n' if self.non_identical_test_sets else '',
            link_formatter=html_hyperlink,
        )
        task_execution_report_st = create_comparison_metareport(
            test_total_count, web_report_index_count, test_failures_count, report_links,
            line_sep='\n',
            warning='',
            link_formatter=startrek_hyperlink,
        )

        report_links_index = CmpReportLinksIndex()
        report_links_index.add_report_links_index(task_execution_report_html)

        report_index_resource = self._create_report_resource(
            resource_class=YabsFtReportIndex,
            description='Report index resource',
            path=report_links_index.get_local_path(),
        )
        report_index_url = self.get_resource_url(self.id, ECmpReportFiles.report_index_html, report_index_resource.id)
        return task_execution_report_html, task_execution_report_st, report_index_url

    def construct_metadata(self, comparison_result):
        metadata = {
            "Tests": comparison_result.test_total_count,
            "Failures": comparison_result.test_failures_count,
            "Pre base revision": self.dump_info['pre']['base_revision'],
            "Test base revision": self.dump_info['test']['base_revision'],
            "Ammo type": self.Context.ammo_type,
            "Default paint substitutes": DEFAULT_USER_SUBSTITUTES_JSON,
            "Default ignored record fields": DEFAULT_IGNORED_RECORD_FIELDS_STR,
            "Default ignored log fields": DEFAULT_IGNORED_LOG_FIELDS_STR,
            "Additional paint substitutes": self.get_additional_substitutes(),
            "Additional ignored record fields": self.get_additional_ignored_record_fields(),
            "Additional ignored log fields": self.get_additional_ignored_log_fields(),
        }
        if self.dump_info['pre']['debug_cookie'] == self.dump_info['test']['debug_cookie']:
            metadata.update({'Debug cookie': self.dump_info['pre']['debug_cookie']})
        else:
            metadata.update(
                {
                    'Pre debug cookie': self.dump_info['pre']['debug_cookie'],
                    'Test debug cookie': self.dump_info['test']['debug_cookie'],
                }
            )
        return metadata

    def get_additional_substitutes(self):
        return dict((item for item in self.Parameters.user_substitutes.iteritems() if item[0] not in DEFAULT_USER_SUBSTITUTES_JSON))

    def get_additional_ignored_record_fields(self):
        return ','.join(set(split_stripped(self.Parameters.skip_record_fields, ',')) - DEFAULT_IGNORED_RECORD_FIELDS_SET)

    def get_additional_ignored_log_fields(self):
        return ','.join(set(split_stripped(self.Parameters.log_ignores, ',')) - DEFAULT_IGNORED_LOG_FIELDS_SET)

    def compare_statuses(self):
        info = ['Status comparison:']
        pre_statuses = sdk2.Task[self.pre_task].Parameters.final_statuses
        test_statuses = sdk2.Task[self.test_task].Parameters.final_statuses
        if not (pre_statuses and test_statuses):
            return
        pre_statuses = json.loads(pre_statuses)
        test_statuses = json.loads(test_statuses)
        statuses = set(pre_statuses.keys() + test_statuses.keys())
        for status in statuses:
            pre_status_count = pre_statuses.get(status, 0)
            test_status_count = test_statuses.get(status, 0)
            info.append('Status: {}. Count {} -> {}'.format(status, pre_status_count, test_status_count))
        self.set_info('\n'.join(info))

    @property
    def diff_resource_type(self):
        return 'YABS_FT_SMART_META_REPORT'

    @property
    def diff_resource_search_attributes(self):
        return {'meta_mode': self.Context.ammo_type}

    @staticmethod
    def get_resource_url(task_id, resource_path, resource_id):
        settings = config.Registry()
        fs_settings = settings.client.fileserver
        if fs_settings.proxy.host:
            return "https://{}/{}".format(fs_settings.proxy.host, resource_id)

        return "http://{}:{}/{}/{}".format(
            settings.server.web.address.host, fs_settings.port, "/".join(task_type.relpath(task_id)), resource_path
        )
