import json
import logging
import os
import posixpath
import tarfile
import subprocess

from sandbox.projects.yabs.qa.utils.general import makedirs_except, startrek_hyperlink
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.compare import REPORT_CHUNKS_COUNT, CLI_DIFF_DIRNAME
from sandbox.projects.yabs.qa.template_utils import load_template
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.report_utils import (
    make_smart_report_text,
    make_logs_statistics_report_text,
)


logger = logging.getLogger(__name__)
templates_dir = os.path.join(
    os.path.dirname(__file__),
    'templates'
)


def create_comparison_metareport(tests_count, web_report_tests_count, failures_count, report_links, warning='', line_sep='\n', link_formatter=startrek_hyperlink):
    """Creates metareport with diff overview and links to actual diff viewer interfaces

    :param tests_count: int, total amount of tests
    :param failures_count: int, amount of failed tests
    :param report_links: list of pairs (link_text,  link_url)
    :param warning: str, addidional information
    :param line_sep: str, line separator
    :param link_formatter: callable, should take two strings (link_text, link_url) as parameters and return string representation of a hyperlink
    """
    task_executing_report_template = line_sep.join(
        ([warning] if warning else []) + [
            '{failures_count} out of {total_tests} tests differ, {web_report_tests_count} tests are available in web report, the rest could be found in cli report.',
            '{links}',
        ]
    )
    return task_executing_report_template.format(
        failures_count=failures_count,
        total_tests=tests_count,
        web_report_tests_count=web_report_tests_count,
        links=line_sep.join(
            link_formatter(link_url, link_text) for link_text, link_url in report_links
        )
    )


class BaseLocalResource:
    def get_local_path(self):
        raise NotImplementedError


class ECmpReportFiles(object):
    # YABS_FT_REPORT resource
    report_results = 'report.json'
    report_index_html = 'index.html'
    log_statistics = 'log_statistics.json'
    failed_test_ids = 'failed_test_ids.json'
    statistics_pre = 'statistics_pre.json'
    statistics_test = 'statistics_test.json'
    statistics_diff_in_percents = 'statistics_diff_in_percents.json'
    statistics_diff_abs = 'statistics_diff_abs.json'
    smart_report = 'smart_report.txt'
    clusterized_smart_reports = 'clusterized_smart_reports.txt'
    logs_statistics_html = 'logs_statistics.html'
    bsrank_results_html = 'bsrank_results.html'

    # YABS_FT_REPORT_INDEX resource
    smart_meta_report = 'smart_meta_report.json'

    # YABS_FT_SMART_META_REPORT resource
    report_links_index_html = 'index.html'


class CmpReport(BaseLocalResource):
    def __init__(self, report_dir='report_dir'):
        self._report_dir = report_dir
        self._path = report_dir
        self._tests_dir = self._make_report_subdir('tests')
        self._diffs_dir = self._make_report_subdir('diff')

    def prepare(self, template_path):
        target_path = self._report_dir
        if not os.path.exists(target_path):
            makedirs_except(target_path)
        with tarfile.open(template_path) as tar:
            for member in tar.getmembers():
                if not member.isreg():
                    continue

                if 'build' in member.name:
                    member.name = os.path.basename(member.name)
                    tar.extract(member, os.path.join(target_path, 'static/build'))
                elif 'static' in member.name:
                    member.name = os.path.basename(member.name)
                    tar.extract(member, os.path.join(target_path, 'static'))
                elif member.name.endswith('main.standalone.chunked.html'):
                    member.name = os.path.basename(member.name)
                    tar.extract(member, target_path)
                    with open(os.path.join(target_path, 'main.standalone.chunked.html')) as index_template_file:
                        index_template = index_template_file.read()
                        index = index_template.format(chunks_count=REPORT_CHUNKS_COUNT)
                    with open(os.path.join(target_path, ECmpReportFiles.report_index_html), 'wb') as index_file:
                        index_file.write(index)

    def _make_report_subdir(self, subdir):
        path = os.path.join(self._report_dir, subdir)
        makedirs_except(path)
        return path

    def _report_file_path(self, rel_path):
        return posixpath.join(self._report_dir, rel_path)

    def add_final_report_results(self, report):
        with open(self._report_file_path(ECmpReportFiles.report_results), 'w') as report_file:
            json.dump(report, report_file)

    def add_log_statistics(self, log_statistics):
        with open(self._report_file_path(ECmpReportFiles.log_statistics), 'w') as f:
            json.dump(log_statistics, f)

    def add_failed_test_ids(self, failed_test_ids):
        with open(self._report_file_path(ECmpReportFiles.failed_test_ids), 'w') as f:
            json.dump([int(item) for item in failed_test_ids], f, indent=2)

    def add_statistics_diff(self, pre, test, diff_in_percents, diff_abs):
        with open(self._report_file_path(ECmpReportFiles.statistics_pre), 'w') as f:
            json.dump(pre, f)
        with open(self._report_file_path(ECmpReportFiles.statistics_test), 'w') as f:
            json.dump(test, f)
        with open(self._report_file_path(ECmpReportFiles.statistics_diff_abs), 'w') as f:
            json.dump(diff_abs, f)
        with open(self._report_file_path(ECmpReportFiles.statistics_diff_in_percents), 'w') as f:
            json.dump(diff_in_percents, f)

    def add_smart_report(self, smart_report_data):
        smart_report_text = make_smart_report_text(smart_report_data)
        with open(self._report_file_path(ECmpReportFiles.smart_report), 'w') as f:
            f.write(smart_report_text)

    def add_clusterized_smart_reports(self, clusterized_smart_reports):
        text = []
        ordered_reports = sorted(clusterized_smart_reports.items(), key=lambda key_val: len(key_val[1]), reverse=True)
        for smart_report, test_ids in ordered_reports:
            trimmed_test_ids = ", ".join(test_ids)
            MAX_TEST_IDS = 5
            if len(test_ids) > MAX_TEST_IDS:
                trimmed_test_ids = ", ".join([test_id for test_id in test_ids][:MAX_TEST_IDS]) + ", ..."
            trimmed_test_ids = "[" + trimmed_test_ids + "]"
            text.append("Number of tests with the following smart report: {}, example test ids: {} \n\n {}".format(len(test_ids), trimmed_test_ids, smart_report))
        with open(self._report_file_path(ECmpReportFiles.clusterized_smart_reports), 'w') as f:
            f.write(("\n" + "-" * 80 + "\n\n").join(text))

    def add_aggregated_logs_statistics(self, logs_statistics_data):
        logs_statistics_text = make_logs_statistics_report_text(logs_statistics_data)
        logs_statistics_path = self._report_file_path(ECmpReportFiles.logs_statistics_html)
        with open(logs_statistics_path, 'w') as f:
            f.write(logs_statistics_text)

    def add_bsrank_results(self, bsrank_results):
        with open(self._report_file_path(ECmpReportFiles.bsrank_results_html), 'w') as f:
            f.write(str(bsrank_results))

    def move_tests_and_diffs_chunks(self, chunks_data):
        # Boths dirs should be empty to this moment
        os.rmdir(self._tests_dir)
        os.rmdir(self._diffs_dir)

        os.rename(chunks_data.get_tests_dir(), self._tests_dir)
        os.rename(chunks_data.get_diffs_dir(), self._diffs_dir)

    def get_local_path(self):
        return self._path

    def compress(self):
        archive_path = os.path.abspath("report.tar.gz")
        logger.debug("Compress contents of %s to %s", self._report_dir, archive_path)
        with tarfile.open(archive_path, "w:gz") as archive:
            for filename in os.listdir(self._report_dir):
                archive.add(os.path.join(self._report_dir, filename), arcname=filename)
        self._path = archive_path


class CmpReportLinksIndex(BaseLocalResource):
    def add_report_links_index(self, task_execution_report_html):
        with open(ECmpReportFiles.report_links_index_html, 'w') as f:
            f.write(task_execution_report_html)

    def get_local_path(self):
        return ECmpReportFiles.report_links_index_html


class CmpSmartMetaReport(BaseLocalResource):
    def add_smart_meta_report(self, smart_report_data):
        with open(ECmpReportFiles.smart_meta_report, 'w') as f:
            json.dump(smart_report_data, f, indent=2)

    def get_local_path(self):
        return ECmpReportFiles.smart_meta_report


def compress_fast(path, archive_path):
    pigz_part = ''
    if os.path.exists('/usr/bin/pigz'):
        logger.info("pigz found, will compress and decompress faster")
        pigz_part = "--use-compress-program='pigz'"
    else:
        pigz_part = '-z'
        logger.info("no pigz, compressing with default tar")

    fname = os.path.basename(path)
    subprocess.check_call("tar {} -cf {} -C {}/.. {}".format(pigz_part, archive_path, path, fname), shell=True)


class CmpCliReport(BaseLocalResource):
    def __init__(self, cli_report_dir='cli_report_dir'):
        self._cli_report_dir = cli_report_dir
        if not os.path.exists(self._cli_report_dir):
            makedirs_except(self._cli_report_dir)

        self._diffs_dir = self._make_report_subdir(CLI_DIFF_DIRNAME)

    def _make_report_subdir(self, subdir):
        path = os.path.join(self._cli_report_dir, subdir)
        makedirs_except(path)
        return path

    def _write_readme(self):
        with open(os.path.join(self._cli_report_dir, 'README.md'), 'w') as f:
            f.write(load_template('cli_report_readme.md', templates_dir=templates_dir).encode('utf-8'))

    def move_from_chunks_data(self, chunks_data):
        os.rmdir(self._diffs_dir)
        self._write_readme()
        os.rename(chunks_data.get_cli_diffs_dir(), self._diffs_dir)
        os.rename(chunks_data.get_cli_diffs_index(), os.path.join(self._cli_report_dir, 'report_index.txt'))

    def compress(self):
        archive = os.path.basename(self._cli_report_dir) + '.tar.gz'
        if os.path.exists(archive):
            return archive
        compress_fast(self._cli_report_dir, archive)
        return archive

    def get_local_path(self):
        return self._cli_report_dir
