import datetime
import json
import logging
import os
import shutil
import tempfile
import uuid

from cProfile import Profile
from collections import defaultdict, namedtuple, OrderedDict
from copy import deepcopy
from multiprocessing import cpu_count, Manager, current_process
from pstats import Stats

from sandbox.common.errors import TaskFailure
from sandbox.projects.yabs.qa.utils.general import decode_text, wrap_diff, makedirs_except

from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.analyze_logs import (
    aggregate_statistics,
    calculate_log_statistics,
    CustomCounter,
    diff_in_percents,
    get_dicts_diff,
    merge_dicts,
)
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.compare_text import compare_text
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.compare_dict import compare_dicts
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.compare_json import compare_json_strings, compare_responses
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.compare_jsonp import compare_jsonp_strings, compare_jsonp_responses
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.diff_methods import detect_diff_method, DiffMethods
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.logs_diff import (
    check_failed as check_failed_logs,
    diff_logs,
    merge_unique,
    merge_changed,
)
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.profile import PROFILE_TMP_DIR
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.report_utils import (
    make_smart_report_data,
    make_smart_report_text,
    preload_template,
)
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.compare_response_trees import add_tree_patch


RESOURCE_FILES_COUNT_LIMIT = 10000  # sandbox-specific limit
STATIC_FILES_COUNT_LIMIT = 20  # upper bound limit for static report files, e.g. .html, .css, .js, etc...
REPORT_CHUNKS_COUNT = (RESOURCE_FILES_COUNT_LIMIT - STATIC_FILES_COUNT_LIMIT) / 2  # each chunk consists of 2 files. test/ and diff/

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.DEBUG)


ComparisonFlag = namedtuple("ComparisonFlag", ("description", "value"))
StatisticsComparisonResult = namedtuple("ComparisonResult", ("pre", "test", "diff_in_percents", "diff_abs"))


DEFAULT_COMPARISON_FLAGS = {
    "json_intelligent_comparison": ComparisonFlag("Compare json as parsed structures", True),
    "jsonp_intelligent_comparison": ComparisonFlag("Compare jsonp as parsed structures", True),
    "ignore_blocks_order": ComparisonFlag("Ignore blocks order within the response with '// yandex-splitter'", False),
    "try_parse_text_as_json": ComparisonFlag("Allow JSON content with \"Content-Type: text/html\"", False),
    "try_parse_jsonp_as_json": ComparisonFlag("Allow JSON content with \"Content-Type: application/javascript\"", True),
}
DEFAULT_UNIQUE_CHANGED_LOGS = {
    'unique': {'pre': set(), 'test': set()},
    'changed': {}
}
DEFAULT_LOG_STATISTICS = {
    response_type: {
        'data': {},
        'counts': CustomCounter(),
    }
    for response_type in ('pre', 'test')
}


def move_hit_p_match_fail_bits(statistics):
    """
    Move data from `statistics['hit']['PMatchFailBits.group']`
    to `statistics['hit.PMatchFailBits']`
    """

    statistics['hit.PMatchFailBits'] = statistics['hit']['PMatchFailBits.group']
    del statistics['hit']['PMatchFailBits.group']


def abbreviate_diff(diff_text, limit=20):
    lines = diff_text.split(u'\n')
    show = u'\n'.join(lines[:limit])
    if len(lines) > limit:
        show += u'\nSee more in detailed view'
    return show


def _compare_one(pre_path, test_path, test_id, statistics_aggregation_config,
                 diff_blocks=None, jsonp_parser=None, comparison_flags=None, timeout=4, diff_context=3):
    with open(os.path.join(pre_path, test_id), "rb") as f:
        pre_raw = f.read()
        pre_dict = json.loads(pre_raw.decode("utf-8", "ignore"))
    with open(os.path.join(test_path, test_id), "rb") as f:
        test_raw = f.read()
        test_dict = json.loads(test_raw.decode("utf-8", "ignore"))

    return _compare_data(
        pre_dict,
        test_dict,
        test_id,
        statistics_aggregation_config,
        diff_blocks=diff_blocks,
        jsonp_parser=jsonp_parser,
        comparison_flags=comparison_flags,
        timeout=timeout,
        diff_context=diff_context,
    )


class SingleComparisonResult(object):
    __slots__ = ('test_id', 'has_diff', 'pre_code', 'test_code', 'handler', 'diff',
                 'request', 'diff_tags', 'unique_changed_logs', 'log_statistics',
                 'response_diff_tree', 'smart_report', 'filter_smart_report')

    def __init__(
        self,
        test_id, has_diff, pre_code, test_code, handler,
        diff_tags=None, diff=u'', request=u'', unique_changed_logs=None, log_statistics=None, response_diff_tree=None
    ):
        self.test_id = test_id
        self.has_diff = has_diff
        self.pre_code = pre_code
        self.test_code = test_code
        self.handler = handler

        self.diff = diff
        self.request = request
        self.response_diff_tree = response_diff_tree or dict()

        self.diff_tags = diff_tags or set()

        self.unique_changed_logs = unique_changed_logs or {}
        self.log_statistics = log_statistics or deepcopy(DEFAULT_LOG_STATISTICS)

        smart_report_data = make_smart_report_data(self)

        self.smart_report = make_smart_report_text(
            data=smart_report_data,
            report_template='smart_report.txt'
        )
        self.filter_smart_report = make_smart_report_text(
            data=smart_report_data,
            report_template='filter_smart_report.txt'
        )

    def __str__(self):
        return 'SingleComparisonResult({})'.format(', '.join(['{}={}'.format(k, getattr(self, k)) for k in self.__slots__]))


def _compare_data(pre_dict, test_dict, test_id, statistics_aggregation_config,
                  diff_blocks=None, jsonp_parser=None, comparison_flags=None, timeout=4, diff_context=3):
    _comparison_flags = DEFAULT_COMPARISON_FLAGS
    if comparison_flags is not None:
        _comparison_flags.update(comparison_flags)

    diff_blocks = set(diff_blocks or pre_dict.keys())
    if diff_blocks - set(pre_dict.keys()):
        raise TaskFailure("Pre data has no diff blocks {}".format(list(diff_blocks - set(pre_dict.keys()))))
    if diff_blocks - set(test_dict.keys()):
        raise TaskFailure("Test data has no diff blocks: {}".format(list(diff_blocks - set(test_dict.keys()))))

    unique_changed_logs = {}
    log_statistics = {}
    has_diff = False
    tags = set()
    diff = u''
    response_diff_tree = {}

    for block_name in diff_blocks:
        diff_method = detect_diff_method(pre_dict[block_name]['diff_method'], test_dict[block_name]['diff_method'])
        if diff_method == DiffMethods.no_diff:
            continue

        block_has_diff = False
        raw_pre_block = pre_dict[block_name]['diff_data']
        raw_test_block = test_dict[block_name]['diff_data']

        try_parse_response_data_text_as_json = all([
            _comparison_flags["try_parse_text_as_json"].value,
            diff_method == DiffMethods.text_diff,
            block_name == "response_data"
        ])

        if diff_method == DiffMethods.log_diff:
            diff_str, unique_logs, changed_logs = diff_logs(raw_pre_block, raw_test_block)
            block_has_diff = check_failed_logs(unique_logs, changed_logs)
            if block_has_diff:
                unique_changed_logs.setdefault(block_name, deepcopy(DEFAULT_UNIQUE_CHANGED_LOGS))
                unique_changed_logs[block_name]['unique'] = unique_logs
                unique_changed_logs[block_name]['changed'] = changed_logs

            log_statistics = {
                'pre': calculate_log_statistics(
                    raw_pre_block,
                    statistics_aggregation_config
                ),
                'test': calculate_log_statistics(
                    raw_test_block,
                    statistics_aggregation_config
                )
            }

        elif diff_method == DiffMethods.json_string_diff or try_parse_response_data_text_as_json:
            pre_decoded_text = decode_text(raw_pre_block)
            test_decoded_text = decode_text(raw_test_block)

            raw_blocks_differ = (pre_decoded_text != test_decoded_text)

            if raw_blocks_differ and _comparison_flags["json_intelligent_comparison"].value:
                if block_name == "response_data":
                    try:
                        response_diff_tree = compare_responses(pre_decoded_text, test_decoded_text)
                    except Exception as exc:
                        logger.warning("Error while trying to build json diff from response_data for test_id={}: {}".format(int(test_id), exc))
                try:
                    block_has_diff, diff_str = compare_json_strings(pre_decoded_text, test_decoded_text,
                                                                    ignore_blocks_order=_comparison_flags["ignore_blocks_order"].value, diff_context=diff_context)
                except Exception as exc:
                    logger.warning("Error while trying to compare jsons as objects in %s for test_id=%s: %s", block_name, int(test_id), exc)
                    block_has_diff, diff_str = compare_text(pre_decoded_text, test_decoded_text, diff_context=diff_context)
                    if not try_parse_response_data_text_as_json:
                        tags.add("invalid_json")
            else:
                block_has_diff, diff_str = compare_text(pre_decoded_text, test_decoded_text, diff_context=diff_context)

        elif diff_method == DiffMethods.jsonp_string_diff:
            pre_decoded_text = decode_text(raw_pre_block)
            test_decoded_text = decode_text(raw_test_block)

            raw_blocks_differ = (pre_decoded_text != test_decoded_text)

            if raw_blocks_differ and _comparison_flags["jsonp_intelligent_comparison"].value:
                if block_name == "response_data":
                    try:
                        response_diff_tree = compare_jsonp_responses(
                            pre_decoded_text,
                            test_decoded_text,
                            jsonp_parser=jsonp_parser,
                            jsonp_parser_options={
                                "allow_identifier_keys": True,
                                "allow_non_portable": True,
                            },
                            timeout=timeout,
                            allow_json=True,
                        )
                    except Exception as exc:
                        logger.warning("Error while trying to build json diff from jsonp reponse_data for test_id={}: {}".format(int(test_id), exc))
                try:
                    block_has_diff, diff_str = compare_jsonp_strings(
                        pre_decoded_text,
                        test_decoded_text,
                        jsonp_parser=jsonp_parser,
                        jsonp_parser_options={
                            "allow_identifier_keys": True,
                            "allow_non_portable": True,
                        },
                        timeout=timeout,
                        allow_json=True,
                        diff_context=diff_context,
                    )
                except Exception as exc:
                    logger.warning("Error while trying to compare jsonps as objects in %s for test_id=%s: %s", block_name, int(test_id), exc)
                    block_has_diff, diff_str = compare_text(pre_decoded_text, test_decoded_text, diff_context=diff_context)
                    tags.add("invalid_jsonp")
            else:
                block_has_diff, diff_str = compare_text(pre_decoded_text, test_decoded_text, diff_context=diff_context)

        elif diff_method == DiffMethods.json_diff:
            block_has_diff, diff_str = compare_dicts(raw_pre_block, raw_test_block, diff_context=diff_context)

        elif diff_method == DiffMethods.text_diff:
            pre_decoded_text = decode_text(raw_pre_block)
            test_decoded_text = decode_text(raw_test_block)
            block_has_diff, diff_str = compare_text(pre_decoded_text, test_decoded_text, diff_context=diff_context)

        else:
            logger.warning(
                'Unknown diff method, comparing it as text.\n'
                'Item: {key}, diff method: {diff_method}'.format(
                    key=block_name,
                    diff_method=diff_method
                )
            )
            pre_decoded_text = decode_text(raw_pre_block)
            test_decoded_text = decode_text(raw_test_block)
            block_has_diff, diff_str = compare_text(pre_decoded_text, test_decoded_text, diff_context=diff_context)

        diff += wrap_diff(block_name, diff_str)

        if block_has_diff:
            has_diff = True
            tags.add(block_name)
            if block_name == "response_data" and not response_diff_tree:
                response_diff_tree = {"response_data": {"Failed to fetch json diff from response_data": None}}

    result = SingleComparisonResult(
        test_id=str(int(test_id)),
        has_diff=has_diff,
        pre_code=int(decode_text(pre_dict['code']['diff_data'])),
        test_code=int(decode_text(test_dict['code']['diff_data'])),
        handler=decode_text(pre_dict['handler']['diff_data']),
        diff_tags=tags,
        diff=diff,
        request=decode_text(pre_dict['request']['diff_data']),
        unique_changed_logs=unique_changed_logs,
        log_statistics=log_statistics,
        response_diff_tree=response_diff_tree
    )

    return result


def _compare_logs_statistics(log_statistics, statistics_aggregation_config):
    statistics = {}
    for item in ('pre', 'test'):
        logging.debug('Start aggregate %s statistics', item)
        statistics[item] = aggregate_statistics(
            statistics=log_statistics[item],
            config=statistics_aggregation_config
        )
        if statistics[item]:
            move_hit_p_match_fail_bits(statistics[item])

    logging.debug('Start diff statistics')
    return StatisticsComparisonResult(
        pre=statistics['pre'],
        test=statistics['test'],
        diff_in_percents=get_dicts_diff(
            statistics['pre'], statistics['test'],
            diff_function=diff_in_percents
        ),
        diff_abs=get_dicts_diff(
            statistics['pre'], statistics['test'],
            diff_function=lambda pre, test: test - pre
        )
    )


CLI_DIFF_DIRNAME = 'cli_diff'


class MultiprocessingTestsLimiter(object):
    LIMIT_PER_GROUP = 500

    def __init__(self, sync_manager):
        # these dicts are used to limit the number of tests for each group presented in web-report
        # since Manager().dict() can't have efficient nested mutable structures, we use such strange dicts
        self._web_report_test_groups_count = sync_manager.dict()  # key is (handler, status, tag), value is number of such tests
        self._web_report_test_groups_count_lock = sync_manager.Lock()
        self._web_report_test_ids = sync_manager.dict()  # key is test_id, value is True for each key

    def _try_add_test_id_tag(self, test_id, handler, failed, tag):
        key = (handler, failed, tag)
        self._web_report_test_groups_count_lock.acquire(True)

        cur_value = self._web_report_test_groups_count.get(key, 0)
        added = False
        if cur_value < self.LIMIT_PER_GROUP:
            added = True
            self._web_report_test_groups_count[key] = cur_value + 1
            self._web_report_test_ids[test_id] = True

        self._web_report_test_groups_count_lock.release()
        return added

    def try_add_test_id(self, test_id, handler, failed, tags):
        if not tags:
            return self._try_add_test_id_tag(test_id, handler, failed, None)

        for tag in tags:
            added = self._try_add_test_id_tag(test_id, handler, failed, tag)
            if added:
                return True
        return False

    def check_test_added(self, test_id):
        return self._web_report_test_ids.get(str(test_id), False)


class ChunksData(object):
    def __init__(self):
        self._chunk_dir = tempfile.mkdtemp(prefix='chunks_data_')
        self._tests_dir = self._make_subdir('tests')
        self._diffs_dir = self._make_subdir('diff')
        self._cli_diffs_dir = self._make_subdir(CLI_DIFF_DIRNAME)
        self._cli_diffs_index_file = os.path.join(self._chunk_dir, 'report_index.txt')
        self._cli_diffs_index_dir = self._make_subdir('report_indices')

        m = Manager()
        # key is request_id, value is a tuple (chunk_name, begin_diff_offset, diff_bytes_count, begin_request_offset, request_bytes_count)
        self._cli_diffs_offsets = m.dict()
        self.tests_limiter = MultiprocessingTestsLimiter(m)

    def _make_subdir(self, subdir):
        path = os.path.join(self._chunk_dir, subdir)
        makedirs_except(path)
        return path

    def add_tests_chunk(self, chunk_results, chunk_name):
        tests_to_write = dict()
        previews_to_write = dict()
        web_report_index = []

        for test_id, result in chunk_results.iteritems():
            added = self.tests_limiter.try_add_test_id(
                test_id=test_id,
                handler=result.handler,
                failed=result.has_diff,
                tags=result.diff_tags
            )
            if added:
                tests_to_write[test_id] = {
                    'status': ('failed' if result.has_diff else 'passed'),
                    'id': int(test_id),
                    'name': str(int(test_id)),
                    'ft_shoot': {
                        'pre.code': result.pre_code,
                        'test.code': result.test_code,
                        'handler': result.handler
                    },
                    'diff': result.diff,
                    'request': result.request,
                }
                previews_to_write[test_id] = {'diff': abbreviate_diff(result.diff)}
                web_report_index.append({
                    'status': ('failed' if result.has_diff else 'passed'),
                    'id': int(test_id),
                    'name': str(int(test_id)),
                    'search': OrderedDict([
                        ('pre.code', result.pre_code),
                        ('test.code', result.test_code),
                        ('handler', result.handler),
                        ('tags', list(result.diff_tags)),
                        ('filter_smart_report', result.filter_smart_report)
                    ]),
                    'diffLinesCount': 20,
                })

        if tests_to_write:
            self._write_tests_chunk(tests_to_write, chunk_name)
            self._write_previews_chunk(previews_to_write, chunk_name)

        return web_report_index

    def _write_tests_chunk(self, chunk_data, chunk_name):
        with open(os.path.join(self._tests_dir, '{}.json'.format(chunk_name)), 'w') as f:
            json.dump(chunk_data, f)

    def _write_previews_chunk(self, chunk_data, chunk_name):
        with open(os.path.join(self._diffs_dir, '{}.json'.format(chunk_name)), 'w') as f:
            json.dump(chunk_data, f)

    def _write_single_request_unified_diff(self, file_obj, diff_data, request_id, handler, pre_code, test_code, tags):
        diff_data_lines = diff_data.split('\n')
        if file_obj.tell() > 0 and diff_data != '':
            file_obj.write('\n\n')
        begin_offset = file_obj.tell()

        if len(diff_data_lines) < 3 or diff_data_lines[0] in (u'@@@LOGS_DATA DIFF:@@@', u'@@@RESPONSE_DATA DIFF:@@@'):
            file_obj.write(diff_data.encode('utf-8'))
            return begin_offset, file_obj.tell()

        if not diff_data_lines[2].startswith('@@') and diff_data_lines[2].endswith('@@'):
            raise RuntimeError('Invalid unified diff', diff_data_lines[:3])

        head, sep, tail = diff_data_lines[2].rpartition('@@')
        tail += ' request.id: {}, handler: {}; pre.code: {}; test.code: {}; tags: {};'.format(
                request_id, handler, pre_code, test_code, ', '.join(tags))
        diff_data_lines[2] = head + sep + tail
        file_obj.write('\n'.join(diff_data_lines).encode('utf-8'))
        return begin_offset, file_obj.tell()

    def _write_single_request(self, file_obj, request_data):
        if file_obj.tell() > 0:
            file_obj.write('\n\n\n')  # valid HTTP requests split is two blank lines
        begin_offset = file_obj.tell()
        file_obj.write(request_data)
        return begin_offset, file_obj.tell()

    def _cli_diffs_file(self, name, full=False):
        base = self._cli_diffs_dir if full else os.path.basename(self._cli_diffs_dir)
        return os.path.join(base, '{}.diff'.format(name))

    def _cli_requests_file(self, name, full=False):
        base = self._cli_diffs_dir if full else os.path.basename(self._cli_diffs_dir)
        return os.path.join(base, '{}.requests'.format(name))

    def _write_unified_diffs_chunk(self, diff_file_obj, requests_file_obj, chunk_results, chunk_name):
        for test_id, result in chunk_results.iteritems():
            if result.has_diff:
                diff_begin_offset, diff_end_offset = self._write_single_request_unified_diff(
                    file_obj=diff_file_obj,
                    diff_data=result.diff,
                    request_id=int(test_id),
                    handler=result.handler,
                    pre_code=result.pre_code,
                    test_code=result.test_code,
                    tags=result.diff_tags
                )
                request_begin_offset, request_end_offset = self._write_single_request(requests_file_obj, result.request)
                self._cli_diffs_offsets[test_id] = (
                    chunk_name,
                    diff_begin_offset,
                    diff_end_offset - diff_begin_offset,
                    request_begin_offset,
                    request_end_offset - request_begin_offset
                )
            else:
                self._cli_diffs_offsets[test_id] = (chunk_name, 0, 0, 0, 0)

    def add_unified_diffs_chunk(self, chunk_results, chunk_name):
        with open(self._cli_diffs_file(chunk_name, full=True), 'wb') as f:
            with open(self._cli_requests_file(chunk_name, full=True), 'wb') as r:
                self._write_unified_diffs_chunk(f, r, chunk_results, chunk_name)

    def _write_single_index_entry(self, file_obj, meta):
        if file_obj.tell() > 0:
            file_obj.write('\n')
        file_obj.write('request.id: {req_id}; status: {st}; handler: {handler}; pre.code: {pc}; test.code: {tc}; tags: {tags};'.format(
            req_id=meta.test_id, handler=meta.handler, pc=meta.pre_code, tc=meta.test_code,
            tags=', '.join(meta.diff_tags), st='failed' if meta.has_diff else 'passed',
        ))
        if meta.test_id in self._cli_diffs_offsets:
            chunk_name, diff_seek_begin, diff_bytes_count, request_seek_begin, request_bytes_count = self._cli_diffs_offsets[meta.test_id]
            if diff_bytes_count != 0:
                file_obj.write(' chunk_path: {path};\n'.format(path=self._cli_diffs_file(chunk_name)))
                file_obj.write('dd if={ifile} ibs=1 skip={skip} count={count} 2> /dev/null\n'.format(ifile=self._cli_diffs_file(chunk_name), skip=diff_seek_begin, count=diff_bytes_count))
            else:
                file_obj.write('\n')
            if request_bytes_count != 0:
                # assert diff_bytes_count != 0, 'this is very weird, handle this exception to current code owners'
                file_obj.write('dd if={ifile} ibs=1 skip={skip} count={count} 2> /dev/null\n'.format(ifile=self._cli_requests_file(chunk_name), skip=request_seek_begin, count=request_bytes_count))
        else:
            file_obj.write('\n')
            logger.debug('Could not find test_id %s in self._cli_diffs_offsets, writing index without paths and requests, full meta dict: %s', meta.test_id, meta)
        return file_obj.tell()

    def add_cli_diffs_index(self, chunks_meta, chunk_name):
        with open(os.path.join(self._cli_diffs_index_dir, str(chunk_name)), 'wb') as f:
            for meta in chunks_meta:
                self._write_single_index_entry(f, meta)

    def merge_cli_diffs_indices(self):
        with open(self._cli_diffs_index_file, 'wb') as f:
            chunk_names = os.listdir(self._cli_diffs_index_dir)
            logger.debug('Found %d chunks: %s', len(chunk_names), chunk_names)
            for chunk_name in sorted(chunk_names):
                with open(os.path.join(self._cli_diffs_index_dir, chunk_name), 'rb') as index_file:
                    data = index_file.read()
                    f.write(data + '\n')
        shutil.rmtree(self._cli_diffs_index_dir)

    def get_tests_dir(self):
        return self._tests_dir

    def get_diffs_dir(self):
        return self._diffs_dir

    def get_cli_diffs_dir(self):
        return self._cli_diffs_dir

    def get_cli_diffs_index(self):
        return self._cli_diffs_index_file

    def clear(self):
        shutil.rmtree(self._chunk_dir)


def add_to_dict(dct, key, value):
    dct.setdefault(key, set())
    dct[key] |= value


class ChunkComparisonResult(object):
    __slots__ = ('test_total_count', 'test_failures_count', 'failed_test_ids', 'pre_codes',
                 'test_codes', 'handlers', 'diff_tags', 'unique_changed_logs', 'log_statistics',
                 'response_diff_tree', 'smart_reports', 'filter_smart_reports')

    def __init__(
        self,
        test_total_count=0, test_failures_count=0, failed_test_ids=None,
        pre_codes=None, test_codes=None, handlers=None, diff_tags=None,
        unique_changed_logs=None, log_statistics=None, response_diff_tree=None,
        smart_reports=None, filter_smart_reports=None
    ):
        self.test_total_count = test_total_count
        self.test_failures_count = test_failures_count
        self.failed_test_ids = failed_test_ids or []

        self.pre_codes = pre_codes or set()
        self.test_codes = test_codes or set()
        self.handlers = handlers or set()

        self.diff_tags = diff_tags or set()

        self.response_diff_tree = response_diff_tree or dict()

        self.smart_reports = smart_reports or dict()
        self.filter_smart_reports = filter_smart_reports or dict()

        self.unique_changed_logs = unique_changed_logs or {}
        self.log_statistics = log_statistics or deepcopy(DEFAULT_LOG_STATISTICS)

    def update(self, comparison_result):
        if isinstance(comparison_result, ChunkComparisonResult):
            self.test_total_count += comparison_result.test_total_count
            self.test_failures_count += comparison_result.test_failures_count
            self.failed_test_ids.extend(comparison_result.failed_test_ids)
            self.pre_codes |= comparison_result.pre_codes
            self.test_codes |= comparison_result.test_codes
            self.handlers |= comparison_result.handlers
            for smart_report, test_ids in comparison_result.smart_reports.items():
                add_to_dict(self.smart_reports, smart_report, test_ids)
            for filter_smart_report, test_ids in comparison_result.filter_smart_reports.items():
                add_to_dict(self.filter_smart_reports, filter_smart_report, test_ids)
        elif isinstance(comparison_result, SingleComparisonResult):
            self.test_total_count += 1
            if comparison_result.has_diff:
                self.test_failures_count += 1
                self.failed_test_ids.append(comparison_result.test_id)
            self.pre_codes.add(comparison_result.pre_code)
            self.test_codes.add(comparison_result.test_code)
            self.handlers.add(comparison_result.handler)
            add_to_dict(self.smart_reports, comparison_result.smart_report, set([comparison_result.test_id]))
            add_to_dict(self.filter_smart_reports, comparison_result.filter_smart_report, set([comparison_result.test_id]))

        self.diff_tags |= comparison_result.diff_tags
        self.response_diff_tree = add_tree_patch(self.response_diff_tree, comparison_result.response_diff_tree)

        for key in comparison_result.unique_changed_logs:
            self.unique_changed_logs.setdefault(key, deepcopy(DEFAULT_UNIQUE_CHANGED_LOGS))
            merge_unique(self.unique_changed_logs[key]['unique'], comparison_result.unique_changed_logs[key]['unique'])
            merge_changed(self.unique_changed_logs[key]['changed'], comparison_result.unique_changed_logs[key]['changed'])

        for response_type in ('pre', 'test'):
            self.log_statistics[response_type]['data'] = merge_dicts(self.log_statistics[response_type]['data'], comparison_result.log_statistics[response_type]['data'])
            self.log_statistics[response_type]['counts'] = sum((self.log_statistics[response_type]['counts'], comparison_result.log_statistics[response_type]['counts'], ), CustomCounter())


def _compare_chunk(
    pre_path, test_path, chunk, chunk_name, chunks_data, statistics_aggregation_config,
    diff_blocks=None, jsonp_parser=None, comparison_flags=None, timeout=4, diff_context=3,
):
    logger.debug('%s: Begin Compare Chunks for %s', current_process().name, chunk_name)
    chunk_comparison_result = ChunkComparisonResult()
    chunk_results = {}

    for idx, test_id in enumerate(chunk):
        logger.debug('%s: processing TestID: %s for %d from %d', current_process().name, test_id, idx, len(chunk))
        comparison_result = _compare_one(pre_path, test_path, test_id, statistics_aggregation_config, diff_blocks, jsonp_parser, comparison_flags, timeout, diff_context=diff_context)
        chunk_results[comparison_result.test_id] = comparison_result
        chunk_comparison_result.update(comparison_result)

    logger.debug('%s: generate report index', current_process().name)
    chunk_web_report_index = chunks_data.add_tests_chunk(chunk_results, chunk_name)
    chunks_data.add_unified_diffs_chunk(chunk_results, chunk_name)
    chunks_data.add_cli_diffs_index(chunk_results.values(), chunk_name)

    logger.debug('%s: finish compare chunks', current_process().name)
    return chunk_comparison_result, chunk_web_report_index


def _compare_chunk_wrapper(*args, **kwargs):
    enable_profiler = 'enable_profiler' in kwargs and kwargs.pop('enable_profiler')
    if enable_profiler:
        profiler = Profile()
        profiler.enable()
    try:
        return _compare_chunk(*args, **kwargs)
    except Exception:
        logger.exception("Failed to compare result chunks")
        raise
    finally:
        if enable_profiler:
            profiler.disable()
            profile_stats = Stats(profiler)
            profile_stats.dump_stats(os.path.join(PROFILE_TMP_DIR, '{}.log'.format(str(uuid.uuid4()))))


def _compare_multiple(
    pre_path, test_path, test_ids, statistics_aggregation_config,
    n_jobs=None, diff_blocks=None, comparison_flags=None, timeout=4, new_paints=True, enable_profiler=True, diff_context=3,
    finish_time=None,
):
    chunks = defaultdict(list)
    jsonp_parser = json.loads
    if not new_paints:
        import demjson
        jsonp_parser = demjson.decode

    for test_id in test_ids:
        chunks[int(test_id) % REPORT_CHUNKS_COUNT].append(test_id)

    chunks_data = ChunksData()  # Chunks names are unique so there is no data race in parallel work with this object

    n_jobs = n_jobs or max(
        int(cpu_count() * 0.8),
        1
    )
    args_list = [
        (
            (pre_path, test_path, chunk, chunk_name, chunks_data, statistics_aggregation_config, diff_blocks, jsonp_parser, comparison_flags, timeout),
            dict(enable_profiler=enable_profiler, diff_context=diff_context)
        )
        for chunk_name, chunk in chunks.items()
    ]

    preload_template('smart_report.txt')
    preload_template('filter_smart_report.txt')

    web_report_index = []
    comparison_result = ChunkComparisonResult()
    logger.debug('Compare threads count: %d', n_jobs)
    logger.debug('Compare jobs count: %d', len(args_list))
    if n_jobs == 1:
        for args, kwargs in args_list:
            chunk_comparison_result, chunk_web_report_index = _compare_chunk_wrapper(*args, **kwargs)
            comparison_result.update(chunk_comparison_result)
            web_report_index.extend(chunk_web_report_index)
    else:
        from concurrent.futures import ProcessPoolExecutor, as_completed
        fs = []
        with ProcessPoolExecutor(max_workers=n_jobs) as process_pool:
            for args, kwargs in args_list:
                fs.append(process_pool.submit(_compare_chunk_wrapper, *args, **kwargs))

            future_timeout = None
            if finish_time:
                future_timeout = (finish_time - datetime.datetime.now() - datetime.timedelta(minutes=10)).total_seconds()

            logger.debug('finish_time: %s, future_timeout: %s', finish_time, future_timeout)

            for f in as_completed(fs, timeout=future_timeout):
                exc = f.exception()
                if exc is not None:
                    raise exc
                chunk_comparison_result, chunk_web_report_index = f.result()
                comparison_result.update(chunk_comparison_result)
                web_report_index.extend(chunk_web_report_index)

    chunks_data.merge_cli_diffs_indices()

    return comparison_result, web_report_index, chunks_data


def compare_multiple(
    pre_path, test_path, test_ids, report, cli_report, statistics_aggregation_config,
    n_jobs=None, diff_blocks=None, comparison_flags=None, timeout=4, new_paints=True, enable_profiler=True, diff_context=3,
    finish_time=None
):
    if test_ids:
        comparison_result, web_report_index, chunks_data = _compare_multiple(
            pre_path, test_path, test_ids, statistics_aggregation_config,
            n_jobs=n_jobs, diff_blocks=diff_blocks, comparison_flags=comparison_flags, timeout=timeout, new_paints=new_paints, enable_profiler=enable_profiler, diff_context=diff_context,
            finish_time=finish_time,
        )
    else:
        comparison_result = ChunkComparisonResult()
        web_report_index = []
        chunks_data = ChunksData()
        chunks_data.merge_cli_diffs_indices()
    report.move_tests_and_diffs_chunks(chunks_data)
    cli_report.move_from_chunks_data(chunks_data)
    chunks_data.clear()

    report.add_log_statistics(comparison_result.log_statistics)
    statistics_comparison_result = _compare_logs_statistics(comparison_result.log_statistics, statistics_aggregation_config)
    logging.debug('End diff statistics, writing to files...')
    report.add_statistics_diff(
        pre=statistics_comparison_result.pre,
        test=statistics_comparison_result.test,
        diff_in_percents=statistics_comparison_result.diff_in_percents,
        diff_abs=statistics_comparison_result.diff_abs,
    )

    return comparison_result, web_report_index, statistics_comparison_result
