import os
import json
from collections import defaultdict
from sandbox.projects.yabs.qa.template_utils import get_template
from enum import Enum
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.compare_response_trees import render_blocks


class ELogChange(Enum):
    added_logs = 'added_logs'
    removed_logs = 'removed_logs'
    changed_logs = 'changed_logs'


class ELogFieldChange(Enum):
    added_fields = '+'
    removed_fields = '-'
    changed_fields = '!'

    max_len_change = 'len'


templates_dir = os.path.join(
    os.path.dirname(__file__),
    '../templates'
)


def _format_diff(number):
    if isinstance(number, float):
        return "{0:+.15}".format(float(number))
    return "{0:+d}".format(number)


def _render_statistics(*args):
    return [
        json.dumps(arg, sort_keys=True, indent=4) if isinstance(arg, dict) else arg
        for arg in args
    ]


template_renderer_dict = {}
zero_diff_report_text = {}


def preload_template(report_template):
    if report_template not in template_renderer_dict:
        renderer = get_template(report_template, templates_dir=templates_dir)
        template_renderer_dict[report_template] = renderer
        if report_template in ('smart_report.txt', 'filter_smart_report.txt'):
            class NoneAttributes(object):
                def __getattribute__(self, __name):
                    return None
            zero_diff_report_text[report_template] = renderer.render(data=make_smart_report_data(NoneAttributes()))


def make_smart_report_text(data, report_template='smart_report.txt'):
    preload_template(report_template)
    template_renderer = template_renderer_dict[report_template]
    text = template_renderer.render(data=data)
    if report_template in zero_diff_report_text and text == zero_diff_report_text[report_template]:
        text = "Zero diff"
    return text


def make_logs_statistics_report_data(statistics_pre, statistics_test, statistics_diff_in_percents, statistics_diff_abs):
    aggregated_statistics = defaultdict(dict)
    for log_name, log_content in statistics_diff_in_percents.iteritems():
        for log_field in log_content:
            diff_in_percents = statistics_diff_in_percents[log_name][log_field]
            diff_abs = statistics_diff_abs[log_name][log_field]
            if diff_in_percents == 0:
                diff_in_percents = ''
                diff_abs = ''
            elif isinstance(diff_in_percents, (int, float)):
                diff_in_percents = _format_diff(float(diff_in_percents))
                diff_abs = _format_diff(diff_abs)
            elif isinstance(diff_in_percents, dict):
                diff_in_percents = {key: _format_diff(float(value)) for key, value in diff_in_percents.items() if value}
                diff_abs = {key: _format_diff(value) for key, value in diff_abs.items() if value}
                if not diff_in_percents:
                    diff_in_percents = ''
                    diff_abs = ''
            aggregated_statistics[log_name][log_field] = _render_statistics(
                statistics_pre.get(log_name, dict()).get(log_field, 0),
                statistics_test.get(log_name, dict()).get(log_field, 0),
                diff_abs,
                diff_in_percents
            )

    return aggregated_statistics


def make_logs_statistics_report_text(aggregated_statistics):
    template = get_template('logs_statistics.html', templates_dir=templates_dir, lstrip_blocks=True, trim_blocks=True)
    return template.render(aggregated_statistics=aggregated_statistics)


def make_smart_report_data(comparison_result):
    unique_changed_logs_global = comparison_result.unique_changed_logs or {}
    diff_tags = comparison_result.diff_tags or set()
    diff_blocks_detailed = {}
    for log_source, smart_diff_dict in unique_changed_logs_global.iteritems():
        diff_blocks_detailed[log_source] = {
            ELogChange.added_logs.name: [],
            ELogChange.removed_logs.name: [],
            ELogChange.changed_logs.name: {},
        }
        unique_dict = smart_diff_dict['unique']
        if unique_dict['pre']:
            diff_blocks_detailed[log_source][ELogChange.removed_logs.name].extend(unique_dict['pre'])
            diff_blocks_detailed[log_source][ELogChange.removed_logs.name].sort()
        if unique_dict['test']:
            diff_blocks_detailed[log_source][ELogChange.added_logs.name].extend(unique_dict['test'])
            diff_blocks_detailed[log_source][ELogChange.added_logs.name].sort()
        changed_dict = smart_diff_dict['changed']
        for log, changed in changed_dict.iteritems():
            diff_blocks_detailed[log_source][ELogChange.changed_logs.name].setdefault(
                log,
                {
                    ELogFieldChange.added_fields.name: [],
                    ELogFieldChange.removed_fields.name: [],
                    ELogFieldChange.changed_fields.name: [],
                    ELogFieldChange.max_len_change.name: 0
                }
            )
            for log_field_change in (ELogFieldChange.added_fields, ELogFieldChange.removed_fields, ELogFieldChange.changed_fields):
                if changed[log_field_change.value]:
                    diff_blocks_detailed[log_source][ELogChange.changed_logs.name][log][log_field_change.name].extend(changed[log_field_change.value])
                    diff_blocks_detailed[log_source][ELogChange.changed_logs.name][log][log_field_change.name].sort()
            if changed[ELogFieldChange.max_len_change.value]:
                diff_blocks_detailed[log_source][ELogChange.changed_logs.name][log][ELogFieldChange.max_len_change.name] = max(
                    diff_blocks_detailed[log_source][ELogChange.changed_logs.name][log][ELogFieldChange.max_len_change.name],
                    changed[ELogFieldChange.max_len_change.value],
                    key=abs
                )
    return {
        'diff_blocks': list(diff_tags),
        'diff_blocks_detailed': diff_blocks_detailed,
        'response_diff': render_blocks(comparison_result.response_diff_tree or {})
    }
