# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure, SandboxException
from sandbox.common.types.task import Status
from sandbox.common.urls import get_task_link
from sandbox.projects.common.build.parameters import ArcadiaPatch

logger = logging.getLogger('YabsServerB2BRerunFuncShoot')
logging.basicConfig(level=logging.INFO)


def is_output_param(task_params, name):
    param_type = getattr(type(task_params), name)
    logger.info('Checking param {} of {}: param_type.__output__={}'.format(name, task_params, param_type.__output__))
    return param_type.__output__


class YabsServerB2BRerunFuncShoot(sdk2.Task):
    name = 'YABS_SERVER_B2B_RERUN_FUNC_SHOOT'

    class Parameters(sdk2.Parameters):

        cmp_task = sdk2.parameters.Task('Func shoot cmp task', required=True)
        description = 'Rerun with new patch'
        arcadia_patch = ArcadiaPatch()

    def check_build_task(self, build_task):
        stage = 'check_build_task'
        logger.info('[{}]: Checking results of build subtask, {}'.format(stage, build_task))
        logger.info('[{}]: BUILD_YABS_SERVER has status {}, resource of BS_RELEASE_TAR is {}'
                    .format(stage, build_task.status, build_task.Parameters.bs_release_tar_resource))
        if build_task.status != Status.SUCCESS or not build_task.Parameters.bs_release_tar_resource:
            error = 'Build task failed' if build_task.status != Status.SUCCESS else 'Can not find BS_RELEASE_TAR'
            error += ', {}'.format(get_task_link(self.Context.build_subtask))
            raise TaskFailure(error)

    def get_shoot_task(self, item='test'):
        cmp_task = self.Parameters.cmp_task
        shoot_task = getattr(cmp_task.Parameters, '{}_task'.format(item)) or getattr(cmp_task.Parameters, '{}_task_id'.format(item))
        if not shoot_task:
            raise SandboxException('Cmp task does not specify {} task'.format(item))
        return shoot_task

    def get_build_task(self, shoot_task):
        resource = shoot_task.Parameters.server_resource
        return sdk2.Task.find(id=resource.task_id).limit(1).first()

    def run_build_server(self):
        stage = 'run_build_server'
        shoot_task = self.get_shoot_task('test')
        logger.info("[{}]: Test shoot task to be re-shooted: #{}".format(stage, shoot_task.id))
        build_task = self.get_build_task(shoot_task)
        logger.info("[{}]: Build task used to create test yabs-server: #{}".format(stage, build_task.id))
        params = {k: v for k, v in build_task.Parameters if not is_output_param(build_task.Parameters, k)}
        params['arcadia_patch'] = self.Parameters.arcadia_patch
        logger.info('[{}]: Run subtask with params {}'.format(stage, params))

        cloned_build_task = type(build_task)(
            self,
            description='Build for func shoot rerun from task #{}'.format(self.id),
            **params
        ).enqueue()
        logger.info('[{}]: Cloned build task: {}'.format(stage, cloned_build_task))
        return cloned_build_task

    def run_shoot_task(self):
        stage = 'run_shoot_task'
        shoot_task = self.get_shoot_task('test')
        logger.info("[{}]: Test shoot task to be re-shooted: #{}".format(stage, shoot_task.id))

        build_task = sdk2.Task[self.Context.build_subtask]
        self.check_build_task(build_task)
        params = {k: v for k, v in shoot_task.Parameters if not is_output_param(shoot_task.Parameters, k)}
        params['server_resource'] = build_task.Parameters.bs_release_tar_resource
        logger.info('[{}]: Run subtask with params {}'.format(stage, params))

        cloned_shoot_task = type(shoot_task)(
            self,
            description='Re-run of test func shoot from #{}'.format(self.id),
            **params
        ).enqueue()
        logger.info('[{}]: Cloned shoot task: {}'.format(stage, cloned_shoot_task))
        return cloned_shoot_task

    def run_cmp_task(self):
        stage = 'run_cmp_task'
        cmp_task = self.Parameters.cmp_task
        shoot_task = sdk2.Task[self.Context.shoot_subtask]
        logger.info('[{}]: Shoot subtask [{}] has status {}'.format(stage, shoot_task, shoot_task.status))
        if shoot_task.status != Status.SUCCESS:
            raise TaskFailure('<a href="{}" target="_blank">Shoot task failed</a>'.format(get_task_link(shoot_task.id)))

        params = {k: v for k, v in cmp_task.Parameters if not is_output_param(cmp_task.Parameters, k)}
        params['test_task'] = shoot_task.id
        logger.info('[{}]: Run subtask with params {}'.format(stage, params))

        cloned_cmp_task = type(cmp_task)(
            self,
            description='Run cmp task from #{}'.format(self.id),
            **params
        ).enqueue()
        logger.info('[{}]: Cloned cmp task: {}'.format(stage, cloned_cmp_task))
        return cloned_cmp_task

    def on_execute(self):
        if not self.Parameters.cmp_task:
            raise TaskFailure('Cmp task must be set!')
        with self.memoize_stage.build_stage:
            self.Context.build_subtask = self.run_build_server().id
            logger.info('Waiting for build')
            raise sdk2.WaitTask(self.Context.build_subtask,
                                Status.Group.FINISH + Status.Group.BREAK,
                                wait_all=True)
        with self.memoize_stage.shoot_stage:
            self.Context.shoot_subtask = self.run_shoot_task().id
            logger.info('Waiting for shoot')
            raise sdk2.WaitTask(self.Context.shoot_subtask,
                                Status.Group.FINISH + Status.Group.BREAK,
                                wait_all=True)
        with self.memoize_stage.cmp_stage:
            cmp_subtask = self.run_cmp_task().id
            self.set_info('Comparison delegated to <a href="{task_url}" target="_blank">another CMP task</a>'.format(
                task_url=get_task_link(cmp_subtask)),
                do_escape=False,
            )
