

from sandbox import sdk2
from sandbox.projects.yabs.qa.constants import ENGINE_ROLES, META_MODES


ALL = [
    (role, meta_mode)
    for role in ENGINE_ROLES
    for meta_mode in META_MODES
]


def field_name(prefix, role, meta_mode):
    return '{prefix}_{role}_{meta_mode}'.format(prefix=prefix, role=role, meta_mode=meta_mode)


class BaseResources(sdk2.Parameters):
    base_resources = sdk2.parameters.Resource('Base resources for {role} at {meta_mode}', multiple=True)


class AggregateResult(sdk2.Parameters):
    total_size = sdk2.parameters.Integer('Summary base size for {role} at {meta_mode}')
    base_sizes = sdk2.parameters.JSON('Base size by base for {role} at {meta_mode}')
    chkdb_resources = sdk2.parameters.JSON('Chkdb resources by base for {role} at {meta_mode}')
    chkdb_without_hash_resources = sdk2.parameters.JSON('Chkdb without hashes resources by base for {role} at {meta_mode}')


class YabsServerBaseSizeAggregate(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        for role, meta_mode in ALL:
            sdk2.helpers.set_parameter(
                field_name('input', role, meta_mode),
                BaseResources(suffix=field_name('', role, meta_mode), label_substs={'meta_mode': meta_mode, 'role': role})
            )

        with sdk2.parameters.Output:
            for role, meta_mode in ALL:
                sdk2.helpers.set_parameter(
                    field_name('aggregation_result', role, meta_mode),
                    AggregateResult(suffix=field_name('', role, meta_mode), label_substs={'meta_mode': meta_mode, 'role': role})
                )

    def aggregate_bases(self, base_resources):
        total = 0
        base_sizes = {}
        chkdb_resources = {}
        chkdb_without_hash_resources = {}

        for base_resource in base_resources:
            tag = base_resource.tag
            size = int(base_resource.unpacked_size)
            total += size
            base_sizes[tag] = size

            chkdb = getattr(base_resource, 'chkdb_resource_id', None)
            chkdb_without_hash = getattr(base_resource, 'chkdb_without_hash_resource_id', None)
            if chkdb:
                chkdb_resources[tag] = int(chkdb)
            if chkdb_without_hash:
                chkdb_without_hash_resources[tag] = int(chkdb_without_hash)

        return total, base_sizes, chkdb_resources, chkdb_without_hash_resources

    def on_execute(self):
        for role, meta_mode in ALL:
            base_resources = getattr(self.Parameters, field_name('base_resources', role, meta_mode)) or []
            total, base_sizes, chkdb_resources, chkdb_without_hash_resources = self.aggregate_bases(base_resources)

            with self.memoize_stage[field_name('set_output', role, meta_mode)]:
                setattr(self.Parameters, field_name('total_size', role, meta_mode), total)
                setattr(self.Parameters, field_name('base_sizes', role, meta_mode), base_sizes)
                setattr(self.Parameters, field_name('chkdb_resources', role, meta_mode), chkdb_resources)
                setattr(self.Parameters, field_name('chkdb_without_hash_resources', role, meta_mode), chkdb_without_hash_resources)
