# -*- coding: utf-8 -*-

import logging
from sandbox import sdk2

from sandbox.projects.yabs.bases.keys import DB_SIZE_CTX_KEY, CHKDB_CTX_KEY, CHKDB_WITHOUT_HASH_CTX_KEY
from sandbox.projects.yabs.qa.compare import compare_chkdb, compare_db_sizes
from sandbox.projects.yabs.qa.resource_types import YabsDbSmartMetaReport
from sandbox.projects.common import file_utils as fu
from sandbox.projects.yabs.qa.tasks.base_compare_task.task import BaseCompareTask
from sandbox.projects.yabs.qa.tasks.base_compare_task.parameters import BaseCompareTaskParameters
from sandbox.projects.yabs.qa.utils.resource import sync_resource
from sandbox.projects import resource_types
from sandbox.common.types.misc import NotExists

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace_calls, trace_entry_point
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.generic import new_resource
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sdk2 import make_resource_ready

from sandbox.projects.yabs.qa.tasks.YabsServerBaseSizeAggregate import ALL, field_name
from sandbox.projects.YabsServerDBSizeCmp import _join_bases, compare_db_data, _get_db_sizes_dict


class YabsServerBaseSizeCmp(BaseCompareTask):

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 120 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Context):
        has_diff = None

    class Parameters(BaseCompareTaskParameters):
        max_diff_percent = sdk2.parameters.Float('Max diff (%)', default=0.1)
        compare_same_bases = sdk2.parameters.Bool('Compare the same bases from pre and test', default=False)

        use_chkdb_without_hash = sdk2.parameters.Bool('Use chkdb without hash', default=True)
        ignored_lines = sdk2.parameters.JSON(
            'Ignored lines (feature under development)',
            default=[],
            description='List of regular expressions ignored in chkdb output (in `diff -I` syntax).'
        )
        shard_key = sdk2.parameters.String('Shard key')

    @trace_calls
    def compare_bases(self, pre_task, test_task, max_diff_percent):
        has_diff = False
        size_reports = []
        chkdb_reports = []
        chkdb_key = 'chkdb_resources' if self.Parameters.use_chkdb_without_hash else 'chkdb_without_hash_resources'
        for role, meta_mode in ALL:
            pre_base_sizes = getattr(pre_task.Parameters, field_name('base_sizes', role, meta_mode))
            test_base_sizes = getattr(test_task.Parameters, field_name('base_sizes', role, meta_mode))

            if not pre_base_sizes or not test_base_sizes:
                continue

            pre_base_chkdb = getattr(pre_task.Parameters, field_name(chkdb_key, role, meta_mode))
            test_base_chkdb = getattr(test_task.Parameters, field_name(chkdb_key, role, meta_mode))

            _has_size_diff, size_report = compare_db_sizes(pre_base_sizes, test_base_sizes, max_diff_percent)
            _has_chkdb_diff, chkdb_report = compare_chkdb(pre_base_chkdb, test_base_chkdb, sync_resource)
            has_diff = has_diff or _has_size_diff or (_has_chkdb_diff and not self.Parameters.use_chkdb_without_hash)
            size_reports.append('=== Base size change {role} {meta_mode} ===\n{report}\n'.format(role=role, meta_mode=meta_mode, report=size_report))
            chkdb_reports.append('=== Chkdb change {role} {meta_mode} ===\n{report}\n'.format(role=role, meta_mode=meta_mode, report=chkdb_report))

        return has_diff, '\n'.join(size_reports + chkdb_reports)

    @trace_calls
    def compare_bases_legacy(self, pre_task, test_task, max_diff_percent, compare_same_bases):
        pre_ctx = get_legacy_context(pre_task)
        test_ctx = get_legacy_context(test_task)

        ctx_key = CHKDB_WITHOUT_HASH_CTX_KEY if self.Parameters.use_chkdb_without_hash else CHKDB_CTX_KEY
        joint_bases = _join_bases(pre_ctx, test_ctx, compare_same_bases)

        logging.info('Will compare bases: %s', joint_bases)

        return compare_db_data(
            _get_db_sizes_dict(pre_ctx, joint_bases),
            _get_db_sizes_dict(test_ctx, joint_bases),
            getattr(pre_ctx, ctx_key),
            getattr(test_ctx, ctx_key),
            max_diff_percent,
            sync_resource,
            chkdb_diff_is_diff=not self.Parameters.use_chkdb_without_hash,
        )

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        if any(task.type.name != 'YABS_SERVER_BASE_SIZE_AGGREGATE' for task in (self.pre_task, self.test_task)):
            has_diff, report = self.compare_bases_legacy(self.pre_task, self.test_task, self.Parameters.max_diff_percent, self.Parameters.compare_same_bases)
        else:
            has_diff, report = self.compare_bases(self.pre_task, self.test_task, self.Parameters.max_diff_percent)

        fu.write_file('diff_resource.txt', report)
        diff_resource = new_resource(
            YabsDbSmartMetaReport,
            task=self,
            description='Smart meta report resource',
            path='diff_resource.txt',
            test_name=self.Parameters.test_name,
            arcanum_review_id=self.Context.arcanum_review_id if self.Context.arcanum_review_id is not NotExists else self.Parameters.arcanum_review_id,
            shard_key=self.Parameters.shard_key,
            has_diff=has_diff
        )
        make_resource_ready(diff_resource)

        fu.write_file('report.txt', report)
        new_resource(resource_types.PLAIN_TEXT, self, 'Yabs server B2B db_size test diff', 'report.txt', ttl=42)

        with self.memoize_stage.set_output:
            self.set_info(report)
            self.Context.has_diff = has_diff
            self.Parameters.has_diff = has_diff
            self.Context.report = report

    @property
    def diff_resource_type(self):
        return 'YABS_DB_SMART_META_REPORT'

    @property
    def diff_resource_search_attributes(self):
        return {'shard_key': self.Parameters.shard_key}


class LegacyContext(object):
    def __init__(self, **kwargs):
        for k, v in kwargs.items():
            setattr(self, k, v)


def get_legacy_context(task):
    if task.type.name == 'YABS_SERVER_DB_SIZE_AGGREGATE':
        return task.Context
    chkdb_resources = {}
    chkdb_without_hash_resources = {}
    base_sizes = {}
    for role, meta_mode in ALL:
        chkdb_resources.update(getattr(task.Parameters, field_name('chkdb_resources', role, meta_mode)))
        chkdb_without_hash_resources.update(getattr(task.Parameters, field_name('chkdb_without_hash_resources', role, meta_mode)))
        base_sizes.update(getattr(task.Parameters, field_name('base_sizes', role, meta_mode)))
    return LegacyContext(**{
        CHKDB_CTX_KEY: chkdb_resources,
        CHKDB_WITHOUT_HASH_CTX_KEY: chkdb_without_hash_resources,
        DB_SIZE_CTX_KEY: base_sizes,
    })
