import collections
import datetime
import json
import logging

from sandbox import sdk2
from sandbox.common import rest
from sandbox.projects.yabs.qa.brave_tests.common import TestType
from sandbox.projects.yabs.qa.brave_tests.precommit_checks import get_green_revision_from_precommit_checks
from sandbox.projects.yabs.qa.brave_tests.trunk import get_green_revision_from_trunk
from sandbox.projects.yabs.qa.solomon.mixin import (
    SolomonTaskMixin,
    SolomonTaskMixinParameters,
)
from sandbox.projects.yabs.qa.utils.arcadia import get_revision_datetime


logger = logging.getLogger(__name__)


class YabsServerBraveTestsBaseRevisionMonitoring(SolomonTaskMixin, sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(SolomonTaskMixinParameters, sdk2.Parameters):
        solomon_project = SolomonTaskMixinParameters.solomon_project(default="yabs_testing")
        solomon_cluster = SolomonTaskMixinParameters.solomon_cluster(default="brave_tests")
        solomon_service = SolomonTaskMixinParameters.solomon_service(default="base_revision")
        solomon_token_vault_name = SolomonTaskMixinParameters.solomon_token_vault_name(default="robot-yabs-cs-sb-mon-solomon-token")

    def on_execute(self):
        sandbox_client = rest.Client()

        base_revisions = collections.defaultdict(dict)
        for test_type in TestType:
            if test_type == TestType.SANITIZE:
                continue
            trunk_base_revision, _ = get_green_revision_from_trunk(0, test_type)
            base_revisions[test_type]["trunk"] = trunk_base_revision

            precommit_base_revision, _ = get_green_revision_from_precommit_checks(sandbox_client, test_type)
            base_revisions[test_type]["precommit"] = precommit_base_revision

        base_revisions[TestType.SANITIZE] = base_revisions[TestType.FT]
        logger.debug("Base revisions: %s", base_revisions)

        metrics = collect_metrics(base_revisions)
        self.solomon_push_client.add(metrics)


def collect_metrics(base_revisions):
    metrics = []

    head_revision_datetime = datetime.datetime.utcnow()
    logger.debug("HEAD revision datetime: %s", head_revision_datetime)

    for test_type in TestType:
        trunk_base_revision = base_revisions[test_type]["trunk"]
        trunk_base_revision_datetime = get_revision_datetime(trunk_base_revision)
        trunk_base_revision_delay = head_revision_datetime - trunk_base_revision_datetime

        best_base_revision = trunk_base_revision
        best_base_revision_delay = trunk_base_revision_delay

        precommit_base_revision = base_revisions[test_type]["precommit"]
        if precommit_base_revision:
            precommit_base_revision_datetime = get_revision_datetime(precommit_base_revision)
            precommit_base_revision_delay = head_revision_datetime - precommit_base_revision_datetime
            best_base_revision = max(trunk_base_revision, precommit_base_revision)
            best_base_revision_delay = min(trunk_base_revision_delay, precommit_base_revision_delay)
        else:
            precommit_base_revision_datetime = None
            precommit_base_revision_delay = None

        logger.debug(json.dumps({
            "test_type": test_type.value,
            "trunk": {
                "revision": trunk_base_revision,
                "revision_datetime": trunk_base_revision_datetime.strftime("%c") if trunk_base_revision_datetime else None,
                "revision_delay": trunk_base_revision_delay.total_seconds() if trunk_base_revision_delay else None,
            },
            "precommit": {
                "revision": precommit_base_revision,
                "revision_datetime": precommit_base_revision_datetime.strftime("%c") if precommit_base_revision_datetime else None,
                "revision_delay": precommit_base_revision_delay.total_seconds() if precommit_base_revision_delay else None,
            },
            "best_delay": best_base_revision_delay.total_seconds(),
        }, indent=2))

        metrics += [
            {
                "labels": {
                    "sensor": "delay",
                    "test_type": test_type.value,
                    "source": "trunk",
                },
                "value": int(trunk_base_revision_delay.total_seconds()),
            },
            {
                "labels": {
                    "sensor": "delay",
                    "test_type": test_type.value,
                    "source": "trunk_or_precommit",
                },
                "value": int(best_base_revision_delay.total_seconds()),
            },
            {
                "labels": {
                    "sensor": "base_revision",
                    "test_type": test_type.value,
                    "source": "trunk",
                },
                "value": int(trunk_base_revision),
            },
            {
                "labels": {
                    "sensor": "base_revision",
                    "test_type": test_type.value,
                    "source": "trunk_or_precommit",
                },
                "value": int(best_base_revision),
            },
        ]

    return metrics
