# -*- coding: utf-8 -*-
import logging
from datetime import timedelta, datetime

from sandbox import common, sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.common.yabs.server.db.yt_bases import prepare_yt_env, YT_POOL, YT_PROXY
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_YT_DIGEST
from sandbox.projects.yabs.qa.utils.general import get_yt_path_html_hyperlink
from sandbox.projects.yabs.qa.utils import yt_utils


# BSSERVER-4844
IMPORT_DIGEST_SKIP_LIST = [
    'HostOptions.data',
    'IStat03MTime.data',
    'IStat01MTime.data',
    # SHMDUTY-66 temporary before stability is fixed by serval@
    'AutobudgetOrderExperimentWeekStat.data',
    'MasterMTime.data',
    # BSSERVER-9160 BSSERVER-9161 emergency measures
    'BroadMatch.data',
    'DSPCreative.data',
    'TNSCategoryMapping.data',
    'SSPOrderBlackList.data',
    'SSPOrderWhiteList.data',
    'YTBusinessRules.data',
    'AdaptiveBannerSizes.data',
    'YTBroadMatchLimits.data',
    'PortalTrustedBanners.data',
    'InterestsMapping.data',
    'YtDump.data',
    'make_bases_log',
    'calculation_log',
    'BannerCompanyDataErrors',
    'BannerUpdateBannerErrors',
    'CaesarBannerErrors',
    'CaesarCountersErrors',
    'CaesarBannersExportErrors',
    'LmServiceBase',
    'LmServiceCounters',
]
FOLDERS_SKIP_LIST = []
COLUMNS_SKIP_LIST = ["_Row"]


class YabsServerCalculateCSImportDigest(sdk2.Task):  # pylint: disable=R0904

    name = 'YABS_SERVER_CALCULATE_CS_IMPORT_DIGEST'

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024
        environments = (
            PipEnvironment('yandex-yt', use_pre=True),
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = timedelta(hours=3).total_seconds()

        yt_pool = sdk2.parameters.String('YT pool', default=YT_POOL)

        with sdk2.parameters.Group('Digest tool options') as digest_tool_options:
            digest_calculator = sdk2.parameters.LastReleasedResource('Digest tool resource', resource_type=YABS_SERVER_YT_DIGEST)
            import_destination_path = sdk2.parameters.String('cs_import destination path to calculate digest')
            intermediate_hashes_table = sdk2.parameters.String('A name of output table for intermediate hashes', default='output_inter_hashes')
            skip_list = sdk2.parameters.List('Skip the tables during calculating yt_digest', required=False, default=IMPORT_DIGEST_SKIP_LIST)
            skip_folders = sdk2.parameters.List('Skip tables inside following folders during calculating yt_digest', required=False, default=FOLDERS_SKIP_LIST)
            skip_columns = sdk2.parameters.List('Skip following table columns during calculating yt_digest', required=False, default=COLUMNS_SKIP_LIST)
            skip_tmp_tables = sdk2.parameters.Bool('Add tmp tables to skip list of calculating yt_digest', default=True)
            ignore_attributes = sdk2.parameters.Bool('Ignore tables attributes', default=False)
            hashes_history_dir = sdk2.parameters.String('Directory to store intermediate hashes history', default="")

    class Context(sdk2.Task.Context):
        digest = None
        intermediate_hashes_full_path = None

    def on_execute(self):
        from yt.wrapper import YtClient, ypath_split

        def ensure_not_slash_terminated(s):
            if s.endswith('/'):
                return s[:-1]
            else:
                return s

        logger = logging.getLogger(__name__)

        node_path = ensure_not_slash_terminated(self.Parameters.import_destination_path)
        intermediate_hashes_table = self.Parameters.intermediate_hashes_table
        intermediate_hashes_full_path = node_path + '/' + intermediate_hashes_table
        need_to_save_intermediate_hashes_table = intermediate_hashes_table != ''
        digest_calculator_path = str(sdk2.ResourceData(self.Parameters.digest_calculator).path)

        self.set_info('Calculating digest for node: {0}'.format(get_yt_path_html_hyperlink(YT_PROXY, node_path)), do_escape=False)
        if need_to_save_intermediate_hashes_table:
            self.set_info('Output intermediate hashes table: {0}'.format(get_yt_path_html_hyperlink(YT_PROXY, intermediate_hashes_full_path)), do_escape=False)
            self.Context.intermediate_hashes_full_path = intermediate_hashes_full_path
        else:
            self.set_info('Output intermediate hashes table was not generated', do_escape=False)
        logger.info('Syncing digest_calculator resource')

        cmd = [
            digest_calculator_path,
            '--prefix', node_path,
            '--proxy', YT_PROXY,
        ]
        if need_to_save_intermediate_hashes_table:
            cmd += ['--intermediate-hashes-table', intermediate_hashes_table]

        yt_token = sdk2.Vault.data('robot-yabs-cs-sbjail', 'yabscs_jailed_yt_token')
        yt_client = YtClient(proxy=YT_PROXY, token=yt_token)
        tmp_cache_path = yt_utils.create_tmp_node(
            yt_client,
            node_path,
            ttl=timedelta(hours=1).total_seconds()
        )

        skip_list = self.Parameters.skip_list
        if self.Parameters.skip_tmp_tables:
            skip_list.append(ypath_split(tmp_cache_path)[-1])

        for skip_table in skip_list:
            cmd += ['--skip', skip_table]

        for skip_folder in self.Parameters.skip_folders:
            cmd += ['--skip-folder', skip_folder]

        for skip_column in self.Parameters.skip_columns:
            cmd += ['--skip-column', skip_column]

        if self.Parameters.ignore_attributes:
            cmd.append('--ignore-attributes')

        env = prepare_yt_env(yt_token, tmp_cache_path, self.Parameters.yt_pool)

        with sdk2.helpers.ProcessLog(self, logger='yt_digest') as log, sdk2.helpers.ProcessRegistry:
            try:
                logger.info('Running digest_calculator on node {0}'.format(node_path))
                output = sdk2.helpers.subprocess.check_output(cmd, stderr=log.stderr, env=env)
                self.Context.digest = output.strip()
                logger.info('Digest: %s', self.Context.digest)
                if self.Parameters.hashes_history_dir:
                    self.copy_intermediate_hashes(
                        yt_client,
                        node_path,
                        intermediate_hashes_full_path,
                        self.Parameters.hashes_history_dir,
                        self.Context.digest
                    )
            except sdk2.helpers.subprocess.CalledProcessError as error:
                error_message = 'yt_digest exited with code {0}'.format(error.returncode)
                raise common.errors.TaskFailure(error_message)

        yt_client.remove(tmp_cache_path, recursive=True, force=True)

    # Copy intermediate hashes table to history directory for longer storage
    def copy_intermediate_hashes(
        self,
        yt_client,
        intermediate_hashes_dir,
        intermediate_hashes_full_path,
        hashes_history_dir,
        digest,
    ):
        target_hashes_path = "{dir}/{task_id}_{datetime}_{digest}".format(
            task_id=intermediate_hashes_dir.rpartition('/')[-1],
            dir=hashes_history_dir,
            datetime=datetime.utcnow().strftime("%Y-%m-%d_%H:%M:%S"),
            digest=digest
        )
        self.set_info('Copy intermediate hashes table to: {0}'.format(target_hashes_path))
        yt_utils.yt_copy(
            yt_client,
            intermediate_hashes_full_path,
            target_hashes_path
        )
        yt_utils.set_yt_node_ttl(
            target_hashes_path,
            timedelta(days=30).total_seconds(),
            yt_client
        )
