import urlparse
from urllib import urlencode
import posixpath
import logging

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

logger = logging.getLogger(__name__)

TEST_TYPES = {'configure', 'build', 'style', 'small', 'medium', 'large', 'te_job'}
YABS_TEST_TYPES = TEST_TYPES - {'large', 'te_job'}

HTTP_TIMEOUT = 120
CI_URL = 'https://ci.yandex-team.ru'
CI_API_URL = urlparse.urljoin(CI_URL, posixpath.join('api', 'v1.0/'))
CA_CERT_PATH = '/etc/ssl/certs/YandexInternalRootCA.pem'


def create_session(ci_token=None):
    session = requests.Session()

    retry = Retry(
        total=10,
        backoff_factor=1.0,
        status_forcelist=(429,)
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount(CI_API_URL, adapter)

    if ci_token is not None:
        session.headers['Authorization'] = 'OAuth {}'.format(ci_token)
    else:
        logger.warn('CI token was not passed. Will make requests to CI without authorization.')

    session.verify = CA_CERT_PATH

    return session


def get_tests_finish_timestamp(revision, session):
    url = urlparse.urljoin(CI_API_URL, posixpath.join('events', str(revision)))
    logger.debug('Url: %s', url)
    r = session.get(
        url=url,
        timeout=HTTP_TIMEOUT
    )
    r.raise_for_status()
    data = r.json()

    finish_timestamps = {
        test_type: test_data['finish_timestamp']
        for test_type, test_data
        in data['statistics'].iteritems()
        if test_type in YABS_TEST_TYPES
    }
    logger.debug('finish timestamps: {}'.format(finish_timestamps))

    if None in finish_timestamps.values():
        return None

    return max(finish_timestamps.values())


def get_tests_count(revision, project_id, session):
    url_params = {
        'revision': revision,
        'project_id': project_id,
    }
    # Trailing slash is important
    url_without_params = urlparse.urljoin(CI_API_URL, 'timeline/')

    scheme, netloc, path, params, _, fragment = list(urlparse.urlparse(url_without_params))
    query_string = urlencode(url_params)

    url = urlparse.urlunparse((scheme, netloc, path, params, query_string, fragment))
    logger.debug('Url: %s', url)

    r = session.get(
        url=url,
        timeout=HTTP_TIMEOUT
    )
    r.raise_for_status()
    return r.json()


def get_ci_check_url_ui(paths, test_types=YABS_TEST_TYPES, branch_name='trunk'):
    parsed_base_url = urlparse.urlparse(CI_URL)
    scheme, netloc = parsed_base_url.scheme, parsed_base_url.netloc
    path = 'tests'
    query_params = {
        'branch': branch_name,
        'type': ','.join(test_types),
        'path': ','.join(paths),
    }
    query_string = urlencode(query_params)
    return urlparse.urlunparse((scheme, netloc, path, None, query_string, None))


def get_ci_check_url(revision, test_path, only_important_diffs=0):
    url_params = {
        'only_important_diffs': only_important_diffs,
        'test_path': test_path,
    }
    url_without_params = urlparse.urljoin(base=CI_URL, url=posixpath.join('commit', str(revision)))
    scheme, netloc, path, params, _, fragment = list(urlparse.urlparse(url_without_params))
    query_string = urlencode(url_params)

    return urlparse.urlunparse((scheme, netloc, path, params, query_string, fragment))
