import logging
import subprocess
import textwrap

from library.python import resource
from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_GCDA_COVERAGE, BS_RELEASE_TAR, YABS_SERVER_GCOV_COVERAGE


class YabsServerCollectGcovCoverage(BaseBinTask):
    class Requirements(sdk2.Requirements):
        cores = 2
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseBinTask.Parameters):
        with sdk2.parameters.Group('Coverage source') as coverage_source:
            gcdas_file = sdk2.parameters.Resource(".tar.gz archive with .gcda coverage data",
                                                  resource_type=YABS_SERVER_GCDA_COVERAGE,
                                                  required=True)
            coverage_build = sdk2.parameters.Resource("yabs-server build containing .gcno files",
                                                      resource_type=BS_RELEASE_TAR, required=True)
            sources_url = sdk2.parameters.ArcadiaUrl("Override SVN arcadia url for acquiring source code")

    def on_execute(self):
        with open("libpreload.so", "wb") as f:
            f.write(resource.find("/libpreload-fakeopen.so"))
        with open("script.bash", "wb") as f:
            f.write(resource.find("/construct_gcov_coverage.bash"))
        with arcadia_sdk.mount_arc_path(self.Parameters.sources_url, use_arc_instead_of_aapi=True,
                                        fetch_all=False) as arcadia_path:
            coverage_build_path = sdk2.ResourceData(self.Parameters.coverage_build).path
            gcdas_file_path = sdk2.ResourceData(self.Parameters.gcdas_file).path
            logging.info("Arcadia mounted at: %s", (arcadia_path,))
            with sdk2.helpers.ProcessLog(self, 'bash') as bash_context:
                subprocess.check_output(["bash", "script.bash", arcadia_path, gcdas_file_path, coverage_build_path],
                                        stderr=bash_context.stderr)
        coverage_resource = YABS_SERVER_GCOV_COVERAGE(self, "Yabs .gcov coverage", "yabs-server-gcov-coverage")
        sdk2.ResourceData(coverage_resource).ready()
        resource_a = textwrap.dedent(
            f"""
            <a
            href='https://proxy.sandbox.yandex-team.ru/{coverage_resource.id}/coverage.html'>
            Coverage collected here!
            </a>"""
        )
        self.set_info(resource_a, do_escape=False)
