import json

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.common.errors import TaskError
from sandbox.projects.common.yql import run_query
from sandbox.projects.common.yabs.server.util.general import try_get_from_vault
from sandbox.projects.yabs.qa.solomon.mixin import SolomonTaskMixin, SolomonTaskMixinParameters
from sandbox.projects.yabs.qa.bases.sample_tables.parameters import SamplingStrategyParameter


YQL_QUERY = """
    select
        ProductType,
        count(*)
    from
        {proxy}.`{prefix}/primary_ammo/0/event`
    group by ProductType
"""


class YabsServerCollectProducts(SolomonTaskMixin, sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        ram = 4096
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

        environments = [
            environments.PipEnvironment('yql', version='1.2.91'),
            environments.PipEnvironment('yandex-yt', use_wheel=True),
        ]

    class Parameters(SolomonTaskMixinParameters):
        with sdk2.parameters.Group("Input data") as input_data:
            uploaded_logs_to_yt_prefix = sdk2.parameters.String('YT path with logs', required=True)
            yt_proxy = sdk2.parameters.String('YT proxy', default='hahn')
            yql_token_vault_name = sdk2.parameters.String('YQL token valut name', default='yql_token')
            push_data_to_solomon = sdk2.parameters.Bool('Push data to solomon', default=False, do_not_copy=True)
            sampling_strategy = SamplingStrategyParameter('Data sampling strategy')

        with sdk2.parameters.Output:
            products = sdk2.parameters.JSON('Counts of products')

    def create_sensors(self, products):
        sensors = []
        for product, count in products.items():
            sensors.append({
                'labels': {
                    'sensor': product + '_count',
                    'status': product + '_' + self.Parameters.sampling_strategy.upper(),
                },
                'value': count
            })
        return sensors

    def on_save(self):
        if self.author in self.server.group['YABS_SERVER_SANDBOX_TESTS'].read()["members"]:
            self.Parameters.owner = 'YABS_SERVER_SANDBOX_TESTS'

    def on_execute(self):
        formatted_yql_query = YQL_QUERY.format(proxy=self.Parameters.yt_proxy, prefix=self.Parameters.uploaded_logs_to_yt_prefix)
        yql_token = try_get_from_vault(self, self.Parameters.yql_token_vault_name)
        request = run_query(
            query_id='Collect products',
            query=formatted_yql_query,
            yql_token=yql_token,
            db=self.Parameters.yt_proxy,
            wait=False,
            syntax_version=1)
        self.set_info('<a href="{}" target="_blank">YQL operation share link</a>'.format(request.share_url),
                      do_escape=False
                      )
        for table in request.get_results(wait=True):
            break
        if not table:
            raise TaskError("No results fetched (either query error or it doesn't return results)")
        table.fetch_full_data()
        products = {}
        for product, count in table.rows:
            products[product] = count

        self.Parameters.products = json.dumps(products)
        self.set_info(self.Parameters.products)

        if self.Parameters.push_data_to_solomon:
            sensors = self.create_sensors(products)
            self.solomon_push_client.add(sensors, cluster='testenv_yabs-20', service='bs_products')
