from sandbox import sdk2

from sandbox.common.types.misc import NotExists
from sandbox.projects.yabs.qa.resource_types import YabsDiffReport
from sandbox.projects.yabs.qa.tasks.base_compare_task.task import BaseCompareTask
from sandbox.projects.yabs.qa.tasks.base_compare_task.parameters import BaseCompareTaskParameters
from sandbox.projects.yabs.qa.tasks.YabsServerB2BFuncShootCmp.utils.compare_dict import compare_dicts

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace_entry_point


class YABS_SERVER_CS_VERSIONS_DIFF(YabsDiffReport):
    pass


DIFF_FILE_NAME = 'cs_versions.diff'


class YabsServerCompareCSVersions(BaseCompareTask):

    name = 'YABS_SERVER_COMPARE_CS_VERSIONS'

    ignore_compare_input_parameters = [
        'bs_release_yt_resource',
    ]

    class Parameters(BaseCompareTaskParameters):
        with sdk2.parameters.Output:
            with sdk2.parameters.Group('Changes') as changes:
                changed_bases = sdk2.parameters.JSON('List of changed bases')
                changed_importers = sdk2.parameters.JSON('List of changed importers')

    @property
    def diff_resource_type(self):
        return 'YABS_SERVER_CS_VERSIONS_DIFF'

    @property
    def diff_resource_search_attributes(self):
        return {}

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        terminate_task = self.check_tasks_parameters()
        if terminate_task:
            return

        has_diff = False
        diff = []
        changed_bases = set()
        changed_importers = set()

        for parameter_name in (
            'base_internal_versions',
            'importer_code_versions',
            'importer_mkdb_versions',
            'importer_settings_versions',
            'importer_code_versions_with_dependencies',
            'importer_mkdb_versions_with_dependencies',
            'importer_settings_versions_with_dependencies',
        ):
            pre = getattr(self.pre_task.Parameters, parameter_name)
            test = getattr(self.test_task.Parameters, parameter_name)
            if pre and test:
                _has_diff, _versions_diff = compare_dicts(pre, test, fromfile="pre {}".format(parameter_name), tofile="test {}".format(parameter_name))
                has_diff = has_diff or _has_diff
                if _has_diff:
                    diff.append(_versions_diff)

                changed = set()
                for k in set(pre.keys()) & set(test.keys()):
                    if pre[k] != test[k]:
                        changed.add(k)
                if parameter_name == 'base_internal_versions':
                    changed_bases |= changed
                else:
                    changed_importers |= changed

        self.set_info('Changed bases: {}\nChanged importers: {}'.format(', '.join(changed_bases), ', '.join(changed_importers)))

        with open(DIFF_FILE_NAME, 'w') as f:
            f.write('\n'.join(diff))

        YABS_SERVER_CS_VERSIONS_DIFF(
            self,
            'CS settings diff resource',
            DIFF_FILE_NAME,
            has_diff=has_diff,
            test_name=self.Parameters.test_name,
            testenv_db=self.Context.testenv_database,
            arcanum_review_id=self.Context.arcanum_review_id if self.Context.arcanum_review_id is not NotExists else self.Parameters.arcanum_review_id,
        )

        self.Parameters.has_diff = has_diff
        self.Context.has_diff = has_diff
        self.Parameters.changed_bases = sorted(list(changed_bases))
        self.Parameters.changed_importers = sorted(list(changed_importers))
