import logging
import textwrap

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_GCDA_COVERAGE, BS_RELEASE_TAR, YABS_SERVER_GCOV_COVERAGE
from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.common.build.YaPackage2 import YaPackage2
from sandbox.projects.yabs.qa.tasks.YabsServerCollectGcovCoverage import YabsServerCollectGcovCoverage


class YabsServerCoverageFromShoot(BaseBinTask):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 512

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseBinTask.Parameters):
        resource_attrs = BaseBinTask.Parameters.resource_attrs(default={"task_type": "YABS_SERVER_COLLECT_GCOV_COVERAGE"})  # noqa
        release_version = BaseBinTask.Parameters.release_version(default=ReleaseStatus.STABLE)  # noqa

        # I'd use sdk2.parameters.Task but cannot import this task type (no py3 support)
        initial_shoot_task_id = sdk2.parameters.Integer("Task id to get coverage from", required=True)

    def on_execute(self):
        with self.memoize_stage.coverage_build:
            old_shoot_params = self.server.task[self.Parameters.initial_shoot_task_id].read()['input_parameters']
            logging.info(f"old_shoot_params: {old_shoot_params}")
            old_yabs_server_resource: BS_RELEASE_TAR = sdk2.Resource[old_shoot_params["meta_server_resource"]]
            old_yabs_server_resource_stat: BS_RELEASE_TAR = sdk2.Resource[old_shoot_params["stat_server_resource"]]
            logging.info(f"old_yabs_server_resource: {old_yabs_server_resource.id}")
            if old_yabs_server_resource != old_yabs_server_resource_stat:
                logging.warning("Stat and meta resources are different in shoot task, using meta")
            old_ya_package: YaPackage2 = YaPackage2.find(
                parent=sdk2.Task[old_yabs_server_resource.task_id], tags=["BS_RELEASE_TAR"]
            ).first()
            new_ya_package = YaPackage2(self, **{k: v for k, v in old_ya_package.Parameters})
            logging.info(f"Copied old YA_PACKAGE_2 {old_ya_package.id} to {new_ya_package.id}")
            new_ya_package.Parameters.packages = "yabs/server/packages/yabs-server-bundle-coverage.json"
            new_ya_package.save()
            new_ya_package.enqueue()
            self.Context.coverage_build_task_id = new_ya_package.id
            self.Context.sources_arcadia_url = old_ya_package.Parameters.checkout_arcadia_from_url
            raise sdk2.WaitTask([new_ya_package.id], (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))

        with self.memoize_stage.shoot:
            coverage_build: BS_RELEASE_TAR = BS_RELEASE_TAR.find(task_id=self.Context.coverage_build_task_id).first()
            self.Context.coverage_build_tar_resource_id = coverage_build.id
            old_shoot_task = self.server.task[self.Parameters.initial_shoot_task_id].read()
            logging.info(f"Old shoot task: {old_shoot_task}")
            json = {
                "source": old_shoot_task["id"],
                "owner": self.owner,
                "children": True,
                "kill_timeout": 7 * 3600,
            }
            new_shoot_task = self.server.task(json)
            update = {
                "custom_fields": [{'name': k, 'value': v} for k, v in dict(
                    meta_server_resource=coverage_build.id,
                    stat_server_resource=coverage_build.id,
                    do_collect_coverage=True
                ).items()]
            }
            self.server.task[new_shoot_task['id']].update(update)
            self.server.batch.tasks.start.update([new_shoot_task['id']])
            self.set_info(f"Child: {new_shoot_task['id']}")
            self.Context.shoot_task_id = new_shoot_task['id']
            raise sdk2.WaitTask([self.Context.shoot_task_id], (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))

        with self.memoize_stage.collect_coverage:
            gcdas_resource = YABS_SERVER_GCDA_COVERAGE.find(task_id=self.Context.shoot_task_id).first()
            collect_coverage_task = YabsServerCollectGcovCoverage(
                self,
                coverage_build=sdk2.Resource[self.Context.coverage_build_tar_resource_id],
                sources_url=self.Context.sources_arcadia_url,
                gcdas_file=gcdas_resource,
                __requirements__={"tasks_resource": self.Requirements.tasks_resource}
            ).save().enqueue()
            self.set_info(f"Collect coveragetask: {collect_coverage_task.id}")
            self.Context.collect_coverage_task_id = collect_coverage_task.id
            raise sdk2.WaitTask([self.Context.collect_coverage_task_id], (ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))  # noqa

        with self.memoize_stage.save_gcov:
            gcov_resource = YABS_SERVER_GCOV_COVERAGE.find(task_id=self.Context.collect_coverage_task_id).first()
            resource_a = textwrap.dedent(
                f"""
                <a
                href='https://proxy.sandbox.yandex-team.ru/{gcov_resource.id}/coverage.html'>
                Coverage collected here!
                </a>"""
            )
            self.set_info(resource_a, do_escape=False)
