import logging

from sandbox import sdk2
from sandbox.common.types.notification import Transport

from sandbox.projects.yabs.qa.tasks.YabsCreateTestenvResources import YtArchiveParameters

from sandbox.projects.yabs.qa.spec.parameters import BaseSpecParameters
from sandbox.projects.yabs.qa.spec.stages import (
    get_engine_resources,
    get_ammo_resources,
    get_misc_resources,
    get_shard_nums,
    get_shard_keys,
    launch_setup_ya_make,
    get_binary_base_tags,
    launch_ft_shoot_tasks,
    launch_ft_stability_shoot_tasks,
    launch_stat_load_baseline,
    launch_meta_load_baseline,
)
from sandbox.projects.yabs.qa.spec.misc import create_spec_resource, fail_if_subtasks_not_ok
from sandbox.projects.yabs.qa.tasks.YabsServerCreateABExperimentSpec.spec import YabsServerABExperimentSpec
from sandbox.projects.yabs.qa.tasks.YabsServerCreateABExperimentSpec.stages import (
    build_validate_ab_experiment_bin,
    check_subtasks_status,
    get_binary_bases,
    get_data_resources,
    get_linear_models_data,
    get_spec_data,
    launch_archive_cs_input_task,
    launch_base_producing_tasks,
)


logger = logging.getLogger(__name__)


_STAGES = [
    get_engine_resources,
    get_ammo_resources,
    get_misc_resources,
    get_shard_nums,
    get_shard_keys,
    launch_setup_ya_make,
    get_binary_base_tags,
    get_linear_models_data,
    launch_ft_shoot_tasks,
    launch_ft_stability_shoot_tasks,
    launch_stat_load_baseline,
    launch_meta_load_baseline,
    build_validate_ab_experiment_bin,
    launch_archive_cs_input_task,
    get_data_resources,
    launch_base_producing_tasks,
    get_binary_bases,
    get_spec_data,
]


class YabsServerCreateABExperimentSpec(sdk2.Task):
    """Create spec for quick AB experiment testing
    """

    name = 'YABS_SERVER_CREATE_AB_EXPERIMENT_SPEC'

    class Requirements(sdk2.Requirements):
        ram = 4 * 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseSpecParameters):

        cs_input_spec_parameters_block = YtArchiveParameters()

        with sdk2.parameters.Group("Developer options") as developer_group:
            debug_mode = sdk2.parameters.Bool('Debug mode', default=False)
            with debug_mode.value[True]:
                reusable_stages = sdk2.parameters.CheckGroup(
                    'Reusable stages',
                    choices=[
                        (_stage.__name__, _stage.__name__)
                        for _stage in _STAGES
                    ],
                    default=None,
                )
        drop_other_experiments = sdk2.parameters.Bool('Drop experiment other than DEFAULT/PRE_DEFAULT', default=False)
        linear_models_data_resource_type = sdk2.parameters.String('Resource type of linear models service data', default='YABS_LINEAR_MODELS_DATA')

    def on_create(self):
        super(YabsServerCreateABExperimentSpec, self).on_create()
        if self.Context.copy_of:
            self.Parameters.notifications = [
                notification for notification in self.Parameters.notifications
                if notification.transport != Transport.JUGGLER
            ]

    def on_execute(self):
        if self.Parameters.debug_mode:
            self.Context.__reusable_stages = self.Parameters.reusable_stages

        get_engine_resources(self)
        get_ammo_resources(self)
        get_misc_resources(self, hamster_ext_service_tags=self.Parameters.hamster_ext_service_tags)
        get_shard_nums(self)
        get_shard_keys(self)

        build_validate_ab_experiment_bin(self)
        launch_setup_ya_make(self)
        launch_archive_cs_input_task(self)
        get_binary_base_tags(self)  # Waits for setup_ya_make_task
        get_data_resources(self)  # Waits for archive_cs_input_task

        launch_base_producing_tasks(self)
        get_binary_bases(self)  # Waits for base_download_task
        get_linear_models_data(self, resource_type=self.Parameters.linear_models_data_resource_type)

        launch_ft_shoot_tasks(
            self,
            use_default_debug_cookie=False,
            response_dumps_ttl=3,
            hamster_ext_service_tags=self.Parameters.hamster_ext_service_tags,
        )
        launch_ft_stability_shoot_tasks(
            self,
            use_default_debug_cookie=False,
            stability_runs=3,
            hamster_ext_service_tags=self.Parameters.hamster_ext_service_tags,
        )

        launch_stat_load_baseline(self)  # Optional
        launch_meta_load_baseline(self, use_default_debug_cookie=False)  # Optional

        broken_subtasks, failed_subtasks = check_subtasks_status(self)
        fail_if_subtasks_not_ok(self, broken_subtasks, failed_subtasks)

        spec = get_spec_data(self, drop_other_experiments=self.Parameters.drop_other_experiments)
        create_spec_resource(
            self,
            YabsServerABExperimentSpec,
            spec,
            'AB-experiment spec',
            'spec.json',
            released_spec=self.Parameters.release_spec,
            stat_load=self.Parameters.stat_load,
            meta_load=self.Parameters.meta_load,
            ft=True,
            use_separated_meta_and_stat=self.Parameters.use_separated_meta_and_stat,
        )
