import logging
from itertools import chain

from sandbox import sdk2
from sandbox.projects.common.yabs.server.util.general import find_last_ready_resource, check_tasks
from sandbox.projects.yabs.qa.resource_types import (
    YABS_CS_INPUT_SPEC,
    YABS_CS_SETTINGS_ARCHIVE,
    YABS_MYSQL_ARCHIVE_CONTENTS,
)
from sandbox.projects.yabs.qa.pipeline.stage import stage, memoize_stage
from sandbox.projects.yabs.qa.pipeline_test_framework.helpers import get_stat_base_tags_to_create, get_meta_base_tags_to_create
from sandbox.projects.yabs.qa.tasks.YabsServerGetProdBases import YabsServerGetProdBases
from sandbox.projects.yabs.qa.tasks.YabsServerGenerateConfigWithNewAbExperiment import YabsServerGenerateConfigWithNewAbExperiment
from sandbox.projects.yabs.qa.spec.misc import filter_not_ok_tasks
from sandbox.projects.yabs.qa.tasks.YabsServerValidateABExperiment import BS_AB_HANDLERS
from sandbox.projects.yabs.qa.tasks.YabsServerCreateABExperimentSpec.spec import ABExperimentSpec
from sandbox.projects.yabs.qa.tasks.YabsServerCreateContentSystemInputArchive import YabsServerCreateContentSystemInputArchive


logger = logging.getLogger(__name__)


@stage(provides='archive_cs_input_task_id')
def launch_archive_cs_input_task(self):
    archive_cs_input_task = YabsServerCreateContentSystemInputArchive(
        self,
        tags=self.Parameters.tags,
        description='Create CS input archive | {}'.format(self.Parameters.description),
        archives_root=self.Parameters.yt_archive_archives_root,
        yt_proxy=self.Parameters.yt_archive_yt_proxy,
        __requirements__={
            "tasks_resource": self.Requirements.tasks_resource,
        },
    ).enqueue()
    return archive_cs_input_task.id


@stage(provides=(
    "mysql_archive_resource_id",
    "cs_input_spec_resource_id",
    "cs_settings_archive_resource_id",
), result_is_dict=True)
def get_data_resources(self, archive_cs_input_task_id):
    mysql_archive_resource = self.Parameters.mysql_archive_resource or find_last_ready_resource(YABS_MYSQL_ARCHIVE_CONTENTS, {'testenv_switch_trigger': None})
    cs_settings_archive_resource = self.Parameters.cs_settings_archive_resource or find_last_ready_resource(YABS_CS_SETTINGS_ARCHIVE, {'testenv_switch_trigger': None})

    check_tasks(self, [archive_cs_input_task_id])
    cs_input_spec_resource = YABS_CS_INPUT_SPEC.find(task_id=archive_cs_input_task_id, limit=1, state='READY').first()

    return {
        "mysql_archive_resource_id": mysql_archive_resource.id,
        "cs_input_spec_resource_id": cs_input_spec_resource.id,
        "cs_settings_archive_resource_id": cs_settings_archive_resource.id,
    }


def _launch_base_producing_tasks(
        self,
        base_tags_map,
        base_ver,
        cs_import_ver,
        gen_bin_bases_flags_resource_id,
        mysql_archive_resource_id, cs_input_spec_resource_id, cs_settings_archive_resource_id,
        bs_release_yt_resource_id, bs_release_tar_resource_id,
        base_tags_to_create,
        bases_to_create=('constant', ),
        desc='',
        bsfrontend_dir='//home/yabs-transport/transport/bsfrontend_dir',
):
    base_tags_to_download = set(chain(*base_tags_map.values())) - (set(bases_to_create) | {'ab_experiment'})

    base_download_task = YabsServerGetProdBases(
        self,
        description='{} | download bases: {} | {}'.format(desc, ', '.join(sorted(list(base_tags_to_download))), self.Parameters.description),
        tags=self.Parameters.tags,
        base_ver=base_ver,
        bases=sorted(list(base_tags_to_download)),
        yt_path=bsfrontend_dir,
        __requirements__={'tasks_resource': self.Requirements.tasks_resource},
    ).enqueue()

    add_base_options = getattr(sdk2.Resource[gen_bin_bases_flags_resource_id], 'add_options', None)
    options_list = add_base_options.strip() if add_base_options is not None else ''

    make_bin_bases_task = sdk2.Task['YABS_SERVER_MAKE_BIN_BASES'](
        self,
        description='{} | make bases: {} | {}'.format(desc, ', '.join(list(sorted(bases_to_create))), self.Parameters.description),
        tags=self.Parameters.tags,
        input_spec=cs_input_spec_resource_id,
        settings_archive=cs_settings_archive_resource_id,
        mysql_archive_contents=mysql_archive_resource_id,
        bs_release_yt_resource=bs_release_yt_resource_id,
        server_resource=bs_release_tar_resource_id,
        options_list=options_list,
        bin_db_list=' '.join(bases_to_create),
        cs_import_ver=cs_import_ver,
        use_cs_cycle=True,
        do_not_restart=True,
        glue_reduce=True,
        reuse_existing_bases=True,
        save_all_inputs=False,
        use_save_input_from_cs=False,
        filter_input_archive_tables_by_orderid=False,
        yt_pool='yabs-cs-sandbox-ab-experiment',
        **base_tags_to_create
    ).enqueue()

    generate_ab_config_task = YabsServerGenerateConfigWithNewAbExperiment(
        YabsServerGenerateConfigWithNewAbExperiment.current,
        description='{} | make bases: ab_experiment | {}'.format(desc, self.Parameters.description),
        tags=self.Parameters.tags,
        ab_handlers=BS_AB_HANDLERS,
        mbb_params={
            'input_spec': cs_input_spec_resource_id,
            'settings_archive': cs_settings_archive_resource_id,
            'mysql_archive_contents': mysql_archive_resource_id,
            'bs_release_yt_resource': bs_release_yt_resource_id,
            'server_resource': bs_release_tar_resource_id,
            'use_cs_cycle': True,
            'use_save_input_from_cs': False,
            'filter_input_archive_tables_by_orderid': False,
            'save_all_inputs': False,
            'run_fetch_tasks': False,
        },
        release_version='stable',
        drop_other_experiments=True,
        __requirements__={'tasks_resource': self.Requirements.tasks_resource},
    ).enqueue()

    return base_download_task.id, make_bin_bases_task.id, generate_ab_config_task.id


@stage(provides=(
    'stat_download_bases_task_id',
    'stat_make_bin_bases_task_id',
    'stat_generate_ab_config_task_id',
    'meta_download_bases_task_id',
    'meta_make_bin_bases_task_id',
    'meta_generate_ab_config_task_id',
))
def launch_base_producing_tasks(
        self,
        stat_base_tags_map,
        stat_base_ver,
        stat_cs_import_ver,
        meta_base_tags_map,
        meta_base_ver,
        meta_cs_import_ver,
        ft_shard_keys,
        stat_load_shard_keys,
        meta_load_shard_keys,
        gen_bin_bases_flags_resource_id,
        mysql_archive_resource_id, cs_input_spec_resource_id, cs_settings_archive_resource_id,
        stat_bs_release_yt_resource_id, stat_bs_release_tar_resource_id,
        meta_bs_release_yt_resource_id, meta_bs_release_tar_resource_id,
        bases_to_create=('constant', ),
):
    shard_keys = list(set(ft_shard_keys + stat_load_shard_keys + meta_load_shard_keys))
    stat_base_tags_to_create = get_stat_base_tags_to_create(stat_base_tags_map, shard_keys, bases_to_create)
    meta_base_tags_to_create = get_meta_base_tags_to_create(meta_base_tags_map, bases_to_create)

    base_tags_to_create = dict(stat_base_tags_to_create)
    base_tags_to_create.update(meta_base_tags_to_create)

    return _launch_base_producing_tasks(
        self,
        stat_base_tags_map,
        stat_base_ver,
        stat_cs_import_ver,
        gen_bin_bases_flags_resource_id,
        mysql_archive_resource_id, cs_input_spec_resource_id, cs_settings_archive_resource_id,
        stat_bs_release_yt_resource_id, stat_bs_release_tar_resource_id,
        stat_base_tags_to_create,
        bases_to_create=bases_to_create,
        desc='stat',
        bsfrontend_dir='//home/yabs-transport/transport/bsfrontend_dir',
    ) + _launch_base_producing_tasks(
        self,
        meta_base_tags_map,
        meta_base_ver,
        meta_cs_import_ver,
        gen_bin_bases_flags_resource_id,
        mysql_archive_resource_id, cs_input_spec_resource_id, cs_settings_archive_resource_id,
        meta_bs_release_yt_resource_id, meta_bs_release_tar_resource_id,
        meta_base_tags_to_create,
        bases_to_create=bases_to_create,
        desc='meta',
        bsfrontend_dir='//home/yabs-transport/transport/bsfrontend_meta_dir',
    )


def _get_binary_bases(self, download_bases_task_id, make_bin_bases_task_id, generate_ab_config_task_id):
    binary_base_resource_id_by_tag = sdk2.Task[download_bases_task_id].Parameters.base_resources

    binary_base_resource_ids = sdk2.Task[make_bin_bases_task_id].Context.bin_base_res_ids
    binary_base_resources = sdk2.Resource.find(id=binary_base_resource_ids).limit(len(binary_base_resource_ids))
    binary_base_resource_id_by_tag.update({
        resource.tag: resource.id
        for resource in binary_base_resources
    })

    new_binary_base_resource_id = sdk2.Task[generate_ab_config_task_id].Context.ab_experiment_db
    new_binary_base_resource = sdk2.Resource[new_binary_base_resource_id]
    binary_base_resource_id_by_tag.update({
        new_binary_base_resource.tag: new_binary_base_resource.id
    })

    return binary_base_resource_id_by_tag


@stage(provides=(
    'stat_binary_base_resource_id_by_tag',
    'meta_binary_base_resource_id_by_tag',
))
def get_binary_bases(
        self,
        stat_download_bases_task_id,
        stat_make_bin_bases_task_id,
        stat_generate_ab_config_task_id,
        meta_download_bases_task_id,
        meta_make_bin_bases_task_id,
        meta_generate_ab_config_task_id,
        use_new_generate_config=True,
):
    check_tasks(
        self, [
            stat_download_bases_task_id, stat_make_bin_bases_task_id, stat_generate_ab_config_task_id,
            meta_download_bases_task_id, meta_make_bin_bases_task_id, meta_generate_ab_config_task_id,
        ]
    )
    return (
        _get_binary_bases(self, stat_download_bases_task_id, stat_make_bin_bases_task_id, stat_generate_ab_config_task_id),
        _get_binary_bases(self, meta_download_bases_task_id, meta_make_bin_bases_task_id, meta_generate_ab_config_task_id),
    )


@memoize_stage()
def check_subtasks_status(
        self,
        ft_shoot_tasks,
        ft_stability_shoot_tasks,
        stat_load_baseline_tasks,
        meta_load_baseline_tasks,
):
    tasks_to_check = list(chain.from_iterable(
        meta_mode_shoot_tasks.values()
        for meta_mode_shoot_tasks in ft_shoot_tasks.values()
    ))

    if self.Parameters.sanity_check:
        tasks_to_check.extend(
            list(chain.from_iterable(
                meta_mode_shoot_tasks.values()
                for meta_mode_shoot_tasks in ft_stability_shoot_tasks.values()
            ))
        )

    if self.Parameters.stat_load:
        tasks_to_check.extend(
            list(chain.from_iterable(
                meta_mode_shoot_tasks.values()
                for meta_mode_shoot_tasks in stat_load_baseline_tasks.values()
            ))
        )

    if self.Parameters.meta_load:
        tasks_to_check.extend(meta_load_baseline_tasks.values())

    return filter_not_ok_tasks(self, tasks_to_check)


@stage(provides='spec')
def get_spec_data(
        self,
        stat_bs_release_tar_resource_id,
        stat_bs_release_yt_resource_id,
        meta_bs_release_tar_resource_id,
        meta_bs_release_yt_resource_id,
        mysql_archive_resource_id,
        cs_input_spec_resource_id,
        cs_settings_archive_resource_id,
        ft_request_log_resource_id_map,
        shard_map_resource_id,
        stat_setup_ya_make_task_id,
        meta_setup_ya_make_task_id,
        gen_bin_bases_flags_resource_id,
        ft_shoot_tasks,
        stat_base_tags_map,
        stat_binary_base_resource_id_by_tag,
        meta_base_tags_map,
        meta_binary_base_resource_id_by_tag,
        ext_service_endpoint_resource_ids,
        stat_load_request_log_resource_id_map,
        stat_load_baseline_tasks,
        meta_load_request_log_resource_id_map,
        meta_load_baseline_tasks,
        validate_ab_experiment_bin_id,
        drop_other_experiments=False,
        use_separated_meta_and_stat=True,
):
    spec_obj = ABExperimentSpec(self.Context, drop_other_experiments=drop_other_experiments,
                                use_separated_meta_and_stat=use_separated_meta_and_stat,
                                validate_ab_experiment_bin_id=validate_ab_experiment_bin_id)
    spec = spec_obj.as_dict()
    return spec


@stage(provides='validate_ab_experiment_bin_id')
def build_validate_ab_experiment_bin(
    self,
    meta_bs_release_tar_resource_id,
):
    build_yabs_server_task = sdk2.Resource[meta_bs_release_tar_resource_id].task
    ya_make_validate_ab_experiment = sdk2.Task['YA_MAKE_2'](
        self,
        use_aapi_fuse=True,
        aapi_fallback=True,
        checkout_arcadia_from_url=build_yabs_server_task.Parameters.checkout_arcadia_from_url,
        arts='yabs/server/tools/validate_ab_experiment/validate_ab_experiment',
        result_rd='binary for task YABS_SERVER_VALIDATE_AB_EXPERIMENT',
        result_rt='YABS_SERVER_VALIDATE_AB_EXPERIMENT_BIN',
        result_single_file=True,
        targets='yabs/server/tools/validate_ab_experiment',
        sandbox_tags='GENERIC & LINUX_FOCAL',
        force_sandbox_tags=True,
    ).enqueue()
    return ya_make_validate_ab_experiment.id


@stage(provides='linear_models_data_resource_id')
def get_linear_models_data(self, resource_type='YABS_LINEAR_MODELS_DATA'):
    return sdk2.Resource.find(
        type=resource_type,
        attrs={
            "released": "stable",
            "shard": "YabsRtModels",
        }
    ).first().id
