import itertools
import json
import logging

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.common.yabs.server.db import yt_bases, utils as db_utils
from sandbox.projects.common.yabs.server.db.task.cs import FilterInputArchiveTablesByOrderID
from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.yabs.qa.bases.sample_tables.parameters import SamplingYtPool
from sandbox.projects.yabs.qa.resource_types import BS_RELEASE_YT, YABS_CS_INPUT_SPEC, YABS_CS_SETTINGS_ARCHIVE
from sandbox.projects.yabs.qa.tasks.YabsServerSaveInput import YabsServerSaveInput, TABLES_TO_BE_FILTERED_BY_ORDER_ID
from sandbox.projects.yabs.qa.utils.base import generate_basenos
from sandbox.projects.yabs.qa.utils.resource import sync_resource


logger = logging.getLogger(__name__)


class YabsServerCreateContentSystemInputArchive(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 8 * 1024
        client_tags = Tag.LINUX_PRECISE
        environments = [
            sdk2.environments.PipEnvironment('yandex-yt', use_wheel=True),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        tokens = sdk2.parameters.YavSecret("YAV secret identifier", default="sec-01d6apzcex5fpzs5fcw1pxsfd5")

        yt_proxy = sdk2.parameters.String("YT proxy", default="hahn")
        archives_root = sdk2.parameters.String(
            "Root node for import archives",
            required=True,
            default="//home/yabs-cs-sandbox/input-archive")

        run_save_input = sdk2.parameters.Bool("Run save_input", default=False)
        with run_save_input.value[True]:
            bs_release_yt_resource = sdk2.parameters.Resource(
                "BS release yt resource",
                resource_type=BS_RELEASE_YT)

            cs_settings_archive_resource = sdk2.parameters.Resource(
                "CS settings archive resource",
                multiple=False,
                resource_type=YABS_CS_SETTINGS_ARCHIVE)

        testenv_switch_trigger = sdk2.parameters.String(
            "Testenv switch trigger",
            description="Set output resource's attribute \"testenv_switch_trigger\"",
            do_not_copy=True)

    def on_execute(self):
        with self.memoize_stage.create_spec(commit_on_entrance=False):
            from yt.wrapper import YtClient, ypath_join

            yt_token = self.Parameters.tokens.data()["yt_token"]
            yt_client = YtClient(proxy=self.Parameters.yt_proxy, token=yt_token)

            archive_root = ypath_join(self.Parameters.archives_root, str(self.id))

            logger.debug("Create map_node \"%s\"", archive_root)
            yt_client.create("map_node", archive_root, ignore_existing=True, recursive=True)

            logger.debug("Set TTL %s seconds for map_node \"%s\"", yt_bases.DEFAULT_CS_INPUT_ARCHIVE_TTL, archive_root)
            yt_bases.renew_expiration_time(yt_client, archive_root, yt_bases.DEFAULT_CS_INPUT_ARCHIVE_TTL)

            self.Context.spec_resource_id = self.create_spec_resource(archive_root).id

        if self.Parameters.run_save_input:
            with self.memoize_stage.run_save_input(commit_on_entrance=False):
                self.Context.save_input_task_id = self.run_save_input(
                    self.Parameters.bs_release_yt_resource,
                    self.Parameters.cs_settings_archive_resource,
                    input_spec_resource_id=self.Context.spec_resource_id,
                ).id
            check_tasks(self, self.Context.save_input_task_id)

    def create_spec_resource(self, archive_root):
        spec = {
            yt_bases.AUXILLARY_DATA_IN_SPEC_KEY: {
                yt_bases.ARCHIVE_ROOT_KEY: archive_root,
                yt_bases.ARCHIVE_ROOT_TTL_KEY: yt_bases.DEFAULT_CS_INPUT_ARCHIVE_TTL,
            },
            "proxy": self.Parameters.yt_proxy
        }

        spec_path = 'input_spec.json'
        with open(spec_path, 'w') as out:
            json.dump(spec, out)

        resource = YABS_CS_INPUT_SPEC(
            task=self,
            description="yabs-cs cs_import input spec for Testenv bases",
            path=spec_path,
            yt_proxy=self.Parameters.yt_proxy,
        )
        if self.Parameters.testenv_switch_trigger:
            resource.testenv_switch_trigger = self.Parameters.testenv_switch_trigger

        sdk2.ResourceData(resource).ready()
        return resource

    def run_save_input(
        self,
        bs_release_yt_resource,
        cs_settings_archive_resource,
        input_spec_resource_id,
    ):
        cs_settings_str = get_cs_settings(cs_settings_archive_resource)

        bs_release_yt_dir = db_utils.get_yabscs(self, bs_release_yt_resource.id)
        cs_import_info = yt_bases.get_cs_import_info(bs_release_yt_dir, cs_settings_str)
        tables_to_save = list(itertools.chain.from_iterable(
            importer_data["tables"]
            for importer_data in cs_import_info.values()
        ))
        logger.info("Save tables %s", tables_to_save)

        baseno_list = sorted(generate_basenos(stats=[1, 2, 5], stats_in_clusters=[12]))
        logger.debug("Baseno list: %s", baseno_list)

        save_input_task = YabsServerSaveInput(
            self,
            kill_timeout=18000,
            owner=self.owner,
            description="Run save_input for task {}".format(self.id),
            tables_to_save=tables_to_save,
            filter_tables_by_orderid=FilterInputArchiveTablesByOrderID.default_value,
            input_spec=input_spec_resource_id,
            dynamic_bundle_name="yabs-cs-sandbox-dynamic",
            no_quota_attrs=True,
            recreate_links=True,
            tables_filtred_by_orderid=TABLES_TO_BE_FILTERED_BY_ORDER_ID,
            baseno_list=baseno_list,
            bs_release_yt_resource=bs_release_yt_resource,
            settings_archive=cs_settings_archive_resource,
            yt_pool=SamplingYtPool.default_value,
            __requirements__={
                "tasks_resource": self.Requirements.tasks_resource,
            },
        )
        save_input_task.enqueue()
        return save_input_task


def get_cs_settings(cs_settings_archive_resource):
    cs_settings_path = sync_resource(cs_settings_archive_resource.id)
    with open(cs_settings_path) as resource_file:
        return resource_file.read()
