import datetime
import json
import os
import logging
import multiprocessing
import re
import subprocess
import tarfile
import time
import uuid

import six
import yaml

import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.misc import NotExists
from sandbox.common.types.task import ReleaseStatus
from sandbox.common.share import skynet_get
from sandbox.projects.common.build.YaMake2 import YaMake2
from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.yabs.qa.hamster.deploy import (
    get_pod_errors,
    get_pod_statuses,
    get_stage_errors,
    get_stage_status,
    is_deploy_unit_ready,
    put_stage,
    remove_stage,
)
from sandbox.projects.yabs.qa.hamster.spec import create_hamster_spec_resource, ExternalServiceType
from sandbox.projects.yabs.qa.hamster.stability import run_stability_shoot
from sandbox.projects.yabs.qa.pipeline.stage import stage
from sandbox.projects.yabs.qa.resource_types import (
    BaseBackupSdk2Resource,
    GOALNET_RELEASE_TAR,
    YABS_SERVER_TESTENV_SHARD_MAP,
    YabsServerExtServiceEndpoint,
)
from sandbox.projects.yabs.qa.template_utils import load_template
from sandbox.projects.yabs.qa.utils.base import unpack_base
from sandbox.projects.yabs.qa.utils.bstr import get_bstr_info, parse_base_filename
from sandbox.projects.yabs.qa.utils.general import (
    get_resource_html_hyperlink,
    get_task_html_hyperlink,
    html_hyperlink,
)
from sandbox.projects.yabs.qa.utils.resource import (
    get_last_released_resource,
    resource_from_json,
)
from sandbox.sandboxsdk.paths import get_logs_folder


logger = logging.getLogger(__name__)


BASES_DIR = "goalnet_bases"
GOALNET_SERVICE_TAG = "goalnet"
GOALNET_SHARDS_COUNT = 360
ST_SHARDS_COUNT = 12
YP_CLUSTER = "sas"


class YabsServerGoalnetBasesLayer(BaseBackupSdk2Resource):
    pass


class YabsServerGoalnetConfigLayer(BaseBackupSdk2Resource):
    pass


class YpObjectStatus(BaseBackupSdk2Resource):
    pass


def filter_bases_by_shard(bases, shards):
    shards = set(shards)
    result = []
    goalst_base_name_pattern = re.compile(r"\d+\.goalst(?P<shard>\d+)\.yabs")
    for base_name in bases:
        match = re.match(goalst_base_name_pattern, base_name)
        if not match:
            result.append(base_name)
            continue

        if int(match.group("shard")) in shards:
            result.append(base_name)

    return result


@stage(provides="build_gen_db_list_task_id")
def build_gen_db_list(self, goalnet_binary_resource_id):
    target = "yabs/server/tools/gen_db_list"
    build_gen_db_list_task = YaMake2(
        self,
        description='Build "{}"'.format(target),
        checkout_arcadia_from_url=sdk2.Resource[goalnet_binary_resource_id].svn_path,
        build_system='semi_distbuild',
        targets=target,
        arts=os.path.join(target, "gen_db_list"),
        result_rt="ARCADIA_PROJECT",
        result_rd=target,
        result_single_file=True,
        use_aapi_fuse=True,
        aapi_fallback=True,
        use_arc_instead_of_aapi=True,
        sandbox_tags=ctc.Tag.Group.LINUX & ~ctc.Tag.PORTOD,
    )
    build_gen_db_list_task.enqueue()
    return build_gen_db_list_task.id


@stage(provides="goalnet_bases")
def get_goalnet_bases(self, build_gen_db_list_task_id):
    check_tasks(self, build_gen_db_list_task_id)

    gen_db_list_resource = sdk2.Resource.find(
        type="ARCADIA_PROJECT",
        task_id=build_gen_db_list_task_id
    ).first()
    gen_db_list_path = str(sdk2.ResourceData(gen_db_list_resource).path)
    logger.debug('gen_db_list_path: %s', gen_db_list_path)
    db_list_str = subprocess.check_output(gen_db_list_path)
    db_list = json.loads(db_list_str)
    goalnet_bases = db_list["goalnetsvc"]
    logger.debug('goalnet_bases: %s', goalnet_bases)
    return goalnet_bases


def watch_futures(executor, futures):
    """Reraise exception if any future raises one and cancel pending futures
    :type executor: concurrent.futures.Executor
    :type futures: List[concurrent.futures.Future]
    :raises e: First exception raised by any future
    """
    import concurrent.futures

    done, not_done = concurrent.futures.wait(futures, return_when=concurrent.futures.FIRST_EXCEPTION)

    for future in done:
        e = future.exception()
        if e:
            executor.shutdown(wait=False)
            for future in not_done:
                future.cancel()
            raise e


def download_bases(output_dir, bases, yt_token):
    import concurrent.futures
    from yt.wrapper import YtClient

    def download_base(base_path, rbtorrent, local_path):
        try:
            skynet_get(rbtorrent, local_path, fallback_to_bb=True)
        except Exception:
            logger.error("Cannot download %s due to exception", base_path, exc_info=True)
            raise
        return os.path.join(local_path, base_path)

    basever, _ = parse_base_filename(bases[0])
    logger.debug("basever: %s", basever)

    bases = [parse_base_filename(base_filename)[1] for base_filename in bases]
    logger.debug("bases: %s", bases)

    yt_proxy = "locke"
    bstr_info_dir = "//home/yabs-transport/transport/bsfrontend_dir"
    yt_client = YtClient(proxy=yt_proxy, token=yt_token)
    bstr_info = get_bstr_info(bases, basever, yt_client, bstr_info_dir)
    logger.debug("bstr_info: %s", json.dumps(bstr_info))

    # Use default max_workers value from docs
    # https://docs.python.org/3/library/concurrent.futures.html#concurrent.futures.ThreadPoolExecutor
    max_workers = min(32, multiprocessing.cpu_count() + 4)
    with concurrent.futures.ThreadPoolExecutor(max_workers=max_workers) as executor:
        futures = [
            executor.submit(download_base, base_bstr_info["name"], base_bstr_info["torrent"], output_dir)
            for base_bstr_info in bstr_info.values()
        ]
        watch_futures(executor, futures)


def create_bases_layer_resource(self, yt_token, base_shards=None):
    build_gen_db_list(self)

    goalnet_bases = get_goalnet_bases(self)
    if base_shards:
        goalnet_bases = filter_bases_by_shard(goalnet_bases, base_shards)

    bases_dir = os.path.abspath("goalnet_bases_packed")
    os.makedirs(bases_dir)
    download_bases(bases_dir, goalnet_bases, yt_token)
    logger.debug("Downloaded bases: %s", os.listdir(bases_dir))

    unpacked_bases_dir = os.path.abspath(BASES_DIR)
    os.makedirs(unpacked_bases_dir)
    for base_filename in os.listdir(bases_dir):
        src_base_path = os.path.join(bases_dir, base_filename)
        dst_base_path = os.path.join(unpacked_bases_dir, os.path.splitext(base_filename)[0])
        unpack_base(src_base_path, dst_base_path)
    logger.debug("Unpacked bases: %s", os.listdir(unpacked_bases_dir))

    goalnet_bases_archive_path = os.path.abspath("goalnet_bases.tar.gz")
    logger.debug("Compress bases dir %s to %s", unpacked_bases_dir, goalnet_bases_archive_path)
    with tarfile.open(goalnet_bases_archive_path, "w:gz") as tar:
        tar.add(os.path.relpath(unpacked_bases_dir))

    # create resource with bases archive
    goalnet_bases_resource = YabsServerGoalnetBasesLayer(
        self, "Goalnet bases", goalnet_bases_archive_path,
        basever=parse_base_filename(goalnet_bases[0])[0],
        bases=json.dumps(goalnet_bases),
        bases_dir=BASES_DIR,
    )
    sdk2.ResourceData(goalnet_bases_resource).ready()
    return goalnet_bases_resource


def st_shards_to_goalnet_shards(st_shards):
    if not st_shards:
        return []

    if min(st_shards) < 1 or max(st_shards) > ST_SHARDS_COUNT:
        raise ValueError("Shards must have a value from 1 to %s: %s", ST_SHARDS_COUNT, st_shards)

    goalnet_shards = []
    st_shard_capacity = GOALNET_SHARDS_COUNT / ST_SHARDS_COUNT
    for st_shard in set(st_shards):
        goalnet_shards += [
            ST_SHARDS_COUNT * i + st_shard
            for i in range(st_shard_capacity)
        ]
    return sorted(goalnet_shards)


@stage(provides="base_shards")
def get_base_shards(self):
    if not self.Parameters.shard_map_resource:
        return None

    data = sdk2.ResourceData(self.Parameters.shard_map_resource)
    with open(str(data.path)) as f:
        shard_map = json.load(f)
    logger.debug("shard map: %s", shard_map)
    st_shards = sorted([int(shard) for shard in shard_map.values()])
    logger.debug("st shards: %s", st_shards)
    base_shards = st_shards_to_goalnet_shards(st_shards)
    logger.debug("base shards: %s", base_shards)
    return base_shards


@stage(provides="goalnet_binary_resource_id")
def get_goalnet_binary_resource_id(self):
    goalnet_binary_resource = self.Parameters.goalnet_binary_resource or get_last_released_resource(GOALNET_RELEASE_TAR)
    return goalnet_binary_resource.id


@stage(provides="goalnet_bases_layer_resource_id")
def get_goalnet_bases_layer_resource_id(self, base_shards, yt_token=None):
    resource = self.Parameters.goalnet_bases_layer_resource or create_bases_layer_resource(self, yt_token, base_shards)
    return resource.id


@stage(provides="goalnet_config_resource_id")
def get_goalnet_config_resource_id(self):
    if self.Parameters.goalnet_config_resource:
        return self.Parameters.goalnet_config_resource.id

    config = load_template("server.cfg", templates_dir=os.path.dirname(__file__))
    config_path = "server.cfg"
    with open(config_path, "w") as f:
        f.write(str(config))

    config_archive_path = "config.tar.gz"
    with tarfile.open(config_archive_path, "w:gz") as tar:
        tar.add(config_path)

    resource = YabsServerGoalnetConfigLayer(self, "Goalnet hamster config", config_archive_path)
    sdk2.ResourceData(resource).ready()
    return resource.id


class YabsServerCreateGoalnetHamster(BaseBinTask):
    class Requirements(sdk2.Requirements):
        ram = 32 * 1024
        cores = 32

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseBinTask.Parameters):
        push_tasks_resource = True

        resource_attrs = BaseBinTask.Parameters.resource_attrs(default={"task_type": "YABS_SERVER_CREATE_GOALNET_HAMSTER"})
        release_version = BaseBinTask.Parameters.release_version(default=ReleaseStatus.STABLE)

        tokens = sdk2.parameters.YavSecret(
            "YAV secret identifier",
            default="sec-01f8yr9r918932eypfrd0evrd7"  # robot-yabs-hamster
        )

        goalnet_binary_resource = sdk2.parameters.Resource(
            "Goalnet binary resource",
            resource_type=GOALNET_RELEASE_TAR,
            attrs={
                "released": "stable",
            },
        )
        goalnet_bases_layer_resource = sdk2.parameters.Resource(
            "Goalnet bases layer resource",
            resource_type=YabsServerGoalnetBasesLayer,
        )
        goalnet_config_resource = sdk2.parameters.Resource(
            "Goalnet config layer resource",
            resource_type=YabsServerGoalnetConfigLayer,
        )
        shard_map_resource = sdk2.parameters.Resource(
            "Shard map resource",
            resource_type=YABS_SERVER_TESTENV_SHARD_MAP,
            multiple=False,
        )

        wait_service_activation = sdk2.parameters.Bool("Wait for service activation", default=True)
        with wait_service_activation.value[True]:
            service_activation_timeout = sdk2.parameters.Integer("Service activation timeout (minutes)", default=4 * 60)
            remove_not_ready_service = sdk2.parameters.Bool("Remove not ready service", default=True)

        run_stability_shoot = sdk2.parameters.Bool(
            "Run stability shoot",
            description="Check if created hamster does not break yabs-server stability",
            default=True,
        )

        with sdk2.parameters.Group("Testenv options") as testenv_options:
            testenv_switch_trigger = sdk2.parameters.Bool(
                "Make hamster usable for testenv",
                description="Set attribute \"testenv_switch_trigger={}\" on spec resource".format(GOALNET_SERVICE_TAG),
                default=False,
                do_not_copy=True)

        with sdk2.parameters.Output():
            external_service_spec = sdk2.parameters.Resource('External service spec resource', resource_type=YabsServerExtServiceEndpoint)

    def is_service_ready(self, stage_id, deploy_unit_id, yp_token):
        if is_deploy_unit_ready(stage_id, deploy_unit_id, yp_token):
            return True

        with self.memoize_stage.wait_service_activation(commit_on_entrance=False, commit_on_wait=False):
            if self.Context.activation_start is NotExists:
                self.Context.activation_start = int(time.time())

            activation_start_datetime = datetime.datetime.fromtimestamp(self.Context.activation_start)
            activation_timeout = datetime.timedelta(minutes=self.Parameters.service_activation_timeout)

            time_until_timeout = activation_start_datetime + activation_timeout - datetime.datetime.now()
            if time_until_timeout.total_seconds() < 0:
                return False

            time_to_wait = min(datetime.timedelta(minutes=10), time_until_timeout).total_seconds()
            raise sdk2.WaitTime(time_to_wait)

    def create_deploy_stage(self, goalnet_binary_resource_id, goalnet_bases_layer_resource_id, goalnet_config_resource_id, base_shards, yp_token):
        def find_object_by_id(objects, _id):
            try:
                return next(six.moves.filter(lambda obj: obj["id"] == _id, objects))
            except StopIteration:
                return None

        # load stage from resource or parameter
        stage_yml = load_template("stage.yml", templates_dir=os.path.dirname(__file__))
        stage = yaml.full_load(stage_yml)

        stage_id = "yabs-goalnet-hamster-{uuid}".format(uuid=str(uuid.uuid4()))
        stage["meta"]["id"] = stage_id
        deploy_unit_id = "deployUnit"
        pod_agent_payload_spec = stage["spec"]["deploy_units"][deploy_unit_id]["replica_set"]["replica_set_template"]["pod_template_spec"]["spec"]["pod_agent_payload"]["spec"]

        # add resource layers
        layers = [
            {
                "checksum": "EMPTY:",
                "id": "goalnet-binary",
                "url": "sbr:{}".format(goalnet_binary_resource_id),
            },
            {
                "checksum": "EMPTY:",
                "id": "goalnet-config",
                "url": "sbr:{}".format(goalnet_config_resource_id),
            },
            {
                "checksum": "EMPTY:",
                "id": "goalnet-bases",
                "url": "sbr:{}".format(goalnet_bases_layer_resource_id),
            },
        ]
        pod_agent_payload_spec["resources"]["layers"] += layers

        # add layer_refs to goalnet-box
        goalnet_box = find_object_by_id(pod_agent_payload_spec["boxes"], "goalnet-box")
        goalnet_box["rootfs"]["layer_refs"] += [layer["id"] for layer in layers]

        # add goalnet-box command line
        goalnet_box["init"].append({
            "command_line": "bash -c 'mkdir /tmpfs/srv; ln -s /{} /tmpfs/srv/bases'".format(BASES_DIR),
        })

        # set goalnet-workload start command line
        if base_shards:
            goalnet_workload = find_object_by_id(pod_agent_payload_spec["workloads"], "goalnet-workload")
            goalnet_workload["start"]["command_line"] = "/goalnetsrv -c ./server.cfg --base-shards {}".format(
                ','.join(str(shard) for shard in base_shards)
            )

        output_stage_path = os.path.join(get_logs_folder(), "{}.yml".format(stage_id))
        logger.debug("Save stage to %s", output_stage_path)
        with open(output_stage_path, "w") as f:
            yaml.dump(stage, f)

        put_stage(output_stage_path, yp_token)

        return stage_id, deploy_unit_id

    def on_execute(self):
        yt_token = self.Parameters.tokens.data()["yt_token"]
        yp_token = self.Parameters.tokens.data()["yp_token"]

        goalnet_binary_resource_id = get_goalnet_binary_resource_id(self)
        base_shards = get_base_shards(self)
        goalnet_bases_layer_resource_id = get_goalnet_bases_layer_resource_id(self, yt_token=yt_token)
        goalnet_config_resource_id = get_goalnet_config_resource_id(self)

        with self.memoize_stage.create_deploy_stage(commit_on_entrance=False):
            self.Context.stage_id, self.Context.deploy_unit_id = self.create_deploy_stage(
                goalnet_binary_resource_id, goalnet_bases_layer_resource_id, goalnet_config_resource_id,
                base_shards, yp_token)
            stage_url = html_hyperlink(
                link="https://deploy.yandex-team.ru/stages/{}".format(self.Context.stage_id),
                text=self.Context.stage_id)
            self.set_info("Created Deploy stage: {}".format(stage_url), do_escape=False)

        if self.Parameters.wait_service_activation:
            service_is_ready = self.is_service_ready(
                self.Context.stage_id, self.Context.deploy_unit_id, yp_token)
            if not service_is_ready:
                stage_status = get_stage_status(self.Context.stage_id, yp_token)
                stage_errors = get_stage_errors(stage_status)
                if stage_errors:
                    self.set_info("Stage failures\n\n{}".format("\n\n".join(stage_errors)))

                pod_statuses = get_pod_statuses(
                    self.Context.stage_id, self.Context.deploy_unit_id, YP_CLUSTER, yp_token)
                pod_errors = get_pod_errors(pod_statuses)
                if pod_errors:
                    self.set_info("Pod failures\n\n{}".format("\n\n".join(pod_errors)))

                stage_status_resource_id = resource_from_json(
                    self, YpObjectStatus, "stage_status", stage_status, description="Stage status")
                pod_statuses_resource_id = resource_from_json(
                    self, YpObjectStatus, "pod_statuses", pod_statuses, description="Pod statuses")
                self.set_info(
                    "YP object statuses\n  - stage {}\n  - pods {}".format(
                        get_resource_html_hyperlink(stage_status_resource_id),
                        get_resource_html_hyperlink(pod_statuses_resource_id),
                    ),
                    do_escape=False)

                if self.Parameters.remove_not_ready_service:
                    remove_stage(self.Context.stage_id, yp_token)
                    self.set_info("Stage {} was removed".format(self.Context.stage_id))
                raise TaskFailure("Service activation timeout")

        with self.memoize_stage.create_spec(commit_on_entrance=False):
            self.Context.external_service_spec_resource_id = create_hamster_spec_resource(
                cluster=YP_CLUSTER,
                endpoint_set="{}.{}".format(self.Context.stage_id, self.Context.deploy_unit_id),
                service_id=self.Context.stage_id,
                service_type=ExternalServiceType.DEPLOY,
                service_tag=GOALNET_SERVICE_TAG,
            ).id
            self.set_info(
                "External service spec: {spec_url}".format(
                    spec_url=get_resource_html_hyperlink(self.Context.external_service_spec_resource_id)),
                do_escape=False)

        if self.Parameters.run_stability_shoot:
            with self.memoize_stage.run_stability_shoot(commit_on_entrance=False):
                logger.debug("Run stability shoot")
                self.Context.stability_shoot_task_id = run_stability_shoot(
                    self,
                    "YABS_SERVER_402_FT_STABILITY_BS_SAMPLED",
                    ext_service_tag=GOALNET_SERVICE_TAG,
                    ext_service_endpoint_resource_id=self.Context.external_service_spec_resource_id,
                )
                self.set_info(
                    "Check yabs-server stability with created hamster: {}"
                    .format(get_task_html_hyperlink(self.Context.stability_shoot_task_id)),
                    do_escape=False,
                )

            check_tasks(self, self.Context.stability_shoot_task_id)

        if self.Parameters.testenv_switch_trigger:
            sdk2.Resource[self.Context.external_service_spec_resource_id].testenv_switch_trigger = GOALNET_SERVICE_TAG

        self.Parameters.external_service_spec = self.Context.external_service_spec_resource_id
