import json
import logging

from sandbox import sdk2
from sandbox.common.types.notification import Transport
from sandbox.projects.yabs.qa.pipeline.stage import stage
from sandbox.projects.yabs.qa.spec.constants import META_MODES
from sandbox.projects.yabs.qa.spec.parameters import BaseSpecParameters
from sandbox.projects.yabs.qa.spec.misc import create_spec_resource, fail_if_subtasks_not_ok
from sandbox.projects.yabs.qa.spec.stages import (
    get_ammo_resources,
    get_binary_base_tags,
    get_data_resources,
    get_engine_resources,
    get_linear_models_data,
    get_misc_resources,
    get_shard_nums,
    get_shard_keys,
    launch_ft_shoot_tasks,
    launch_ft_stability_shoot_tasks,
    launch_ft_validation,
    launch_generate_linear_models_service_data,
    launch_meta_load_baseline,
    launch_setup_ya_make,
    launch_stat_load_baseline,
)
from sandbox.projects.yabs.qa.tasks.YabsServerCreateOneShotSpec.spec import YABS_SERVER_HOUSE_SPEC
from sandbox.projects.yabs.qa.tasks.YabsServerCreateOneShotSpec.stages import (
    check_subtasks_status,
    get_binary_bases,
    get_spec_data,
    launch_chkdb,
    launch_make_bin_bases,
)


_STAGES = [
    get_engine_resources,
    get_ammo_resources,
    get_misc_resources,
    get_data_resources,
    get_shard_nums,
    get_shard_keys,
    launch_setup_ya_make,
    get_binary_base_tags,
    launch_ft_validation,
    launch_ft_shoot_tasks,
    launch_ft_stability_shoot_tasks,
    launch_meta_load_baseline,
    launch_make_bin_bases,
    get_binary_bases,
    launch_chkdb,
    launch_generate_linear_models_service_data,
    get_linear_models_data,
    launch_stat_load_baseline,
    get_spec_data,
]


logger = logging.getLogger(__name__)

DEVELOPER_PRE_DEFINED_STAGE_RESULTS = (
    "chkdb_task_id",
    "ft_shoot_tasks",
    "ft_stability_shoot_tasks",
    "ft_validation_tasks",
    "load_shoot_tasks",
    "make_bin_bases_task_id",
    "prepare_stat_stub_tasks",
    "setup_ya_make_task_id",
)


class YabsServerCreateOneShotSpec(sdk2.Task):
    """ Betta yabs_server one shot task
    """

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseSpecParameters):

        with sdk2.parameters.Group("Spec generation settings") as generation_group:
            # deprecated block
            create_load_spec = sdk2.parameters.Bool('Create load spec', default=True)
            with create_load_spec.value[True]:
                add_meta_load = sdk2.parameters.Bool('Add meta in load spec', default=False)
                load_shards = sdk2.parameters.List('LOAD shards', default=['A', 'B', 'C'])
            generate_separated_bases = sdk2.parameters.Bool('Generate additional yabs, bs bases', default=False)
            generate_not_separated_bases = sdk2.parameters.Bool('Generate bases that have additional (yabs, bs)', default=True)

        with sdk2.parameters.Group("Resources") as resources_group:
            request_log_resource_id_map = sdk2.parameters.JSON("Role to request_log resource id mapping, will find last resources if empty", default={meta_mode: None for meta_mode in META_MODES})

        with sdk2.parameters.Group("Developer options") as developer_group:
            debug_mode = sdk2.parameters.Bool('Debug mode', default=False)
            with debug_mode.value[True]:
                reusable_stages = sdk2.parameters.CheckGroup(
                    'Reusable stages',
                    description='If the task is a clone, then copy the stages results from the original task',
                    choices=[
                        (_stage.__name__, _stage.__name__)
                        for _stage in _STAGES
                    ],
                    default=None,
                )

            enable_developer_options = sdk2.parameters.Bool("Enable developer options", default=False)
            with enable_developer_options.value[True]:
                setup_ya_make_task_id = sdk2.parameters.Integer("Setup task id")
                make_bin_bases_task_id = sdk2.parameters.Integer("Make bin bases task id")
                chkdb_task_id = sdk2.parameters.Integer("Chkdb task id")
                ft_shoot_tasks = sdk2.parameters.JSON("FT shoot tasks")
                ft_stability_shoot_tasks = sdk2.parameters.JSON("FT stability shoot tasks")
                ft_validation_tasks = sdk2.parameters.JSON("FT validation shoot tasks")
                load_shoot_tasks = sdk2.parameters.JSON("Load shoot tasks")
                prepare_stat_stub_tasks = sdk2.parameters.JSON("Prepare stub stat tasks for meta load shoot")

    def on_create(self):
        super(YabsServerCreateOneShotSpec, self).on_create()
        self.Parameters.tags = list(set(self.Parameters.tags + ['NEW_SPEC', 'ONESHOT-TEST']))
        if self.Context.copy_of:
            self.Parameters.notifications = [
                notification for notification in self.Parameters.notifications
                if notification.transport != Transport.JUGGLER
            ]

    def on_save(self):
        super(YabsServerCreateOneShotSpec, self).on_save()

        if not (self.Parameters.generate_separated_bases or self.Parameters.generate_not_separated_bases):
            raise ValueError('Either generate_separated_bases or generate_not_separated_bases should be set to True')

    @stage(provides=DEVELOPER_PRE_DEFINED_STAGE_RESULTS, result_is_dict=True)
    def load_developer_options(self):
        result = {}
        for parameter_name in DEVELOPER_PRE_DEFINED_STAGE_RESULTS:
            result[parameter_name] = getattr(self.Parameters, parameter_name) or None
        return result

    def on_execute(self):
        if self.Parameters.debug_mode:
            self.Context.__reusable_stages = self.Parameters.reusable_stages

        get_engine_resources(self)
        get_ammo_resources(self)
        get_misc_resources(self, hamster_ext_service_tags=self.Parameters.hamster_ext_service_tags)
        get_data_resources(self)

        get_shard_nums(self)
        get_shard_keys(self)

        if self.Parameters.enable_developer_options:
            self.load_developer_options()

        launch_setup_ya_make(self)
        get_binary_base_tags(self)  # Waits for setup_ya_make_task

        launch_make_bin_bases(self)
        launch_generate_linear_models_service_data(self)

        get_linear_models_data(self)

        get_binary_bases(self)  # Waits for make_bin_bases_task
        launch_chkdb(self)

        launch_ft_shoot_tasks(
            self,
            hamster_ext_service_tags=self.Parameters.hamster_ext_service_tags,
            headers_update_dict={  # TODO: Remove after switching to new test oneshot task
                "x-yabs-debug-options-json": json.dumps({
                    "business": False,
                    "trafaret": False,
                    "logs": True,
                    "trace": False,
                    "keywords": False,
                    "mx_zero_features": False,
                    "ext_http_entities": True,
                    "mx": False,
                    "exts": True,
                    "debug_log": False,
                    # Differ from default (WHY??)
                    "filter_log": False,
                    "match_log": False,
                }),
            }
        )
        launch_ft_stability_shoot_tasks(
            self,
            hamster_ext_service_tags=self.Parameters.hamster_ext_service_tags,
        )

        launch_stat_load_baseline(self)  # Optional
        launch_meta_load_baseline(self)  # Optional

        launch_ft_validation(self)

        broken_subtasks, failed_subtasks = check_subtasks_status(self)
        fail_if_subtasks_not_ok(self, broken_subtasks, failed_subtasks)

        spec = get_spec_data(self, use_separated_meta_and_stat=self.Parameters.use_separated_meta_and_stat)
        spec_resource_attrs = {
            "good": True,
            "released_spec": self.Parameters.release_spec,
        }
        if self.Parameters.meta_load and self.Parameters.stat_load:
            spec_resource_attrs["good_with_load"] = True
        create_spec_resource(
            self,
            YABS_SERVER_HOUSE_SPEC,
            spec,
            'Oneshot spec',
            'oneshot_spec.json',
            **spec_resource_attrs
        )
