import json
import logging

from sandbox import sdk2

import sandbox.projects.release_machine.components.all as rmc
from sandbox.projects.release_machine.helpers import svn_helper as rm_svn
from sandbox.projects.common.testenv_client.api_client import TestenvApiClient
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_STAT_SPEC, BS_RELEASE_TAR, BS_RELEASE_YT
from sandbox.projects.yabs.qa.utils.general import get_resource_html_hyperlink
from sandbox.projects.yabs.qa.spec.misc import find_last_released_resource

logger = logging.getLogger(__name__)


TE_DATABASE = "yabs-2.0"


def job_run_on_revision(job_name, revision, te_client, te_base=TE_DATABASE):
    params = dict(
        revision_gte=revision,
        revision_lte=revision,
    )
    job_runs = te_client.projects[te_base]['jobs'][job_name]['runs'].GET(params=params)
    try:
        return job_runs[0]
    except IndexError:
        logger.error("No %s runs on revision %s", job_name, revision)
        return None


def get_stat_resources(revision, ci_token):
    te_client = TestenvApiClient(token=ci_token)

    build_release_job_run = job_run_on_revision("YABS_SERVER_11_BUILD_RELEASE", revision, te_client)
    build_release_task_id = build_release_job_run["sandbox_task"]["task_id"]
    content_system_resource = sdk2.Task[build_release_task_id].Parameters.bs_release_yt_resource
    stat_server_resource = sdk2.Task[build_release_task_id].Parameters.bs_release_tar_resource

    return stat_server_resource, content_system_resource


def get_stat_resources_and_revision(stat_server_resource_search_politic, ci_token):
    if stat_server_resource_search_politic == 'last_stable_release_resource':
        bs_release_tar_resource = find_last_released_resource(BS_RELEASE_TAR, {"released": "stable", "build_mode": "release", "component_name": "yabs_server"})
        bs_release_yt_resource = find_last_released_resource(BS_RELEASE_YT, {"build_mode": "release"}, task_id=bs_release_tar_resource.task.id)

        return bs_release_tar_resource, bs_release_yt_resource, bs_release_yt_resource.revision

    component_info = rmc.get_component('yabs_server')

    if stat_server_resource_search_politic == 'last_stable_branch_revision':
        revision = component_info.first_rev - 1
    elif stat_server_resource_search_politic == 'last_branch_revision':
        last_scope_num = component_info.last_scope_num
        release_path = component_info.full_scope_path(last_scope_num)
        revision = rm_svn.SvnHelper.get_last_trunk_revision_before_copy(
            release_path,
            repo_base_url=component_info.svn_cfg__repo_base_url,
            trunk_url=component_info.svn_cfg__trunk_url,
        )
        logger.info('Detected last branch trunk revision %s by last scope %s', revision, last_scope_num)
    else:
        raise ValueError('Unknown politic: {}'.format(stat_server_resource_search_politic))

    bs_release_tar_resource, bs_release_yt_resource = get_stat_resources(revision, ci_token)
    return bs_release_tar_resource, bs_release_yt_resource, revision


def create_spec_resource(
        stat_server_resource,
        content_system_resource,
        revision,
        testenv_switch_trigger
):
    spec_data = {
        "bs_release_tar_resource_id": stat_server_resource.id,
        "bs_release_yt_resource_id": content_system_resource.id,
    }

    spec_filename = "stat_spec.json"
    with open(spec_filename, "w") as f:
        json.dump(spec_data, f)

    attrs = {
        "revision": revision,
    }
    if testenv_switch_trigger:
        attrs["testenv_switch_trigger"] = testenv_switch_trigger
    resource = YABS_SERVER_STAT_SPEC(
        task=sdk2.Task.current,
        description="Yabs server stat spec",
        path=spec_filename,
        **attrs
    )
    sdk2.ResourceData(resource).ready()

    return resource


def find_stat_spec_resource(revision):
    return YABS_SERVER_STAT_SPEC.find(
        attrs={"revision": revision}
    ).order(-sdk2.Resource.id).first()


class YabsServerCreateStatSpec(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        push_tasks_resource = True

        tokens = sdk2.parameters.YavSecret("YAV secret identifier", default="sec-01d6apzcex5fpzs5fcw1pxsfd5")
        reuse_spec = sdk2.parameters.Bool("Reuse existing spec", default=True)
        testenv_switch_trigger = sdk2.parameters.Bool(
            "Allow to use spec in testenv",
            description="Set attribute \"testenv_switch_trigger\" on spec resource",
            default=False,
            do_not_copy=True)
        stat_server_resource_search_politic = sdk2.parameters.String(
            'Stat resource search politic', choices=[
                ('Last created branch revision', 'last_branch_revision'),
                ('Last stable branch revision', 'last_stable_branch_revision'),
                ('Last stable released resource', 'last_stable_release_resource'),
            ],
            default='last_branch_revision',
        )

        with sdk2.parameters.Output:
            stat_spec_resource = sdk2.parameters.Resource("Stat spec resource")

    def on_execute(self):
        bs_release_tar_resource, bs_release_yt_resource, stable_stat_revision = get_stat_resources_and_revision(
            self.Parameters.stat_server_resource_search_politic, self.Parameters.tokens.data()["ci.token"]
        )
        logger.info("Stable stat revision: %s", stable_stat_revision)

        if self.Parameters.reuse_spec:
            existing_spec_resource = find_stat_spec_resource(stable_stat_revision)
            if existing_spec_resource:
                self.set_info(
                    "Reuse existing spec resource for revision {revision}: {resource_link}".format(
                        revision=stable_stat_revision,
                        resource_link=get_resource_html_hyperlink(existing_spec_resource.id),
                    ),
                    do_escape=False)
                self.Parameters.stat_spec_resource = existing_spec_resource
                return

        logger.info("Stat server resource: %s", bs_release_tar_resource.id)
        logger.info("Content system resource: %s", bs_release_yt_resource.id)

        testenv_switch_trigger = str(self.id) if self.Parameters.testenv_switch_trigger else None
        self.Parameters.stat_spec_resource = create_spec_resource(
            bs_release_tar_resource,
            bs_release_yt_resource,
            stable_stat_revision,
            testenv_switch_trigger,
        )
        logger.info("Stat spec resource: %s", self.Parameters.stat_spec_resource)
