import datetime
import difflib
import logging
import os
import time
import uuid

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.misc import NotExists
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.common.yabs.server.util.general import (
    check_tasks,
    find_last_ready_resource,
)
from sandbox.projects.websearch.begemot.resources import BEGEMOT_REALTIME_PACKAGE
from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.yabs.qa.hamster.nanny.create import (
    AllocationRequest,
    create_hamster_service,
    PersistentVolume,
)
from sandbox.projects.yabs.qa.hamster.nanny.utils import get_active_snapshot_id
from sandbox.projects.yabs.qa.hamster.stability import (
    check_service_stability,
    get_ext_requests_from_yt,
    run_stability_shoot,
)
from sandbox.projects.yabs.qa.hamster.nanny.remove import remove_nanny_service
from sandbox.projects.yabs.qa.hamster.spec import create_hamster_spec_resource, ExternalServiceType
from sandbox.projects.yabs.qa.hamster.http_request import HTTPRequest
from sandbox.projects.yabs.qa.resource_types import YabsServerExtServiceEndpoint, YabsServerExtServiceStabilityReport
from sandbox.projects.yabs.qa.utils.general import (
    get_resource_html_hyperlink,
    get_task_html_hyperlink,
    html_hyperlink,
)
from sandbox.sandboxsdk.paths import get_logs_folder


logger = logging.getLogger(__name__)

SERVICE_TAG = "rsya_hit_models_heavy_01"
PRODUCTION_SERVICE_ID = "sas_yabs_hit_models"
REALTIME_WIZARD_DATA_RESOURCE_ATTRS = {
    "released": "stable",
    "shard": "YabsHitModelsShard"
}
# https://wiki.yandex-team.ru/runtime-cloud/nanny/howto-manage-service/#instance-tags
INSTANCE_TAGS = {
    "metaprj": "unknown",
    "itype": "yabs_hit_models_hamster",
    "ctype": "test",
    "prj": "bs"
}


def get_nanny_service_resources(nanny_client, production_service_id, resource_type, blacklist=()):
    logger.debug("Get %s from nanny service %s", resource_type, production_service_id)
    active_runtime_attrs = nanny_client.get_service_active_runtime_attrs(production_service_id)
    logger.debug("Active snapshot id: %s", active_runtime_attrs["_id"])
    service_files = active_runtime_attrs["content"]["resources"][resource_type]
    logger.debug("%s: %s", resource_type, service_files)

    blacklist = set(blacklist)
    return [
        _file
        for _file in service_files
        if _file.get("local_path") not in blacklist
    ]


def ensure_trailing_newline(s):
    if not s.endswith("\n"):
        s += "\n"
    return s


def get_static_files(nanny_client, production_service_id):
    from library.python import resource

    production_static_files = get_nanny_service_resources(
        nanny_client,
        production_service_id,
        resource_type="static_files",
    )

    hamster_static_files = {
        static_file["local_path"]: static_file
        for static_file in production_static_files
    }

    diffs_dir = os.path.join(get_logs_folder(), 'static_files_diff')
    if not os.path.exists(diffs_dir):
        os.mkdir(diffs_dir)

    custom_files = {
        "instancectl.conf": "/instancectl.conf",
        "testing_config.cfgproto.txt": "/testing_config.cfgproto.txt",
    }

    for custom_file_local_path in custom_files:
        # load file from resource
        resource_key = custom_files[custom_file_local_path]
        custom_static_file_content = ensure_trailing_newline(resource.find(resource_key).decode("utf-8"))

        # check if file exists
        if custom_file_local_path in hamster_static_files:
            logger.debug("File \"%s\" exists both in production files and in custom files", custom_file_local_path)

            # ensure file has trailing newline so that the diff works correctly
            production_file = hamster_static_files[custom_file_local_path]
            production_static_file_content = ensure_trailing_newline(production_file["content"])

            # calculate diff
            diff_lines = difflib.unified_diff(
                production_static_file_content.splitlines(True),
                custom_static_file_content.splitlines(True),
                fromfile="{} (production)".format(custom_file_local_path),
                tofile="{} (hamster)".format(custom_file_local_path),
            )
            diff_path = os.path.join(diffs_dir, custom_file_local_path + ".diff")
            with open(diff_path, "w") as f:
                f.writelines(diff_lines)
            logger.debug("Saved diff to %s", diff_path)

        hamster_static_files[custom_file_local_path] = {
            "content": custom_static_file_content,
            "local_path": custom_file_local_path,
        }

    return list(hamster_static_files.values())


def get_sandbox_files(nanny_client, production_service_id, realtime_wizard_data_resource):
    hamster_sandbox_files = get_nanny_service_resources(
        nanny_client,
        production_service_id,
        resource_type="sandbox_files",
        blacklist=["bstr", "bstr_caller", "callback", "gperftools"]
    )

    hamster_sandbox_files.append({
        "task_type": "BUILD_YABS_HIT_MODELS_PACKAGE",
        "task_id": str(realtime_wizard_data_resource.task_id),
        "resource_id": str(realtime_wizard_data_resource.id),
        "local_path": "realtime_wizard_data.tar",
        "resource_type": "BEGEMOT_REALTIME_PACKAGE",
    })

    return hamster_sandbox_files


class YabsServerCreateYabsHitModelsHamster(BaseBinTask):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 512

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseBinTask.Parameters):
        resource_attrs = BaseBinTask.Parameters.resource_attrs(
            default={"task_type": "YABS_SERVER_CREATE_YABS_HIT_MODELS_HAMSTER"})
        release_version = BaseBinTask.Parameters.release_version(default=ReleaseStatus.STABLE)

        remove_broken_service = sdk2.parameters.Bool("Remove broken service", default=True)
        wait_service_activation = sdk2.parameters.Bool("Wait for service activation", default=True)
        with wait_service_activation.value[True]:
            service_activation_timeout = sdk2.parameters.Integer("Service activation timeout (minutes)", default=3 * 60)
            check_service_stability = sdk2.parameters.Bool(
                "Check service responses stability",
                description="Send the same request to all service's instances and compare responses",
                default=True,
            )
            with check_service_stability.value[True]:
                stability_check_requests_count = sdk2.parameters.Integer("Requests count", default=100)
                remove_unstable_service = sdk2.parameters.Bool("Remove unstable service", default=True)
            run_stability_shoot = sdk2.parameters.Bool(
                "Run stability shoot",
                description="Check if created hamster does not break yabs-server stability",
                default=True,
            )

        tokens = sdk2.parameters.YavSecret("YAV secret identifier", default="sec-01f8yr9r918932eypfrd0evrd7")
        service_category = sdk2.parameters.String(
            "Service category",
            description="Category to put service in",
            default="/yabs/testing/hamster/yabs_hit_models",
        )
        abc_service_id = sdk2.parameters.Integer(
            "ABC service ID",
            description="For YP quotas accounting and host infrastructure maintenance notifications",
            default=34888,
            required=True
        )
        admin_abc_services = sdk2.parameters.List(
            "Administrator ABC services",
            description="ABC service ids",
            default=[34888, 34887, 2409],
        )
        cluster = sdk2.parameters.String("Cluster", choices=[("SAS", "SAS")], default="SAS")
        production_service_id = sdk2.parameters.String(
            "Production service id",
            description="Will get static and sandbox files from this service",
            default=PRODUCTION_SERVICE_ID,
        )
        realtime_wizard_data_resource = sdk2.parameters.Resource(
            "realtime_wizard_data resource",
            multiple=False,
            resource_type=BEGEMOT_REALTIME_PACKAGE,
            attrs=REALTIME_WIZARD_DATA_RESOURCE_ATTRS,
        )
        with sdk2.parameters.Group("YP.Lite allocation request") as allocation_request:
            use_tmp_quota = sdk2.parameters.Bool("Use resources from temporary quota", default=False)
            replicas = sdk2.parameters.Integer("Replicas count", default=1)
            snapshots_count = sdk2.parameters.Integer("Snapshots count", default=3)
            root_volume_storage_class = sdk2.parameters.String("Root volume storage class", default="hdd")
            root_fs_quota_gigabytes = sdk2.parameters.Integer("Root volume quota for single snapshot (GB)", default=10)
            work_dir_quota_gigabytes = sdk2.parameters.Integer("Work dir quota for single snapshot (GB)", default=100)
            root_bandwidth_guarantee_megabytes_per_sec = sdk2.parameters.Integer("Bandwidth guarantee (MB/s)", default=20)
            root_bandwidth_limit_megabytes_per_sec = sdk2.parameters.Integer("Bandwidth limit (MB/s)", default=40)
            vcpu_guarantee = sdk2.parameters.Integer(
                "CPU guarantee",
                description="Milliseconds per second, 1000 for 1 core",
                default=10000)
            memory_guarantee_megabytes = sdk2.parameters.Integer("Memory guarantee (MB)", default=100 * 1024)
            network_macro = sdk2.parameters.String("Network macro", default="_YABS_MODELS_SERVICES_TEST_NETS_")
            network_bandwidth_guarantee_megabytes_per_sec = sdk2.parameters.Integer("Network bandwidth guarantee (MB/s)", default=10)
            network_bandwidth_limit_megabytes_per_sec = sdk2.parameters.Integer("Network bandwidth limit (MB/s)", default=0)

        with sdk2.parameters.Group("Testenv options") as testenv_options:
            testenv_switch_trigger = sdk2.parameters.Bool(
                "Make hamster usable for testenv",
                description="Set attribute \"testenv_switch_trigger={}\" on spec resource".format(SERVICE_TAG),
                default=False,
                do_not_copy=True)

        with sdk2.parameters.Output():
            external_service_spec = sdk2.parameters.Resource('External service spec resource', resource_type=YabsServerExtServiceEndpoint)

    @property
    def realtime_wizard_data_resource(self):
        return self.Parameters.realtime_wizard_data_resource or find_last_ready_resource(BEGEMOT_REALTIME_PACKAGE, attrs=REALTIME_WIZARD_DATA_RESOURCE_ATTRS)

    def create_service(self, service_id):
        tokens = self.Parameters.tokens.data()
        nanny_client = NannyClient(
            api_url="http://nanny.yandex-team.ru/",
            oauth_token=tokens["nanny_token"],
        )

        sandbox_files = get_sandbox_files(
            nanny_client,
            self.Parameters.production_service_id,
            self.realtime_wizard_data_resource)
        logger.debug("Sandbox files: %s", sandbox_files)

        static_files = get_static_files(nanny_client, self.Parameters.production_service_id)
        logger.debug("Static files: %s", static_files)

        endpoint_set_id = service_id.replace("_", "-")

        logs_volume = PersistentVolume(
            disk_quota_gigabytes=5,
            mount_point="/logs",
            storage_class="hdd",
            bandwidth_guarantee_megabytes_per_sec=5,
            bandwidth_limit_megabytes_per_sec=5,
        )

        allocation_request = AllocationRequest(
            replicas=self.Parameters.replicas,
            snapshots_count=self.Parameters.snapshots_count,
            root_volume_storage_class=self.Parameters.root_volume_storage_class,
            root_fs_quota_gigabytes=self.Parameters.root_fs_quota_gigabytes,
            work_dir_quota_gigabytes=self.Parameters.work_dir_quota_gigabytes,
            root_bandwidth_guarantee_megabytes_per_sec=self.Parameters.root_bandwidth_guarantee_megabytes_per_sec,
            root_bandwidth_limit_megabytes_per_sec=self.Parameters.root_bandwidth_limit_megabytes_per_sec,
            vcpu_guarantee=self.Parameters.vcpu_guarantee,
            memory_guarantee_megabytes=self.Parameters.memory_guarantee_megabytes,
            network_macro=self.Parameters.network_macro,
            network_bandwidth_guarantee_megabytes_per_sec=self.Parameters.network_bandwidth_guarantee_megabytes_per_sec,
            network_bandwidth_limit_megabytes_per_sec=self.Parameters.network_bandwidth_limit_megabytes_per_sec,
            persistent_volumes=[logs_volume]
        )

        # https://wiki.yandex-team.ru/runtime-cloud/nanny/service-labels/
        service_labels = [
            {
                "key": "geo",
                "value": self.Parameters.cluster.lower()
            },
            {
                "key": "ctype",
                "value": "hamster"
            },
            {
                "key": "itype",
                "value": "yabs_hit_models"
            },
            {
                "key": "prj",
                "value": "shm"
            }
        ]

        service_description = "Hamster installation of yabs_hit_models service.\nCreated by sandbox task #{}".format(self.id)

        snapshot_id = create_hamster_service(
            description=service_description,
            nanny_token=tokens["nanny_token"],
            staff_token=tokens["staff_token"],
            service_id=service_id,
            service_labels=service_labels,
            cluster=self.Parameters.cluster,
            category=self.Parameters.service_category,
            abc_service_id=self.Parameters.abc_service_id,
            allocation_request=allocation_request,
            instance_tags=INSTANCE_TAGS,
            endpoint_set_id=endpoint_set_id,
            admin_logins=[self.author],
            admin_abc_services=self.Parameters.admin_abc_services,
            static_files=static_files,
            sandbox_files=sandbox_files,
            activate=True,
            use_tmp_quota=self.Parameters.use_tmp_quota,
        )

        return endpoint_set_id, snapshot_id

    def check_service_stability(self, yql_token, nanny_token):
        raw_requests = get_ext_requests_from_yt(yql_token, SERVICE_TAG, limit=self.Parameters.stability_check_requests_count)

        stability_report_dir = "stability_report"
        requests_dir = os.path.join(stability_report_dir, "raw_requests")
        logger.debug("Create dir \"%s\"", requests_dir)
        os.makedirs(requests_dir)

        http_requests = []
        for idx, request in enumerate(raw_requests):
            http_request = HTTPRequest(request)
            http_request.update_query([("hr", "da")])
            http_requests.append(http_request)

            request_file_path = os.path.join(requests_dir, str(idx))
            logger.debug("Save request to \"%s\"", request_file_path)
            with open(request_file_path, "wb") as f:
                f.write(http_request.serialize())

        response_diffs = check_service_stability(self.Context.endpoint_set, self.Parameters.cluster, http_requests)

        for response_diff in response_diffs:
            diff_filename = "{diff.request_id}_{diff.host_a}_{diff.host_b}.diff".format(diff=response_diff)
            with open(os.path.join(stability_report_dir, diff_filename), "w") as f:
                f.write(response_diff.diff)

        stability_report_resource = YabsServerExtServiceStabilityReport(
            self,
            description="yabs_hit_models service stability report",
            path=stability_report_dir)
        sdk2.ResourceData(stability_report_resource).ready()

        self.set_info(
            "Stability report: {}".format(get_resource_html_hyperlink(stability_report_resource.id)),
            do_escape=False)

        if response_diffs:
            if self.Parameters.remove_unstable_service:
                remove_nanny_service(nanny_token, self.Context.nanny_service_id, self.Parameters.cluster)
                self.set_info("Service {} removed".format(self.Context.nanny_service_id))

            raise TaskFailure("Service is unstable.")

    def on_execute(self):
        tokens = self.Parameters.tokens.data()

        with self.memoize_stage.create_service(commit_on_entrance=False):
            self.Context.nanny_service_id = "hamster_yabs_hit_models_{cluster}_{uuid}".format(
                uuid=str(uuid.uuid4()),
                cluster=self.Parameters.cluster.lower(),
            )
            logger.debug("Nanny service id: %s", self.Context.nanny_service_id)

            try:
                self.Context.endpoint_set, self.Context.snapshot_id = self.create_service(self.Context.nanny_service_id)
            except Exception as e:
                logger.exception("Got exception while creating service")
                if self.Parameters.remove_broken_service:
                    logger.info("Remove service %s", self.Context.nanny_service_id)
                    remove_nanny_service(
                        nanny_token=tokens["nanny_token"],
                        service_id=self.Context.nanny_service_id,
                        cluster=self.Parameters.cluster,
                        force=True)
                    self.set_info("Service {} removed".format(self.Context.nanny_service_id))
                raise e

            nanny_service_url = html_hyperlink(
                link="https://nanny.yandex-team.ru/ui/#/services/catalog/{}/".format(self.Context.nanny_service_id),
                text=self.Context.nanny_service_id)
            yasm_url = html_hyperlink(
                text="YASM",
                link="https://yasm.yandex-team.ru/template/panel/yabs_hit_models/nanny={service_id};ctype={ctype};itype={itype}".format(
                    service_id=self.Context.nanny_service_id,
                    ctype=INSTANCE_TAGS["ctype"],
                    itype=INSTANCE_TAGS["itype"],
                ))
            self.set_info(
                "Nanny service: {service_url}\nMonitoring: {yasm_url}".format(
                    service_url=nanny_service_url,
                    yasm_url=yasm_url),
                do_escape=False)

        with self.memoize_stage.create_spec(commit_on_entrance=False):
            endpoint_set = self.Context.endpoint_set
            cluster = self.Parameters.cluster.lower()
            self.Context.external_service_spec_resource_id = create_hamster_spec_resource(
                cluster=cluster,
                endpoint_set=endpoint_set,
                service_id=self.Context.nanny_service_id,
                service_type=ExternalServiceType.NANNY,
                service_tag=SERVICE_TAG,
                meta_env={
                    "rsya_hit_models_prod_heavy_01_endpoint_set": endpoint_set,
                    "rsya_hit_models_cluster": cluster,
                }
            ).id

            self.set_info(
                "External service spec: {spec_url}".format(
                    spec_url=get_resource_html_hyperlink(self.Context.external_service_spec_resource_id)),
                do_escape=False)

        if self.Parameters.wait_service_activation:
            with self.memoize_stage.wait_service_activation(commit_on_entrance=False, commit_on_wait=False):
                if self.Context.activation_start is NotExists:
                    self.Context.activation_start = time.time()

                activation_start_datetime = datetime.datetime.fromtimestamp(self.Context.activation_start)
                activation_timeout = datetime.timedelta(minutes=self.Parameters.service_activation_timeout)
                if (datetime.datetime.now() - activation_start_datetime) > activation_timeout:
                    raise TaskFailure("Service activation timeout")

                active_snapshot_id = get_active_snapshot_id(tokens["nanny_token"], self.Context.nanny_service_id)
                if self.Context.snapshot_id != active_snapshot_id:
                    raise sdk2.WaitTime(datetime.timedelta(minutes=10).total_seconds())

            if self.Parameters.check_service_stability:
                with self.memoize_stage.check_service_stability(commit_on_entrance=False):
                    self.check_service_stability(tokens["yql_token"], tokens["nanny_token"])

            if self.Parameters.run_stability_shoot:
                with self.memoize_stage.run_stability_shoot(commit_on_entrance=False):
                    logger.debug("Run stability shoot")
                    self.Context.stability_shoot_task_id = run_stability_shoot(
                        self,
                        "YABS_SERVER_402_FT_STABILITY_BS_SAMPLED",
                        ext_service_tag=SERVICE_TAG,
                        ext_service_endpoint_resource_id=self.Context.external_service_spec_resource_id,
                    )
                    self.set_info(
                        "Check yabs-server stability with created hamster: {}"
                        .format(get_task_html_hyperlink(self.Context.stability_shoot_task_id)),
                        do_escape=False,
                    )

                check_tasks(self, self.Context.stability_shoot_task_id)

        if self.Parameters.testenv_switch_trigger:
            sdk2.Resource[self.Context.external_service_spec_resource_id].testenv_switch_trigger = SERVICE_TAG

        self.Parameters.external_service_spec = self.Context.external_service_spec_resource_id
