import logging
import os
import subprocess
import textwrap

import requests

from sandbox import sdk2
from sandbox.common import rest
from sandbox.common.errors import TaskFailure
from sandbox.projects.yabs.qa.brave_tests.base_revision import get_base_revision
from sandbox.projects.yabs.qa.brave_tests.common import TestType, TESTENV_TEST_NAMES
from sandbox.projects.yabs.qa.hamster.utils import (
    get_bad_hamsters,
    get_hamster_url,
)
from sandbox.projects.yabs.qa.utils.arcanum import comment_review as comment_arcanum_review
from sandbox.projects.yabs.qa.utils.general import (
    get_resource_html_hyperlink,
    get_task_html_hyperlink,
    get_task_markdown_hyperlink,
    html_hyperlink,
)
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sdk2.vcs import svn


logger = logging.getLogger(__name__)


def upload_to_paste(text, oauth_token, syntax="plain"):
    """Upload text to paste.yandex-team.ru
    API Docs: https://paste.yandex-team.ru/api/v1/docs

    :param text: Text to upload
    :type text: str
    :param oauth_token: OAuth token without specified scope
    :type oauth_token: str
    :param syntax: Text syntax, defaults to "plain"
    :type syntax: str, optional
    :return: Link to textual representation on paste.yandex-team.ru or None if upload failed
    :rtype: [str|None]
    """
    response = requests.post(
        "https://paste.yandex-team.ru/api/v1/pastes",
        headers={
            "Authorization": "OAuth {}".format(oauth_token),
        },
        json={
            "text": text,
            "syntax": syntax,
        },
    )
    if not response.ok:
        logger.debug("response: %s %s", response.status_code, response.text)
        return None

    paste_uuid = response.json()["uuid"]
    return "https://paste.yandex-team.ru/{}/text".format(paste_uuid)


class OutputParameters(sdk2.Parameters):
    base_revision = sdk2.parameters.Integer("Base revision")
    bs_task_id = sdk2.parameters.Integer("BS task id")
    yabs_task_id = sdk2.parameters.Integer("YABS task id")
    bsrank_task_id = sdk2.parameters.Integer("BSRANK task id")

    meta_bs_task_id = sdk2.parameters.Integer("meta with stable stat BS task id")
    meta_yabs_task_id = sdk2.parameters.Integer("meta with stable stat YABS task id")
    meta_bsrank_task_id = sdk2.parameters.Integer("meta with stable stat BSRANK task id")


class YabsServerFindBraveTestsBaseRevision(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        ssh_key_owner = sdk2.parameters.String("SSH key owner", default="robot-yabs-cs-sb")
        yav_secret = sdk2.parameters.YavSecret(
            "Yav secret",
            default="sec-01d6apzcex5fpzs5fcw1pxsfd5")

        test_type = sdk2.parameters.String(
            "Test type",
            choices=[(test_type.value, test_type.value) for test_type in TestType],
            default=TestType.FT.value)
        base_revision_upper_limit = sdk2.parameters.Integer(
            "Base revision upper limit",
            description="Found base revision will be less or equal. Empty means no limit.")
        arcanum_review_id = sdk2.parameters.Integer("Arcanum review ID")
        testenv_mode = sdk2.parameters.Bool(
            "The task is running in TestEnv",
            description="Return immediately if self.Context.is_patch_check evaluates to False",
            default=True,
        )
        check_hamsters_readiness = sdk2.parameters.Bool(
            "Check hamsters readiness",
            description="Fail if any found shoot task has not ready hamster",
            default=True,
        )

        with sdk2.parameters.Output():
            output_parameters = OutputParameters()
            arcadia_patch = sdk2.parameters.String("Arcadia patch")

    def get_patch(self, base_revision, arcanum_review_id):
        arcadia_dir = "arcadia"

        ya = svn.Arcadia.export("arcadia:/arc/trunk/arcadia/ya", os.path.realpath("ya"))

        cmd = [
            ya, "clone",
            "-r", str(base_revision),
            "--no-junk",
            arcadia_dir
        ]
        logger.debug("Run: \"%s\"", " ".join(cmd))
        with sdk2.helpers.ProcessLog(self, logger="ya_clone") as pl:
            subprocess.check_call(
                cmd,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

        os.chdir(arcadia_dir)

        env = os.environ
        env["YA_USER"] = self.Parameters.ssh_key_owner
        logger.debug("env: %s", env)

        private_ssh_key = self.Parameters.yav_secret.data()["ssh_key_private"]

        cmd = [
            ya, "pr", "checkout",
            "--draft",
            str(arcanum_review_id),
        ]
        logger.debug("Run: \"%s\"", " ".join(cmd))
        with sdk2.helpers.ProcessLog(self, logger="ya_pr_checkout") as pl, ssh.Key(private_part=private_ssh_key):
            subprocess.check_call(
                cmd,
                stdout=pl.stdout,
                stderr=pl.stderr,
                env=env,
            )

        patch = svn.Arcadia.diff(url=".")
        return patch

    def check_hamsters_readiness(self, task_ids, nanny_token, yp_token):
        """Return error if any shoot task has not ready hamster.

        :return: Error in html format
        :rtype: str
        """
        bad_hamsters = {
            task_id: get_bad_hamsters(task_id, nanny_token, yp_token)
            for task_id in task_ids
        }
        logger.debug("bad hamsters: %s", bad_hamsters)

        error_message = ""
        for task_id, task_bad_hamsters in bad_hamsters.items():
            if not task_bad_hamsters:
                continue

            error_message += get_task_html_hyperlink(task_id) + "\n"
            for service_tag, resource_id in task_bad_hamsters.items():
                error_message += "  {hamster_link} {resource_link}\n".format(
                    hamster_link=html_hyperlink(get_hamster_url(resource_id), service_tag),
                    resource_link=get_resource_html_hyperlink(resource_id),
                )
            error_message += "\n"

        return error_message

    def on_execute(self):
        secrets = self.Parameters.yav_secret.data()

        if self.Parameters.testenv_mode and not self.Context.is_patch_check:
            self.set_info("Exit: testenv_mode is on and self.Context.is_patch_check evaluates to false")
            return

        requested_test_type = TestType[self.Parameters.test_type.upper()]
        test_type = TestType.FT if requested_test_type == TestType.SANITIZE else requested_test_type
        sandbox_client = rest.Client()
        self.Parameters.base_revision, base_revision_tasks = get_base_revision(
            test_type,
            sandbox_client,
            max_revision=self.Parameters.base_revision_upper_limit,
        )

        for role, test_name in TESTENV_TEST_NAMES[test_type.value].items():
            task_id = base_revision_tasks[test_name]
            task_id_parameter_name = "{role}_task_id".format(role=role)
            setattr(self.Parameters, task_id_parameter_name, task_id)

        arcanum_review_id = self.Context.arcanum_review_id or self.Parameters.arcanum_review_id

        if self.Parameters.check_hamsters_readiness:
            bad_hamsters_error_message = self.check_hamsters_readiness(
                base_revision_tasks.values(),
                secrets["nanny_token"],
                secrets["yp_token"],
            )
            if bad_hamsters_error_message:
                self.set_info("Bad hamsters in tasks\n\n{}".format(bad_hamsters_error_message), do_escape=False)
                if arcanum_review_id:
                    comment = textwrap.dedent("""\
                        Failed to find base revision. Rebase pull request and run brave tests again.\n\n
                        More info in task {}""".format(get_task_markdown_hyperlink(self.id)))
                    comment_arcanum_review(
                        arcanum_token=secrets["arc_token"],
                        review_id=arcanum_review_id,
                        comment=comment,
                    )
                raise TaskFailure("Bad hamsters in shoot tasks")

        if not arcanum_review_id:
            logger.info("arcanum_review_id not set, will not calculate patch")
            return

        patch = self.get_patch(self.Parameters.base_revision, arcanum_review_id)
        patch_path = os.path.join(get_logs_folder(), "patch")
        logger.debug("Save patch to %s", patch_path)
        with open(patch_path, 'w') as f:
            f.write(patch)

        patch_paste_link = upload_to_paste(patch, self.Parameters.yav_secret.data()["sandbox_token"], syntax="diff")
        if not patch_paste_link:
            raise TaskFailure("Failed to upload patch to paste.yandex-team.ru")

        logger.debug("paste link to the patch: %s", patch_paste_link)
        self.Parameters.arcadia_patch = patch_paste_link
