import datetime
import logging
import os
import time

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.yabs.qa.saas.coordinator import SaasKVRSCoordinatorClient
from sandbox.projects.yabs.qa.saas.topology import topology_from_states
from sandbox.projects.yabs.qa.saas.record import add_record
from sandbox.projects.yabs.qa.resource_types import BaseBackupSdk2Resource

from sandbox.projects.yabs.qa.tasks.YabsServerFreezeCurrentKvrsSaasSnapshot.utils import validate_states


logger = logging.getLogger(__name__)


DEFAULT_ENDPOINT_SET_ID = "sas-hamster2-saas2.coordinator"
DEFAULT_CLUSTER = "sas"
DEFAULT_TTL = int(datetime.timedelta(hours=12).total_seconds())
YT_PROXY = "hahn"


class YabsServerSaasTopology(BaseBackupSdk2Resource):
    auto_backup = True


class YabsServerFreezeCurrentKvrsSaasSnapshot(BaseBinTask):
    class Parameters(BaseBinTask.Parameters):
        resource_attrs = BaseBinTask.Parameters.resource_attrs(default={"task_type": "YABS_SERVER_FREEZE_CURRENT_KVRS_SAAS_SNAPSHOT"})
        release_version = BaseBinTask.Parameters.release_version(default=ReleaseStatus.STABLE)

        yav_secret_id = sdk2.parameters.YavSecret("YaV secret identifier", default="sec-01d6apzcex5fpzs5fcw1pxsfd5")

        with sdk2.parameters.Group("Testenv options") as testenv_options:
            testenv_switch_trigger = sdk2.parameters.Bool(
                "Make topology resource usable for testenv",
                description="Set \"testenv_switch_trigger\" attribute on topology resource",
                default=False,
                do_not_copy=True)

        with sdk2.parameters.Output():
            saas_topology_resource = sdk2.parameters.Resource('SaaS topology resource', resource_type=YabsServerSaasTopology)

    def on_execute(self):
        from yt.wrapper import YtClient

        yt_token = self.Parameters.yav_secret_id.data()["yt_token"]
        yt_client = YtClient(proxy=YT_PROXY, token=yt_token)

        saas_coordinator_client = SaasKVRSCoordinatorClient(DEFAULT_CLUSTER, DEFAULT_ENDPOINT_SET_ID)

        frozen_states = saas_coordinator_client.freeze_last_state()
        logger.debug("Freeze last state response: %s", frozen_states)

        missing_states = validate_states(frozen_states)
        if missing_states:
            logger.error("States validation failed, missing states: %s", missing_states)
            raise TaskFailure("States validation failed. See missing states in logs")

        topology = topology_from_states(frozen_states)
        topology_path = os.path.abspath("topology")
        with open(topology_path, "w") as f:
            f.write(str(topology))

        saas_topology_resource = YabsServerSaasTopology(self, "SaaS banner resources topology", topology_path)
        sdk2.ResourceData(saas_topology_resource).ready()
        self.Parameters.saas_topology_resource = saas_topology_resource

        add_record(yt_client, resource_id=saas_topology_resource.id, ttl=DEFAULT_TTL)

        if self.Parameters.testenv_switch_trigger:
            saas_topology_resource.testenv_switch_trigger = int(time.time())
