import json
import logging
import datetime

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Semaphores
from sandbox.sandboxsdk import environments
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_SAAS_FREEZE_DATA
from sandbox.projects.yabs.qa.tasks.YabsServerShmResourceGC.saas import (
    freeze_snapshot_metas,
    get_active_snapshot_metas,
    get_new_snapshots_to_freeze,
    get_old_snapshots_to_freeze,
    SaasCoordinatorClient,
    SNAPSHOTS_TABLE_PATH,
    SNAPSHOTS_TABLE_PROXY,
    SEMAPHORE_NAME,
)


logger = logging.getLogger(__name__)


class YabsServerFreezeSaasSnapshotsParameters(sdk2.Parameters):
    streams = sdk2.parameters.List("Stream ids to freeze", default=["0"])
    do_freeze = sdk2.parameters.Bool("Perform modifying actions", default=False, do_not_copy=True)


class YabsServerFreezeSaasSnapshots(sdk2.Task):
    """Freezes new saas snaphots along with active freezed ones"""

    class Requirements(sdk2.Requirements):
        ram = 4 * 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

        environments = (
            environments.PipEnvironment('yandex-yt', use_wheel=True),
        )

        semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name=SEMAPHORE_NAME, weight=1),
            ],
        )

    class Parameters(YabsServerFreezeSaasSnapshotsParameters):

        with sdk2.parameters.Group('Testenv options') as testenv_options:
            switch_testenv_bases = sdk2.parameters.Bool('Switch testenv bases to new resource', default=False, do_not_copy=True)
            testenv_switch_trigger_value = sdk2.parameters.String('Use following string as testenv switch trigger value, backup date will be used if empty', default='')

        with sdk2.parameters.Group('Auth') as auth_params:
            yav_secret_id = sdk2.parameters.String("YaV secret id", default="sec-01d6apzcex5fpzs5fcw1pxsfd5")

        with sdk2.parameters.Output:
            saas_freeze_data = sdk2.parameters.Resource('SaaS freeze data', resource_type=YABS_SERVER_SAAS_FREEZE_DATA)

    def on_execute(self):
        from yt.wrapper import YtClient
        secret = sdk2.yav.Secret(self.Parameters.yav_secret_id)
        yt_token = secret.data()['yt_token']
        yt_client = YtClient(proxy=SNAPSHOTS_TABLE_PROXY, token=yt_token)
        saas_coordinator_client = SaasCoordinatorClient()

        available_saas_snapshots = saas_coordinator_client.get_snapshots()["SnapshotRecords"]
        active_snapshot_metas = get_active_snapshot_metas(yt_client, SNAPSHOTS_TABLE_PATH)

        new_snapshots_to_freeze = get_new_snapshots_to_freeze(available_saas_snapshots, self.Parameters.streams)
        old_snapshots_to_freeze = get_old_snapshots_to_freeze(available_saas_snapshots, active_snapshot_metas)

        freeze_snapshot_metas(saas_coordinator_client, yt_client, new_snapshots_to_freeze, old_snapshots_to_freeze, do_freeze=self.Parameters.do_freeze)

        if not new_snapshots_to_freeze:
            raise TaskFailure('No new snapshots to freeze found')

        new_frozen_snapshots = [
            {
                "stream": snapshot["Meta"]["Stream"],
                "id": snapshot["Meta"]["Id"],
                "endpoint_set_id": snapshot["Status"]["EndpointSets"][0]["Id"],
                "endpoint_set_cluster": snapshot["Status"]["EndpointSets"][0]["Cluster"],
            }
            for snapshot in new_snapshots_to_freeze
        ]

        snapshots_file_name = 'snapshots.json'
        with open(snapshots_file_name, 'w') as snapshots_file:
            json.dump(new_frozen_snapshots, snapshots_file, indent=2)

        attrs = {}
        if self.Parameters.switch_testenv_bases:
            testenv_switch_trigger_value = self.Parameters.testenv_switch_trigger_value or datetime.date.today().strftime('%Y%m%d')
            attrs.update(testenv_switch_trigger=testenv_switch_trigger_value)

        self.Parameters.saas_freeze_data = YABS_SERVER_SAAS_FREEZE_DATA(
            self,
            "SaaS freeze data for yabs_server tests",
            snapshots_file_name,
            **attrs
        )
