import base64
import json
import logging

from sandbox import sdk2
from sandbox.common.types.task import ReleaseStatus

from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.yabs.qa.bases.save_input import (
    destination_table_path,
    save_table_if_new,
)
from sandbox.projects.yabs.qa.tasks.cs_helpers import get_spec_archive_root_path
from sandbox.projects.yabs.qa.tasks.YabsServerSaveInput import YabsServerSaveInput
from sandbox.projects.yabs.qa.tasks.YabsServerValidateABExperiment import get_ab_experiment
from sandbox.projects.yabs.qa.spec.constants import META_MODES
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_AB_EXPERIMENT_SETTINGS

logger = logging.getLogger(__name__)


HANDLER_BS = 'BS'
HANDLER_BIGB = 'BIGB_MAIN'

DEFAULT_YT_TOKEN_VAULT_NAME = 'yabs-cs-sb-yt-token'
BIGB_PROD_LONG_CONFIG_PATH = '//home/yabs-cs/ab_experiments/bigb_prod_long_config_table'
TMP_AB_EXPERIMENT_PATH = '//home/yabs-cs-sandbox/tmp/ab_experiments'


def save_config_to_yt(yt, config, table_path):
    if not yt.exists(table_path):
        yt.create(type='table', path=table_path, recursive=True)
    yt.write_table(table_path, [{"config": config}])


def get_config_from_yt(yt, path, archive_root):
    from yt.wrapper import ypath_join

    config_table_ypath = ypath_join(archive_root, 'yt_tables', path.lstrip('/'))
    config = next(yt.read_table(yt.TablePath(config_table_ypath)))['config']
    return config


class YabsServerGenerateConfigWithNewAbExperiment(BaseBinTask):
    """task for creating base ab_experiment with new AB experiment config"""

    # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
    class Requirements(sdk2.Requirements):
        cores = 4  # 4 cores or less
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(BaseBinTask.Parameters):
        resource_attrs = BaseBinTask.Parameters.resource_attrs(default={"task_type": "YABS_SERVER_GENERATE_CONFIG_WITH_NEW_AB_EXPERIMENT"})
        release_version = BaseBinTask.Parameters.release_version(default=ReleaseStatus.STABLE)

        ab_experiment = sdk2.parameters.JSON('Json with AB experiments')
        ab_experiment_resource = sdk2.parameters.Resource('Resource with json AB experiment', resource_type=YABS_SERVER_AB_EXPERIMENT_SETTINGS)
        ab_handlers = sdk2.parameters.List('AB HANDLER to take TESTID value from', default=[HANDLER_BS, HANDLER_BIGB])
        mbb_params = sdk2.parameters.JSON('Required for YABS_SERVER_MAKE_BIN_BASES parameters')
        yt_token_vault_name = sdk2.parameters.String('Vault name for YT token', default=DEFAULT_YT_TOKEN_VAULT_NAME)
        kill_timeout = 3600
        hash_type = sdk2.parameters.String('TESTID hash type', default='HT_REQID')
        drop_other_experiments = sdk2.parameters.Bool('Drop experiment other than DEFAULT/PREDEFAULT', default=False)

    @staticmethod
    def clear_flags_with_experiment_type(node, name):
        import adminka.src.usersplit_lib.common as usersplit_lib

        if not isinstance(node, usersplit_lib.Experiment):
            return

        exps = json.loads(base64.b64decode(node._flags))
        for i in range(len(exps)):
            if exps[i].get('HANDLER') != HANDLER_BIGB:
                continue
            try:
                testid = int(exps[i].get('TESTID', [-1])[0])
            except ValueError:  # broken json can be added to config
                continue
            try:
                if exps[i]['CONTEXT']['MAIN']['BIGB_MAIN']['MetaParameters']['ExperimentType'] == name:
                    del exps[i]['CONTEXT']['MAIN']['BIGB_MAIN']['MetaParameters']
                    exps[i]['CONTEXT']['MAIN']['BIGB_MAIN']['BSParameters'] = [{'Condition': {}, 'Settings': {}}]
                    node._flags = base64.b64encode(json.dumps(exps))
                    return testid  # can't have more than one
            except KeyError:
                continue

    def insert_experiment(self, exps_parameters, config_decomposed, dump_file='new_config.txt'):
        import adminka.src.usersplit_lib.common as usersplit_lib
        import adminka.src.config_lib.composer.config_composer as config_composer

        if exps_parameters:
            self.Context.testids = [
                int(e['TESTID'][0])
                for e in exps_parameters
                if e.get('HANDLER') in self.Parameters.ab_handlers
            ]

            root = config_decomposed['tree'].get_root()
            splitter = usersplit_lib.Splitter()
            splitter.add_child(root)

            for i in range(len(exps_parameters)):
                try:
                    self.Context.exp_type = exps_parameters[i]['CONTEXT']['MAIN']['BIGB_MAIN']['MetaParameters']['ExperimentType']
                except KeyError:
                    pass
                if self.Context.exp_type not in ('DEFAULT', 'PRE_DEFAULT'):
                    continue
                self.Context.erased_defaults = []
                for node in config_decomposed['tree']._nodes:
                    erased_testid = self.clear_flags_with_experiment_type(
                        node,
                        self.Context.exp_type,
                    )
                    if erased_testid is not None:
                        self.Context.erased_defaults.append(erased_testid)

            exp = usersplit_lib.Experiment(self.Context.testids[0], json.dumps(exps_parameters))
            if self.Context.exp_type == 'DEFAULT' or not self.Parameters.hash_type:
                splitter.add_child(exp)
            else:
                h = usersplit_lib.ConditionHasherUniversal(
                    salt='icecream',
                    buckets=100,
                    slots=1,
                    hash_type=usersplit_lib.HASH_TYPES[
                        'HT_ADS_USER_ID' if self.Context.exp_type == 'PRE_DEFAULT' else self.Parameters.hash_type][
                        'value'],
                    yuid_min_age=-1,
                )
                h.set_default(exp)
                splitter.add_child(h)

            config_decomposed['tree'] = usersplit_lib.ConfigTree(splitter, 0)

        config_composer.write_composed_config(config_decomposed, dump_file)
        with open(dump_file, 'r') as f:
            new_config = f.read()
        return new_config

    def delete_other_experiments(self, config):
        import adminka.src.usersplit_lib.common as usersplit_lib
        import adminka.src.config_lib.composer.config_decomposer as config_decomposer

        config_decomposed = config_decomposer.decompose_full_config(config)

        if not self.Parameters.drop_other_experiments:
            return config_decomposed

        for node in config_decomposed['tree']._nodes:
            if not isinstance(node, usersplit_lib.Experiment):
                continue

            exps = json.loads(base64.b64decode(node._flags))
            for i in range(len(exps)):
                if exps[i].get('HANDLER') != HANDLER_BIGB:
                    continue

                try:
                    if exps[i]['CONTEXT']['MAIN']['BIGB_MAIN']['MetaParameters']['ExperimentType'] not in ('DEFAULT', 'PRE_DEFAULT'):
                        exps[i]['CONTEXT']['MAIN']['BIGB_MAIN'] = {}
                except KeyError:
                    exps[i]['CONTEXT']['MAIN']['BIGB_MAIN'] = {}
                    continue

            node._flags = base64.b64encode(json.dumps(exps))

        return config_decomposed

    def on_execute(self):
        from yt.wrapper import YtClient, ypath_join

        yt_token = sdk2.Vault.data(self.Parameters.yt_token_vault_name)
        yt_client = YtClient(proxy="hahn", token=yt_token)

        input_spec_resource_id = self.Parameters.mbb_params["input_spec"]
        self.Context.archive_root = get_spec_archive_root_path(input_spec_resource_id)

        with self.memoize_stage.save_input_yt_tables_with_config(commit_on_entrance=False):
            required_tables = [
                {
                    "path": BIGB_PROD_LONG_CONFIG_PATH,
                    "id": "bigb_prod_long_config_table",
                },
            ]
            tables_to_save = []
            for table in required_tables:
                path = destination_table_path(table["path"], self.Context.archive_root)
                if yt_client.exists(path):
                    logger.info("Table %s already exists", path)
                    continue
                tables_to_save.append(table)

            if tables_to_save:
                save_input_task = YabsServerSaveInput(
                    self,
                    tags=self.Parameters.tags,
                    owner=self.owner,
                    dynamic_bundle_name='yabs-cs-sandbox-dynamic',
                    no_quota_attrs=True,
                    recreate_links=True,
                    tables_to_save=tables_to_save,
                    input_spec=input_spec_resource_id,
                ).enqueue()
                self.Context.save_input_task_id = save_input_task.id

        check_tasks(self, self.Context.save_input_task_id or [])

        with self.memoize_stage.create_yt_tables_with_config(commit_on_entrance=False):
            ab_experiment = get_ab_experiment(self.Parameters.ab_experiment_resource, self.Parameters.ab_experiment)
            logger.info('AB experiment data: %s\n', json.dumps(ab_experiment, indent=2))

            bigb_prod_long_config = get_config_from_yt(yt_client, BIGB_PROD_LONG_CONFIG_PATH, self.Context.archive_root)
            bigb_prod_long_config_decoded = json.loads(bigb_prod_long_config)

            ab_experiment_long_config = bigb_prod_long_config_decoded["configs"][0]["data"]
            ab_experiment_long_config_decomposed = self.delete_other_experiments(ab_experiment_long_config)
            ab_experiment_long_new_config = self.insert_experiment(ab_experiment, ab_experiment_long_config_decomposed)

            bigb_prod_long_config_decoded["configs"][0]["data"] = ab_experiment_long_new_config

            self.Context.tmp_dir_path = ypath_join(TMP_AB_EXPERIMENT_PATH, str(self.id))

            self.Context.ab_experiment_long_yt_path = ypath_join(
                self.Context.tmp_dir_path, 'bigb_prod_long_config_table_with_{}'.format(self.Context.testids[0])
            )
            save_config_to_yt(yt_client, json.dumps(bigb_prod_long_config_decoded), self.Context.ab_experiment_long_yt_path)
            save_table_if_new(
                yt_client,
                src_path=self.Context.ab_experiment_long_yt_path,
                dst_path=destination_table_path(self.Context.ab_experiment_long_yt_path, self.Context.archive_root),
                save_input_no_quota_attrs=False,
            )

        with self.memoize_stage.run_make_bin_bases(commit_on_entrance=False):
            mbb_params = dict(self.Parameters.mbb_params)
            mbb_params['bin_db_list'] = 'ab_experiment'
            for meta_mode in META_MODES:
                mbb_params['base_tags_meta_{}'.format(meta_mode)] = ['ab_experiment']
                mbb_params['base_tags_stat_{}'.format(meta_mode)] = ['ab_experiment']
            mbb_params['settings_spec'] = json.dumps(
                {
                    "ab_experiment_long_config_path": [
                        {"value": self.Context.ab_experiment_long_yt_path},
                    ]
                }
            )
            mbb_task = sdk2.Task['YABS_SERVER_MAKE_BIN_BASES'](
                self, tags=self.Parameters.tags, description='Make ab_experiments for YABS_SERVER_AB_EXPERIMENT_TEST', **mbb_params
            ).enqueue()
            self.Context.make_bin_bases_task_id = mbb_task.id

        check_tasks(self, [self.Context.make_bin_bases_task_id])
        mbb_task = sdk2.Task[self.Context.make_bin_bases_task_id]
        self.Context.ab_experiment_db = mbb_task.Context.bin_base_res_ids[0]

        yt_client.remove(self.Context.tmp_dir_path, recursive=True)
