import logging

import sandbox.projects.yabs.qa.resource_types as yabs_resource_type

from sandbox import sdk2
from sandbox.projects.common.yabs.server.db import yt_bases
from sandbox.projects.common.yabs.server.db.task.cs import SaveAllInputs, YtPool
from sandbox.projects.common.yabs.server.util.general import check_tasks
from sandbox.projects.runtime_models.components.yabs.BuildYabsRtModelsPackage import BuildYabsRtModelsPackage
from sandbox.projects.yabs.qa.pipeline.stage import stage
from sandbox.projects.yabs.qa.tasks.cs_helpers import CSHelper
from sandbox.projects.yabs.qa.tasks.YabsServerRunCSImportWrapper import YabsServerRunCSImportWrapper
from sandbox.projects.yabs.qa.tasks.YabsServerSaveInput import YabsServerSaveInput
from sandbox.sandboxsdk import environments

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace_entry_point
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.generic import enqueue_task


def get_saved_glm_dumps_path(input_archive_path, glm_dumps_path):
    from yt.wrapper import ypath_join

    return ypath_join(input_archive_path, "yt_tables", glm_dumps_path[1:])


class InputParameters(sdk2.Parameters):
    imports_root_dir = sdk2.parameters.String("Import results root directory", default=yt_bases.IMPORTS_ROOT)

    with sdk2.parameters.Group("Input data") as input_data:
        input_spec = sdk2.parameters.Resource(
            "Input spec for cs_import",
            resource_type=yabs_resource_type.YABS_CS_INPUT_SPEC,
            required=True)

        settings_archive = sdk2.parameters.Resource(
            "CS settings archive",
            resource_type=yabs_resource_type.YABS_CS_SETTINGS_ARCHIVE)
        cs_settings_patch = sdk2.parameters.Resource(
            "Updates CS settings using jsondiff",
            resource_type=yabs_resource_type.YABS_CS_SETTINGS_PATCH)
        settings_spec = sdk2.parameters.String("CS Settings spec (json) to be passed with --settings-spec")

        mysql_archive_contents = sdk2.parameters.Resource(
            "MySQL archive contents",
            resource_type=yabs_resource_type.YABS_MYSQL_ARCHIVE_CONTENTS,
            required=True,
            multiple=False)

        glm_dumps_path = sdk2.parameters.String(
            "GLM dumps path",
            default="//home/yabs-cs/ads_quality/linear_models/service_under_meta_glm")

    with sdk2.parameters.Group("Additional parameters for run cs import") as run_cs_import:
        import_result_ttl = sdk2.parameters.Integer(
            "Lower limit for output TTL, days (does not work for runs with patch)",
            default=1)

        reuse_import_results = sdk2.parameters.Bool("Reuse import results", default=True)

        calc_digest = sdk2.parameters.Bool("Calculate digest of import result", default=False)
        with calc_digest.value[True]:
            wait_digest = sdk2.parameters.Bool("Wait for digest of import result", default=False)
        drop_import_result = sdk2.parameters.Bool("Remove node with import result after finish", default=False)
        save_all_inputs = SaveAllInputs(default=False)
        yt_pool = YtPool()

    with sdk2.parameters.Group("Common mkdb parameters") as common_params:
        bs_release_yt_resource = sdk2.parameters.Resource(
            "BS release yt resource",
            resource_type=yabs_resource_type.BS_RELEASE_YT)
        publish_spec = sdk2.parameters.Bool("Publish spec to be used for local debugging", default=True)
        propagate_tags = sdk2.parameters.Bool("Use own tags when creating child tasks", default=False)


class YabsServerGenerateLinearModelsServiceData(sdk2.Task, CSHelper):
    description = "Generate data for linear models service"

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        environments = [
            environments.PipEnvironment("jsondiff", version="1.2.0"),
            environments.PipEnvironment("yandex-yt", version="0.10.8"),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        input_parameters = InputParameters()

        build_resource_type = sdk2.parameters.String("Package resource type", default="YABS_LINEAR_MODELS_DATA")
        with sdk2.parameters.Output():
            linear_models_service_data_resource = sdk2.parameters.Resource("Resource with data for the linear models service")

        # Pass tasks binary to child tasks
        # Details: https://docs.yandex-team.ru/sandbox/dev/binary-task#sticky-resource
        push_tasks_resource = True

    @stage(provides='import_task_id')
    def run_import_task(self):
        import_task = enqueue_task(YabsServerRunCSImportWrapper(
            self,
            description="Run importers LmServiceBase, LmServiceCounters with dependencies",
            owner=self.owner,
            tags=self.Parameters.tags,
            hints=list(self.hints),
            bin_db_list="qtail_linear_model",  # dirty hack to use_cs_cycle
            importers=["LmServiceBase", "LmServiceCounters"],
            use_cs_cycle=True,
            lower_reusable_ttl_limit=self.Parameters.import_result_ttl,
            **dict(self.Parameters)
        ))
        return import_task.id

    @stage(provides='save_input_task_id')
    def run_save_input_task(self):
        tables_to_save = [{
            "id": "service_under_meta_glm",
            "path": self.Parameters.glm_dumps_path,
        }]
        save_input_task = enqueue_task(YabsServerSaveInput(
            self,
            description="Save data for linear model service",
            owner=self.owner,
            tags=self.Parameters.tags,
            hints=list(self.hints),
            tables_to_save=tables_to_save,
            input_spec=self.Parameters.input_spec,
        ))
        return save_input_task.id

    def run_build_models_package_task(self, index_table_path, glm_dumps_path, build_resource_type, yt_proxy="hahn"):
        task = enqueue_task(BuildYabsRtModelsPackage(
            self,
            description="Generate data for linear model service",
            owner=self.owner,
            tags=self.Parameters.tags,
            hints=list(self.hints),
            index_table_path=index_table_path,
            glm_dumps_path=glm_dumps_path,
            yt_proxy=yt_proxy,
            yt_token_secret_name="yabscs_yt_token",
            resource_ttl=7,
            upload_to_mds=True,
            sync_upload_to_mds=True,
            build_resource_type=build_resource_type,
        ))
        return task

    @stage(provides='build_models_package_task_id')
    def build_models_package(self, import_task_id, save_input_task_id, build_resource_type='YABS_LINEAR_MODELS_DATA'):
        check_tasks(self, [import_task_id, save_input_task_id])

        index_table_path = sdk2.Task[import_task_id].Parameters.import_node_path
        logging.debug("index_table_path: %s", index_table_path)

        glm_dumps_path = get_saved_glm_dumps_path(self.archive_root_path, self.Parameters.glm_dumps_path)
        logging.debug("glm_dumps_path: %s", glm_dumps_path)

        build_models_package_task = self.run_build_models_package_task(
            index_table_path=index_table_path,
            glm_dumps_path=glm_dumps_path,
            build_resource_type=build_resource_type
        )
        return build_models_package_task.id

    def get_models_resource(self, build_models_package_task_id):
        check_tasks(self, build_models_package_task_id)
        return sdk2.Task[build_models_package_task_id].Parameters.package_resource

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        self.run_import_task()
        self.run_save_input_task()
        build_models_package_task_id = self.build_models_package(build_resource_type=self.Parameters.build_resource_type)
        self.Parameters.linear_models_service_data_resource = self.get_models_resource(build_models_package_task_id)
