
import logging
import re

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.common.yabs.server.db import yt_bases
from sandbox.projects.common.yabs.server.db import utils as dbutils
from sandbox.projects.yabs.qa.utils import importer as importer_utils
from sandbox.projects.yabs.qa.resource_types import (
    BS_RELEASE_YT,
    YABS_CS_SETTINGS_ARCHIVE,
    YABS_CS_SETTINGS_PATCH,
)
from sandbox.projects.yabs.qa.tasks.cs_helpers import CSHelper
from sandbox.projects.yabs.qa.tasks.YabsServerRunCSImportWrapper.utils import get_importer_with_dependencies_version

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace_entry_point


logger = logging.getLogger(__name__)
base_tag_pattern = re.compile(r'(?P<tag>[a-z_]*[a-z]{1})(_?)([0-9]*)')


class YabsServerGetCSVersions(sdk2.Task, CSHelper):

    name = 'YABS_SERVER_GET_CS_VERSIONS'

    class Parameters(sdk2.Parameters):
        bs_release_yt_resource = sdk2.parameters.Resource("BS_RELEASE_YT resource", resource_type=BS_RELEASE_YT)
        settings_archive = sdk2.parameters.Resource("CS settings archive", resource_type=YABS_CS_SETTINGS_ARCHIVE, required=False)
        settings_spec = sdk2.parameters.String("CS Settings spec (json) to be passed with --settings-spec")
        cs_settings_patch = sdk2.parameters.Resource("Updates CS settings using jsondiff", resource_type=YABS_CS_SETTINGS_PATCH)

        with sdk2.parameters.Output:
            with sdk2.parameters.Group('Base versions') as base_versions:
                base_internal_versions = sdk2.parameters.JSON('Base internal versions')

            with sdk2.parameters.Group('Raw importer versions') as raw_importer_versions:
                importer_code_versions = sdk2.parameters.JSON('Importer code versions')
                importer_mkdb_versions = sdk2.parameters.JSON('Importer code versions')
                importer_settings_versions = sdk2.parameters.JSON('Importer code versions')

            with sdk2.parameters.Group('Importer versions with dependencies') as importer_versions_with_dependencies:
                importer_code_versions_with_dependencies = sdk2.parameters.JSON('Importer code versions with dependencies')
                importer_mkdb_versions_with_dependencies = sdk2.parameters.JSON('Importer code versions with dependencies')
                importer_settings_versions_with_dependencies = sdk2.parameters.JSON('Importer code versions with dependencies')

    class Requirements(sdk2.Requirements):
        cores = 2
        ram = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

        environments = (
            PipEnvironment('jsondiff', version='1.2.0'),
            PipEnvironment('networkx', version='2.2', use_wheel=True),
        )

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        importers_info = yt_bases.get_cs_import_info(self._get_yabscs(), self.cs_settings, False)
        base_internal_versions_sharded = dbutils.get_base_internal_vers(self._get_yabscs())
        base_internal_versions = {}
        for base_tag, version in base_internal_versions_sharded.items():
            if base_tag.endswith('lmng_0'):
                base_internal_versions[base_tag] = version
                continue
            _base_tag = base_tag_pattern.match(base_tag).groupdict()['tag']
            base_internal_versions[_base_tag] = version

        with self.memoize_stage.set_base_versions(commit_on_entrance=False):
            self.Parameters.base_internal_versions = base_internal_versions

        with self.memoize_stage.set_importer_code_versions(commit_on_entrance=False):
            self.Parameters.importer_code_versions = self.importer_code_version
        with self.memoize_stage.set_importer_mkdb_versions(commit_on_entrance=False):
            self.Parameters.importer_mkdb_versions = {
                importer_name: importer_utils.get_importer_mkdb_info_version(importer_info, self.mkdb_info)
                for importer_name, importer_info in importers_info.items()
            }
        with self.memoize_stage.set_importer_settings_versions(commit_on_entrance=False):
            self.Parameters.importer_settings_versions = self.cs_settings_version

        with self.memoize_stage.set_importer_code_versions_with_dependencies(commit_on_entrance=False):
            self.Parameters.importer_code_versions_with_dependencies = {
                importer: get_importer_with_dependencies_version(importer, importers_info, self.Parameters.importer_code_versions)
                for importer in importers_info.keys()
            }
        with self.memoize_stage.set_importer_mkdb_versions_with_dependencies(commit_on_entrance=False):
            self.Parameters.importer_mkdb_versions_with_dependencies = {
                importer: get_importer_with_dependencies_version(importer, importers_info, self.Parameters.importer_mkdb_versions)
                for importer in importers_info.keys()
            }
        with self.memoize_stage.set_importer_settings_versions_with_dependencies(commit_on_entrance=False):
            self.Parameters.importer_settings_versions_with_dependencies = {
                importer: get_importer_with_dependencies_version(importer, importers_info, self.Parameters.importer_settings_versions)
                for importer in importers_info.keys()
            }
