import json
import logging
import os
import subprocess
import textwrap

from sandbox import sdk2
import sandbox.common.types.resource as ctr

from sandbox.projects.resource_types import CACHE_DAEMON
from sandbox.projects.common.dolbilka.resources import DPLANNER_EXECUTABLE
from sandbox.projects.common.yabs.server.util.general import try_get_from_vault
from sandbox.projects.common.yabs.server.util.general import check_tasks

from sandbox.projects.yabs.qa.tasks.BuildAmmoAndStubsFromYT import BuildAmmoAndStubsFromYT, GenAmmoAndStubFromYtBinary
from sandbox.projects.yabs.qa.resource_types import BaseBackupSdk2Resource

from .spec import DEFAULT_SPEC, validate_spec

CACHEDAEMON_DUMP_KEY = 'cachedaemon_dump_res_id'


class YabsServerGenAmmoTables(BaseBackupSdk2Resource):
    """ yabs/server/tools/gen_sandbox_stub/generate_yabs_tables binary """
    releasers = ["YABS_SERVER_SANDBOX_TESTS"]
    release_subscribers = []
    releasable = True


class YabsServerGetYtRequestData2(sdk2.Task):

    class Parameters(sdk2.Parameters):
        # common parameters
        description = "Generate yabs-server ammo and stubs from access_stat and ext_stat logs"
        max_restarts = 3
        kill_timeout = 60 * 60 * 6

        # custom parameters
        gen_ammo_tables_binary = sdk2.parameters.LastReleasedResource(
            "Ammo tables generator binary",
            resource_type=YabsServerGenAmmoTables,
            state=(ctr.State.READY, ctr.State.NOT_READY),
            required=True
        )

        gen_ammo_and_stub_from_yt_binary = sdk2.parameters.LastReleasedResource(
            "Stub and ammo generator binary",
            resource_type=GenAmmoAndStubFromYtBinary,
            state=(ctr.State.READY, ctr.State.NOT_READY),
            required=True
        )

        cache_daemon_resource = sdk2.parameters.LastReleasedResource(
            'Cache daemon binary',
            resource_type=CACHE_DAEMON,
            state=(ctr.State.READY, ctr.State.NOT_READY),
            required=True
        )

        d_planner_resource = sdk2.parameters.LastReleasedResource(
            'Dolbilka planner binary',
            resource_type=DPLANNER_EXECUTABLE,
            state=(ctr.State.READY, ctr.State.NOT_READY),
            required=True
        )

        spec = sdk2.parameters.JSON("Spec", default=DEFAULT_SPEC)
        format_yql = textwrap.dedent('''
            SELECT ... as requestID
            ...
            FROM {table name without time(//logs/bs-proto-accessstat-log/1h)}
            ...
        ''')
        yql_query_description = "Custom YQL request in format: \n" + format_yql
        yql_query = sdk2.parameters.String(yql_query_description, multiline=True)
        yql_query_role = sdk2.parameters.String("Role of custom requests (yabs, bs, bsrank)")

        yt_token_vault_name = sdk2.parameters.String("Vault name to get YT token from",
                                                     default="yabs-cs-sb-yt-token")
        yql_token_vault_name = sdk2.parameters.String("Vault name to get YQL token from",
                                                      default="yabs-cs-sb-yql-token")
        pool = sdk2.parameters.String("YT pool to launch operations in")
        enable_page_id_coverage = sdk2.parameters.Bool("Enable PageIDs coverage", default=False)
        days_interval = sdk2.parameters.Integer("Interval in days to get requests", default=1, required=True)
        logs_interval = sdk2.parameters.String("Logs interval: 1d or 1h", default="1h", required=True)
        testenv_switch_trigger = sdk2.parameters.Bool("Switch Testenv to generated resources", default=False, do_not_copy=True)
        testenv_switch_trigger_value = sdk2.parameters.String('Use following string as testenv switch trigger value, backup date will be used if empty', default='')
        resource_ttl = sdk2.parameters.Integer("Output resources ttl", default=30)

        with sdk2.parameters.Output:
            result_spec = sdk2.parameters.JSON('Output generated ammo spec', default={})

    class Context(sdk2.Context):
        ammo_tables_dict = None
        resource_generation_subtask_id = None

    def on_enqueue(self):
        spec = self.Parameters.spec
        validate_spec(spec)

    def on_execute(self):
        if self.Context.ammo_tables_dict is None:
            validate_spec(self.Parameters.spec)

            stubgen_env = self._add_tokens_to_env()

            gen_sandbox_stub_path = unicode(sdk2.ResourceData(self.Parameters.gen_ammo_tables_binary).path)
            yt_work_prefix = '//home/yabs-cs-sandbox/gen_sandbox_stub/{}'.format(self.id)
            yt_page_id_coverage_prefix = "//home/yabs-cs-sandbox"

            ammo_tables_gen_cmdline = [
                gen_sandbox_stub_path,
                '--yt-work-prefix', yt_work_prefix,
                '--spec', json.dumps(self.Parameters.spec),
                '--proxy', 'hahn',
                '--days', str(self.Parameters.days_interval),
                '-a', "//logs/bs-proto-accessstat-log/" + self.Parameters.logs_interval,
                '-e', "//logs/bs-proto-extstat-log/" + self.Parameters.logs_interval
            ]
            if self.Parameters.pool:
                ammo_tables_gen_cmdline.extend(['--pool', self.Parameters.pool])
            if self.Parameters.yql_query != "":
                ammo_tables_gen_cmdline.extend(['--query', self.Parameters.yql_query])
            if self.Parameters.enable_page_id_coverage:
                ammo_tables_gen_cmdline.append('--enable-page-id-coverage')
                ammo_tables_gen_cmdline.extend(['--yt-page-id-coverage-prefix', yt_page_id_coverage_prefix])
            logging.debug('Invoking ammo tables generator executable: %s', ' '.join(ammo_tables_gen_cmdline))

            with sdk2.helpers.ProcessLog(self, logger='gen_yabs_tables') as process_log:
                self.Context.ammo_tables_dict = json.loads(subprocess.check_output(ammo_tables_gen_cmdline, stderr=process_log.stderr, env=stubgen_env))

        if self.Context.resource_generation_subtask_id is None:
            resource_generation_subtask = BuildAmmoAndStubsFromYT(
                self,
                description='Ammo and stub resource generator subtask',
                gen_ammo_and_stub_from_yt_binary=self.Parameters.gen_ammo_and_stub_from_yt_binary,
                cache_daemon_resource=self.Parameters.cache_daemon_resource,
                d_planner_resource=self.Parameters.d_planner_resource,
                input_tables=self.Context.ammo_tables_dict,
                spec=self.Parameters.spec,
                yt_token_vault_name=self.Parameters.yt_token_vault_name,
                yt_work_prefix=yt_work_prefix,
                key_header='X-YaBS-Request-Id',
                testenv_switch_trigger=self.Parameters.testenv_switch_trigger,
                testenv_switch_trigger_value=self.Parameters.testenv_switch_trigger_value,
                resource_ttl=self.Parameters.resource_ttl,
            )
            resource_generation_subtask.Requirements.disk_space = 100 << 10
            resource_generation_subtask.save().enqueue()
            self.Context.resource_generation_subtask_id = resource_generation_subtask.id
        check_tasks(self, [self.Context.resource_generation_subtask_id])

        result_spec = sdk2.Task[self.Context.resource_generation_subtask_id].Parameters.result_spec
        self.Parameters.result_spec = result_spec

    def _add_tokens_to_env(self):
        env = os.environ.copy()
        yql_token = try_get_from_vault(self, self.Parameters.yql_token_vault_name)
        yt_token = try_get_from_vault(self, self.Parameters.yt_token_vault_name)
        logging.info("Token lengths: YT %d, YQL %d", len(yt_token), len(yql_token))
        env['YQL_TOKEN'] = yql_token
        env['YT_TOKEN'] = yt_token
        return env
